package ru.yandex.travel.orders.integration.train;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicReference;

import org.junit.Test;
import org.mockito.stubbing.Answer;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.TestConfiguration;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TJson;
import ru.yandex.travel.dicts.rasp.proto.TTrainTariffInfo;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.configurations.TrainTariffInfoDataProviderProperties;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.integration.train.factories.ImOrderInfoResponseFactory;
import ru.yandex.travel.orders.integration.train.factories.ImReservationCreateResponseFactory;
import ru.yandex.travel.orders.integration.train.factories.TrainOrderItemFactory;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.proto.ERefundPartType;
import ru.yandex.travel.orders.proto.TAddInsuranceReq;
import ru.yandex.travel.orders.proto.TCalculateRefundReqV2;
import ru.yandex.travel.orders.proto.TCheckoutReq;
import ru.yandex.travel.orders.proto.TCreateOrderReq;
import ru.yandex.travel.orders.proto.TCreateOrderRsp;
import ru.yandex.travel.orders.proto.TCreateServiceReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.proto.TOrderInfo;
import ru.yandex.travel.orders.proto.TRefundPartContext;
import ru.yandex.travel.orders.proto.TReserveReq;
import ru.yandex.travel.orders.proto.TStartPaymentReq;
import ru.yandex.travel.orders.proto.TStartRefundReq;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.repository.TrainOrderItemRepository;
import ru.yandex.travel.orders.services.mock.MockTrustClient;
import ru.yandex.travel.orders.services.orders.RefundPartsService;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.train.tariffinfo.TrainTariffInfoDataProvider;
import ru.yandex.travel.orders.services.train.tariffinfo.TrainTariffInfoService;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflows.orderitem.train.ImHelpers;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.train.model.ReservationPlaceType;
import ru.yandex.travel.train.model.TrainPassenger;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.train.model.TrainTicketRefundStatus;
import ru.yandex.travel.train.partners.im.ImClient;
import ru.yandex.travel.train.partners.im.model.AutoReturnRequest;
import ru.yandex.travel.train.partners.im.model.AutoReturnResponse;
import ru.yandex.travel.train.partners.im.model.ImBlankStatus;
import ru.yandex.travel.train.partners.im.model.PlaceWithType;
import ru.yandex.travel.train.partners.im.model.RailwayAutoReturnResponse;
import ru.yandex.travel.train.partners.im.model.RailwayReturnAmountResponse;
import ru.yandex.travel.train.partners.im.model.RailwayReturnBlankResponse;
import ru.yandex.travel.train.partners.im.model.ReturnAmountResponse;
import ru.yandex.travel.train.partners.im.model.insurance.InsuranceCheckoutResponse;
import ru.yandex.travel.train.partners.im.model.insurance.InsurancePricingResponse;
import ru.yandex.travel.train.partners.im.model.insurance.RailwayTravelPricingResult;
import ru.yandex.travel.train.partners.im.model.insurance.RailwayTravelProductPricingInfo;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationStatus;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderInfoResponse;
import ru.yandex.travel.yt_lucene_index.TestLuceneIndexBuilder;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyBoolean;
import static org.mockito.ArgumentMatchers.anyInt;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static ru.yandex.travel.commons.proto.ProtoUtils.fromTJson;
import static ru.yandex.travel.orders.integration.IntegrationUtils.waitForPredicateOrTimeout;

@SuppressWarnings("ResultOfMethodCallIgnored")
public class GenericOrderFlowTrainRebookingTests extends AbstractTrainOrderFlowTest {

    @MockBean
    private ImClient imClient;

    @Autowired
    private TransactionTemplate transactionTemplate;

    @Autowired
    private OrderRepository orderRepository;

    @Autowired
    private TrainOrderItemRepository trainOrderItemRepository;

    @Autowired
    private TrustClient trustClient;

    @Test
    public void testRebookingSenior() {
        var factory = new TrainOrderItemFactory();
        factory.setTariffCode("senior");
        factory.setDocumentNumber("3709123454");
        var trainOrderItem = factory.createTrainOrderItem();
        var reservationResponse1 = ImReservationCreateResponseFactory.createForOrder(trainOrderItem, null)
                .createReservationCreateResponse();
        var imConfirmTill1 = ImHelpers.toLocalDateTime(Instant.now().plusSeconds(1), ImHelpers.MSK_TZ);
        reservationResponse1.setConfirmTill(imConfirmTill1);
        reservationResponse1.getReservationResults().get(0).setConfirmTill(imConfirmTill1);
        var reservationResponse2 = ImReservationCreateResponseFactory.createForOrder(trainOrderItem, null)
                .createReservationCreateResponse();
        when(imClient.reservationCreate(any(), any())).thenReturn(reservationResponse1).thenReturn(reservationResponse2);
        var imOrderInfoResponse = createImOrderInfoResponse(ImOperationStatus.OK, ImOperationStatus.OK, false);
        when(imClient.orderInfo(anyInt())).thenReturn(imOrderInfoResponse);
        when(imClient.orderInfo(anyInt(), any())).thenReturn(imOrderInfoResponse);
        when(imClient.insurancePricing(any())).thenReturn(createInsurancePricingResponse());
        when(imClient.insuranceCheckout(any())).thenReturn(createInsuranceCheckoutResponse());

        TCreateOrderReq.Builder createOrderReq = createOrderRequest(trainOrderItem.getPayload());
        TCreateOrderRsp resp = client.createOrder(createOrderReq.build());
        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED &&
                        getTrainReservation(rsp3).getIsRebookingFor() != null &&
                        rsp3.getResult().getService(0).getServiceInfo().getGenericOrderItemState() ==
                                EOrderItemState.IS_RESERVED,
                Duration.ofSeconds(10), "Order must be in OS_RESERVED state after rebooking");
        verify(imClient).reservationCancel(reservationResponse1.getOrderId());
        getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getPayload(getOrderInfoRsp.getResult()).getPassengers().get(0).getTariffCode()).isEqualTo("senior");
        var ticket = ProtoUtils.fromTJson(
                getOrderInfoRsp.getResult().getService(0).getServiceInfo().getPayload(),
                TrainReservation.class).getPassengers().get(0).getTicket();
        assertThat(ticket.getImBlankStatus()).isNull();
        assertThat(ticket.isPendingElectronicRegistration()).isEqualTo(false);

        verify(imClient, times(2)).insurancePricing(any());
        verify(imClient, times(2)).reservationCreate(any(), any());

        client.addInsurance(TAddInsuranceReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED
                        && getTrainReservation(rsp3).getInsuranceStatus() == InsuranceStatus.CHECKED_OUT,
                Duration.ofSeconds(10), "Order must be in OS_RESERVED state and InsuranceStatus must be CHECKED_OUT");

        verify(imClient, times(1)).insuranceCheckout(any());

        client.checkout(TCheckoutReq.newBuilder().setOrderId(orderId).build());
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_failed_payment_url").build()); // safely ignoring response, as we don't need
        // payment url

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getCurrentInvoice().getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                Duration.ofSeconds(10), "Invoice must be in IS_WAIT_FOR_PAYMENT state");

        failPayment(orderId);

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getCurrentInvoice().getTrustInvoiceState() == ETrustInvoiceState.IS_PAYMENT_NOT_AUTHORIZED,
                Duration.ofSeconds(10), "Current invoice must be ins IS_PAYMENT_NOT_AUTHORIZED state");

        client.checkout(TCheckoutReq.newBuilder().setOrderId(orderId).build());
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_success_payment_url").build()); // safely ignoring response, as we don't need

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getCurrentInvoice().getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                Duration.ofSeconds(10), "Invoice must be in IS_WAIT_FOR_PAYMENT state");

        authorizePayment(orderId);

        TGetOrderInfoRsp confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CONFIRMED,
                Duration.ofSeconds(10), "Order must be in OS_CONFIRMED state");
        assertThat(confirmedOrderResponse.getResult().getServiceCount()).isEqualTo(1);
        assertThat(confirmedOrderResponse.getResult().getInvoiceCount()).isEqualTo(2);

        var reservation = getTrainReservation(confirmedOrderResponse);
        assertThat(reservation.getPassengers().get(0).getTicket().getBookedTariffCode()).isEqualTo("senior");

        var blankToUpdate = imOrderInfoResponse.findBuyRailwayItems().get(0).getOrderItemBlanks().get(0);
        blankToUpdate.setBlankStatus(ImBlankStatus.REFUNDED);
        var getOrderInfoRspUpdated = client.getOrderInfo(
                TGetOrderInfoReq.newBuilder().setOrderId(orderId).setUpdateOrderOnTheFly(true).build());

        waitForPredicateOrTimeout(client, orderId,
                rsp4 -> obtainPayload(rsp4).getPassengers().get(0).getTicket().getImBlankStatus()
                        == ImBlankStatus.REFUNDED,
                Duration.ofSeconds(10), "Ticket must be in REFUNDED ImBlankStatus");
    }

    @Test
    public void testOrderRefund() {
        AtomicReference<AutoReturnRequest> autoReturnRequestRef = new AtomicReference<>();
        AtomicReference<OrderInfoResponse> orderInfoResponseRef = new AtomicReference<>();

        var nonRefundedOrderInfoResponse = new ImOrderInfoResponseFactory(ImOperationStatus.OK, null, 1,
                null, false, LocalDateTime.now().plusDays(1)).create();
        orderInfoResponseRef.set(nonRefundedOrderInfoResponse);
        when(imClient.orderInfo(anyInt())).thenAnswer((Answer<OrderInfoResponse>) invocation -> orderInfoResponseRef.get());
        when(imClient.orderInfo(anyInt(), any())).thenAnswer((Answer<OrderInfoResponse>) invocation -> orderInfoResponseRef.get());

        var orderId = createSuccessfulOrder();
        var autoReturnRsp = createAutoReturnResponse();
        when(imClient.autoReturn(any())).thenAnswer(invocation -> {
            autoReturnRequestRef.set(invocation.getArgument(0));
            return autoReturnRsp;
        });

        var orderInfoRsp = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId)
                .setUpdateOrderOnTheFly(true).build());
        var calculateRefundRsp = client.calculateRefundV2(TCalculateRefundReqV2.newBuilder().setOrderId(orderId)
                .addContext(RefundPartsService.partContextToString(TRefundPartContext.newBuilder()
                        .setServiceId(orderInfoRsp.getResult().getService(0).getServiceId())
                        .setType(ERefundPartType.RPT_SERVICE)
                        .build()))
                .build());
        client.startRefund(TStartRefundReq.newBuilder().setOrderId(orderId)
                .setRefundToken(calculateRefundRsp.getRefundToken()).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> getTrainReservation(rsp1).getPassengers().get(0).getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDING &&
                        autoReturnRequestRef.get() != null,
                Duration.ofSeconds(10), "Ticket must be REFUNDING");

        orderInfoResponseRef.set(
                createImOrderInfoResponseWithRefund(
                        autoReturnRequestRef.get().getServiceAutoReturnRequest().getAgentReferenceId()));

        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_REFUNDED &&
                        getTrainReservation(rsp1).getPassengers().get(0).getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDED,
                Duration.ofSeconds(10), "Ticket must be REFUNDED");
    }

    @Test
    public void testRebookingFailed() {
        var trainOrderItem = new TrainOrderItemFactory().createTrainOrderItem();
        var reservationResponse1 = ImReservationCreateResponseFactory.createForOrder(trainOrderItem, null)
                .createReservationCreateResponse();
        var imConfirmTill1 = ImHelpers.toLocalDateTime(Instant.now().plusSeconds(1), ImHelpers.MSK_TZ);
        var place1 = new PlaceWithType();
        place1.setNumber("1408");
        place1.setType(ReservationPlaceType.LAST_KUPE_UPPER);
        reservationResponse1.setConfirmTill(imConfirmTill1);
        reservationResponse1.getReservationResults().get(0).setConfirmTill(imConfirmTill1);
        reservationResponse1.getReservationResults().get(0).getPassengers().get(0).setPlacesWithType(List.of(place1));
        var reservationResponse2 = ImReservationCreateResponseFactory.createForOrder(trainOrderItem, null)
                .createReservationCreateResponse();
        when(imClient.reservationCreate(any(), any())).thenReturn(reservationResponse1).thenReturn(reservationResponse2);
        var imOrderInfoResponse = createImOrderInfoResponse(ImOperationStatus.OK, ImOperationStatus.OK, false);
        when(imClient.orderInfo(anyInt())).thenReturn(imOrderInfoResponse);
        when(imClient.insurancePricing(any())).thenReturn(createInsurancePricingResponse());
        when(imClient.insuranceCheckout(any())).thenReturn(createInsuranceCheckoutResponse());

        TCreateOrderReq.Builder createOrderReq = createOrderRequest(new TrainOrderItemFactory().createTrainReservation());
        TCreateOrderRsp resp = client.createOrder(createOrderReq.build());
        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_CANCELLED &&
                        getTrainReservation(rsp3).getIsRebookingFor() != null,
                Duration.ofSeconds(10), "Order must be in OS_CANCELLED state after rebooking");

        verify(imClient, times(2)).reservationCreate(any(), any());
        verify(imClient, times(2)).reservationCancel(anyInt());
    }

    @Test
    public void testRebookingRoundTrip() {
        var reservationResponseFactory = ImReservationCreateResponseFactory.create(true, 1);
        var reservationResponse1 = reservationResponseFactory.createReservationCreateResponse();
        var reservationResponse2 = reservationResponseFactory.createReservationCreateResponse();
        var imConfirmTill1 = ImHelpers.toLocalDateTime(Instant.now().plusSeconds(1), ImHelpers.MSK_TZ);
        reservationResponse2.setConfirmTill(imConfirmTill1);
        reservationResponse2.getReservationResults().get(0).setConfirmTill(imConfirmTill1);
        reservationResponse2.getReservationResults().get(1).setConfirmTill(imConfirmTill1);
        // fix place numbers
        for (int i = 0; i < reservationResponse1.getReservationResults().size(); i++) {
            var reservationItem1 = reservationResponse1.getReservationResults().get(i);
            var reservationItem2 = reservationResponse2.getReservationResults().get(i);
            for (int j = 0; j < reservationItem1.getPassengers().size(); j++) {
                var reservationPassenger1 = reservationItem1.getPassengers().get(j);
                var reservationPassenger2 = reservationItem2.getPassengers().get(j);
                reservationPassenger2.setPlacesWithType(reservationPassenger1.getPlacesWithType());
            }
        }
        when(imClient.reservationCreate(any(), any())).thenReturn(reservationResponse2).thenReturn(reservationResponse1);
        var imOrderInfoResponse = createImOrderInfoResponse(ImOperationStatus.OK, ImOperationStatus.OK, true);
        when(imClient.orderInfo(anyInt())).thenReturn(imOrderInfoResponse);
        when(imClient.orderInfo(anyInt(), any())).thenReturn(imOrderInfoResponse);
        when(imClient.insurancePricing(any())).thenReturn(createInsurancePricingResponse());
        when(imClient.insuranceCheckout(any())).thenReturn(createInsuranceCheckoutResponse());

        TCreateOrderReq createOrderReq = createOrderRequestRoundTrip(1);
        TCreateOrderRsp resp = client.createOrder(createOrderReq);
        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                // we have to check both order's and item's state because of our default isolation level
                // (read committed): order is read from one DB state while items are loaded from some other;
                // as the result we get the order in OS_RESERVED state while the item in the IS_NEW/RESERVING state
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED &&
                        rsp3.getResult().getService(0).getServiceInfo().getGenericOrderItemState() == EOrderItemState.IS_RESERVED &&
                        getTrainReservation(rsp3).getIsRebookingFor() != null,
                Duration.ofSeconds(10), "Order must be in OS_RESERVED state after rebooking");
        verify(imClient).reservationCancel(reservationResponse1.getOrderId());
        getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getService(0).getServiceInfo().getGenericOrderItemState())
                .isEqualTo(EOrderItemState.IS_RESERVED);
        var ticket = ProtoUtils.fromTJson(
                getOrderInfoRsp.getResult().getService(0).getServiceInfo().getPayload(),
                TrainReservation.class).getPassengers().get(0).getTicket();
        assertThat(ticket.getImBlankStatus()).isNull();
        assertThat(ticket.isPendingElectronicRegistration()).isEqualTo(false);

        verify(imClient, times(4)).insurancePricing(any());
        verify(imClient, times(2)).reservationCreate(any(), any());

        client.addInsurance(TAddInsuranceReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED
                        && getTrainReservation(rsp3).getInsuranceStatus() == InsuranceStatus.CHECKED_OUT,
                Duration.ofSeconds(10), "Order must be in OS_RESERVED state and InsuranceStatus must be CHECKED_OUT");

        verify(imClient, times(2)).insuranceCheckout(any());

        client.checkout(TCheckoutReq.newBuilder().setOrderId(orderId).build());
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_success_payment_url").build()); // safely ignoring response, as we don't need

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getCurrentInvoice().getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                Duration.ofSeconds(10), "Invoice must be in IS_WAIT_FOR_PAYMENT state");

        authorizePayment(orderId);

        TGetOrderInfoRsp confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CONFIRMED,
                Duration.ofSeconds(10), "Order must be in OS_CONFIRMED state");
        assertThat(confirmedOrderResponse.getResult().getServiceCount()).isEqualTo(2);
    }

    private TrainReservation getTrainReservation(TGetOrderInfoRsp response) {
        var payloadJson = response.getResult().getService(0).getServiceInfo().getPayload();
        return fromTJson(payloadJson, TrainReservation.class);
    }

    private InsuranceCheckoutResponse createInsuranceCheckoutResponse() {
        var response = new InsuranceCheckoutResponse();
        response.setAmount(BigDecimal.valueOf(100));
        response.setOrderCustomerId(10000001);
        response.setOrderItemId(30000002);
        return response;
    }

    private InsurancePricingResponse createInsurancePricingResponse() {
        return createInsurancePricingResponse(1);
    }

    private InsurancePricingResponse createInsurancePricingResponse(int passengers) {
        var response = new InsurancePricingResponse();
        var result = new RailwayTravelPricingResult();
        response.setPricingResult(result);
        result.setProductPricingInfoList(new ArrayList<>());
        for (int i = 0; i < passengers; i++) {
            var product = new RailwayTravelProductPricingInfo();
            result.getProductPricingInfoList().add(product);
            product.setCompensation(BigDecimal.valueOf(100500));
            product.setAmount(BigDecimal.valueOf(100));
            product.setOrderCustomerId(10000001 + i);
            product.setCompany("CCC");
            product.setProductPackage("random");
            product.setProvider("PPP");
        }
        return response;
    }

    private OrderInfoResponse createImOrderInfoResponse(ImOperationStatus buyTicketStatus,
                                                        ImOperationStatus buyInsuranceStatus,
                                                        boolean roundTrip) {
        var factory = new ImOrderInfoResponseFactory();
        factory.setBuyTicketStatus(buyTicketStatus);
        factory.setBuyInsuranceStatus(buyInsuranceStatus);
        factory.setRoundTrip(roundTrip);
        return factory.create();
    }

    private OrderInfoResponse createImOrderInfoResponseWithRefund(String refundReferenceId) {
        var factory = new ImOrderInfoResponseFactory();
        factory.setRefundReferenceId(refundReferenceId);
        return factory.create();
    }

    private ReturnAmountResponse createReturnAmountResponse() {
        var rsp = new ReturnAmountResponse();
        rsp.setServiceReturnResponse(new RailwayReturnAmountResponse());
        var blank1 = new RailwayReturnBlankResponse();
        rsp.getServiceReturnResponse().setBlanks(List.of(blank1));
        blank1.setAmount(BigDecimal.valueOf(350));
        blank1.setServicePrice(BigDecimal.valueOf(156));
        blank1.setPurchaseOrderItemBlankId(20000001);
        return rsp;
    }

    private AutoReturnResponse createAutoReturnResponse() {
        var rsp = new AutoReturnResponse();
        rsp.setServiceReturnResponse(new RailwayAutoReturnResponse());
        rsp.getServiceReturnResponse().setAgentReferenceId("");
        var blank1 = new RailwayReturnBlankResponse();
        rsp.getServiceReturnResponse().setBlanks(List.of(blank1));
        blank1.setAmount(BigDecimal.valueOf(350));
        blank1.setServicePrice(BigDecimal.valueOf(156));
        blank1.setPurchaseOrderItemBlankId(20000001);
        return rsp;
    }

    private void authorizePayment(String orderId) {
        transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            assertThat(order.getCurrentInvoice()).isNotNull();
            Invoice invoice = order.getCurrentInvoice();
            String purchaseToken = invoice.getPurchaseToken();
            MockTrustClient mockTrustClient = (MockTrustClient) trustClient;
            mockTrustClient.paymentAuthorized(purchaseToken);
            return null;
        });
    }

    private void failPayment(String orderId) {
        transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            assertThat(order.getCurrentInvoice()).isNotNull();
            Invoice invoice = order.getCurrentInvoice();
            MockTrustClient mockTrustClient = (MockTrustClient) trustClient;
            mockTrustClient.paymentNotAuthorized(invoice.getPurchaseToken());
            return null;
        });
    }

    private String createSuccessfulOrder() {
        return createSuccessfulOrder(createOrderRequest(1).build());
    }

    private String createSuccessfulOrder(TCreateOrderReq createOrderReq) {
        var reservationResponse1 = ImReservationCreateResponseFactory.create().createReservationCreateResponse();
        reservationResponse1.setOrderId(1234567891);
        var reservationResponse2 = ImReservationCreateResponseFactory.create().createReservationCreateResponse();
        reservationResponse2.setOrderId(1234567890);
        when(imClient.reservationCreate(any(), any())).thenReturn(reservationResponse1).thenReturn(reservationResponse2);
        when(imClient.insurancePricing(any())).thenReturn(createInsurancePricingResponse());
        when(imClient.insuranceCheckout(any())).thenReturn(createInsuranceCheckoutResponse());
        when(imClient.getReturnAmount(any())).thenReturn(createReturnAmountResponse());
        when(imClient.autoReturn(any())).thenReturn(createAutoReturnResponse());
        TCreateOrderRsp resp = client.createOrder(createOrderReq);
        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");
        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED &&
                        getTrainReservation(rsp3).getIsRebookingFor() == null,
                Duration.ofSeconds(10), "Order must be in OS_RESERVED state before rebooking");

        client.checkout(TCheckoutReq.newBuilder().setOrderId(orderId).build());
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_return_url").build()); // safely ignoring response, as we don't need payment url
        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                Duration.ofSeconds(10), "Invoice must be in IS_WAIT_FOR_PAYMENT state");
        UUID orderItemId1 = ProtoUtils.fromStringOrNull(client.getOrderInfo(TGetOrderInfoReq.newBuilder()
                .setOrderId(orderId).build()).getResult().getService(0).getServiceId());
        assertThat(orderItemId1).isNotNull();
        transactionTemplate.execute(ignore -> {
            TrainOrderItem item = trainOrderItemRepository.getOne(orderItemId1);
            item.setExpiresAt(Instant.now());
            trainOrderItemRepository.save(item);
            return null;
        });
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_WAITING_PAYMENT &&
                        getTrainReservation(rsp3).getIsRebookingFor() != null,
                Duration.ofSeconds(10), "Order must be in OS_WAITING_PAYMENT state after rebooking");
        authorizePayment(orderId);
        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CONFIRMED,
                Duration.ofSeconds(10), "Order must be in OS_CONFIRMED state");

        return orderId;
    }

    private TrainReservation getPayload(TOrderInfo orderInfo) {
        return getPayload(orderInfo.getService(0).getServiceInfo().getPayload());
    }

    private TrainReservation getPayload(TJson payloadJson) {
        return ProtoUtils.fromTJson(payloadJson, TrainReservation.class);
    }

    private TCreateOrderReq createOrderRequestRoundTrip(int passengers) {
        var factory = new TrainOrderItemFactory();
        TrainReservation payload = factory.createTrainReservation();
        payload.setPartnerMultiOrder(true);
        for (int i = 1; i < passengers; i++) {
            TrainPassenger extraPassenger = factory.createTrainPassenger();
            extraPassenger.setFirstName(extraPassenger.getFirstName() + 'A' + i);
            extraPassenger.setDocumentNumber(extraPassenger.getDocumentNumber() + i);
            payload.getPassengers().add(extraPassenger);
        }
        TrainReservation payloadBack = factory.createTrainReservationBack();
        payloadBack.setPartnerMultiOrder(true);
        for (int i = 1; i < passengers; i++) {
            TrainPassenger extraPassenger = factory.createTrainPassenger();
            extraPassenger.setFirstName(extraPassenger.getFirstName() + 'A' + i);
            extraPassenger.setDocumentNumber(extraPassenger.getDocumentNumber() + i);
            payload.getPassengers().add(extraPassenger);
        }
        return createOrderRequest(ProtoUtils.toTJson(payload).getValue(), ProtoUtils.toTJson(payloadBack).getValue())
                .build();
    }

    private static TrainReservation obtainPayload(TGetOrderInfoRsp response) {
        return ProtoUtils.fromTJson(
                response.getResult().getService(0).getServiceInfo().getPayload(), TrainReservation.class);
    }

    @Override
    protected TCreateServiceReq.Builder getServiceForPayload(String payloadJson) {
        return TCreateServiceReq.newBuilder()
                .setServiceType(EServiceType.PT_TRAIN)
                .setSourcePayload(TJson.newBuilder().setValue(payloadJson).build());
    }

    @TestConfiguration
    static class IntegrationTestConfiguration {
        @Bean
        @Primary
        public TrustClient trainTrustClient() {
            return new MockTrustClient();
        }

        @Bean
        public TrustClientProvider trustClientProvider(@Autowired TrustClient trainTrustClient) {
            return paymentProfile -> trainTrustClient;
        }

        @Bean
        @Primary
        public TrainTariffInfoDataProvider trainTariffInfoDataProvider() {
            var tariffInfos = Arrays.asList(
                    TTrainTariffInfo.newBuilder().setId(1)
                            .setCode("full")
                            .setTitleRu("Полный")
                            .setImRequestCode("Full")
                            .setImResponseCodes("Full")
                            .setWithoutPlace(false)
                            .setMinAge(0)
                            .setMinAgeIncludesBirthday(false)
                            .setMaxAge(150)
                            .setMaxAgeIncludesBirthday(false)
                            .setNeedDocument(false).build(),

                    TTrainTariffInfo.newBuilder().setId(2)
                            .setCode("senior")
                            .setTitleRu("Сеньор")
                            .setImRequestCode("Senior")
                            .setImResponseCodes("Senior")
                            .setWithoutPlace(false)
                            .setMinAge(0)
                            .setMinAgeIncludesBirthday(false)
                            .setMaxAge(150)
                            .setMaxAgeIncludesBirthday(false)
                            .setNeedDocument(false).build()
            );

            TrainTariffInfoDataProviderProperties config = new TrainTariffInfoDataProviderProperties();
            config.setTablePath("tablePath");
            config.setIndexPath("./train-tariff-index");
            config.setProxy(new ArrayList<>());

            TestLuceneIndexBuilder<TTrainTariffInfo> luceneIndexBuilder = new TestLuceneIndexBuilder<TTrainTariffInfo>()
                    .setLuceneData(tariffInfos);

            return new TrainTariffInfoService(config, luceneIndexBuilder);
        }
    }
}
