package ru.yandex.travel.orders.integration.train;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.UUID;

import lombok.extern.slf4j.Slf4j;
import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.TestConfiguration;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.TestPropertySource;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TJson;
import ru.yandex.travel.dicts.rasp.proto.TTrainTariffInfo;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.commons.proto.ETrainChangeElectronicRegistrationOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainInsuranceCheckoutConfirmOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainInsuranceCheckoutOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainInsurancePricingOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainRefundCheckoutOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainRefundPricingOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainReservationConfirmOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainReservationCreateOutcome;
import ru.yandex.travel.orders.commons.proto.TTrainTestContext;
import ru.yandex.travel.orders.configurations.TrainTariffInfoDataProviderProperties;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.integration.train.factories.TrainOrderItemFactory;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.proto.ERefundPartType;
import ru.yandex.travel.orders.proto.TAddInsuranceReq;
import ru.yandex.travel.orders.proto.TCalculateRefundReqV2;
import ru.yandex.travel.orders.proto.TChangeTrainRegistrationStatusReq;
import ru.yandex.travel.orders.proto.TCheckoutReq;
import ru.yandex.travel.orders.proto.TCreateOrderReq;
import ru.yandex.travel.orders.proto.TCreateOrderRsp;
import ru.yandex.travel.orders.proto.TCreateServiceReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.proto.TOrderInfo;
import ru.yandex.travel.orders.proto.TOrderServiceInfo;
import ru.yandex.travel.orders.proto.TRefundPartContext;
import ru.yandex.travel.orders.proto.TReserveReq;
import ru.yandex.travel.orders.proto.TStartCancellationReq;
import ru.yandex.travel.orders.proto.TStartPaymentReq;
import ru.yandex.travel.orders.proto.TStartRefundReq;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.services.cloud.s3.InMemoryS3Object;
import ru.yandex.travel.orders.services.cloud.s3.S3Service;
import ru.yandex.travel.orders.services.mock.MockImClient;
import ru.yandex.travel.orders.services.mock.MockTrustClient;
import ru.yandex.travel.orders.services.orders.RefundPartsService;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.train.tariffinfo.TrainTariffInfoDataProvider;
import ru.yandex.travel.orders.services.train.tariffinfo.TrainTariffInfoService;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.train.model.TrainTicketRefundStatus;
import ru.yandex.travel.train.partners.im.model.ImBlankStatus;
import ru.yandex.travel.yt_lucene_index.TestLuceneIndexBuilder;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyBoolean;
import static org.mockito.Mockito.when;
import static ru.yandex.travel.commons.proto.ProtoUtils.fromTJson;
import static ru.yandex.travel.orders.integration.IntegrationUtils.waitForPredicateOrTimeout;

@TestPropertySource(
        properties = {
                "quartz.enabled=true",
                "workflow-processing.pending-workflow-polling-interval=10ms",
                "trust-hotels.clearing-refresh-timeout=1s",
                "trust-hotels.payment-refresh-timeout=1s",
                "single-node.auto-start=true",
                "train-workflow.check-ticket-refund-delay=1s",
                "train-workflow.check-ticket-refund-task-period=10ms",
                "train-workflow.check-ticket-refund-max-tries=50",
                "train-workflow.check-passenger-discounts-enabled=true",
                "train-workflow.refund.return-fee-time=0s",
                "train-workflow.check-expiration-task-period=100ms",
                "train-workflow.insurance-enabled=true",
                "mock-im-client.enabled=true",
                // needed for testOrderExpired
                "generic-workflow.train-rebooking-enabled=false"
        }
)
@DirtiesContext
@Slf4j
@SuppressWarnings("ResultOfMethodCallIgnored")
public class GenericRoundTripTrainOrderFlowTests extends AbstractTrainOrderFlowTest {

    private static final Duration TIMEOUT = Duration.ofSeconds(10);
    @Rule
    public TestName testName = new TestName();

    @Autowired
    private TransactionTemplate transactionTemplate;

    @Autowired
    private OrderRepository orderRepository;

    @Autowired
    private TrustClient trustClient;

    @MockBean
    private S3Service s3Service;

    @Before
    public void setUpCredentialsContext() {
        log.info("Starting the {} test", testName.getMethodName());
        when(s3Service.readObject(any())).thenReturn(InMemoryS3Object.builder().data(new byte[0]).build());
    }

    @Test
    public void testOrderReserved() {
        TCreateOrderReq.Builder createOrderReq = createOrderRequest(1);
        TCreateOrderRsp resp = client.createOrder(createOrderReq.build());
        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                TIMEOUT, "Order must be in OS_RESERVED state");
        getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getService(0).getServiceInfo().getGenericOrderItemState())
                .isEqualTo(EOrderItemState.IS_RESERVED);
        var ticket = ProtoUtils.fromTJson(
                getOrderInfoRsp.getResult().getService(0).getServiceInfo().getPayload(),
                TrainReservation.class).getPassengers().get(0).getTicket();
        assertThat(ticket.getImBlankStatus()).isNull();
        assertThat(ticket.isPendingElectronicRegistration()).isEqualTo(false);

        addInsurance(resp.getNewOrder());
        waitUntilInsuranceIsReserved(orderId);

        client.checkout(TCheckoutReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_WAITING_PAYMENT,
                TIMEOUT, "Order must be in OS_WAITING_PAYMENT state");

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_failed_payment_url").build()); // safely ignoring response, as we don't need
        // payment url

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getCurrentInvoice().getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                TIMEOUT, "Invoice must be in IS_WAIT_FOR_PAYMENT state");

        failPayment(orderId);

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getCurrentInvoice().getTrustInvoiceState() == ETrustInvoiceState.IS_PAYMENT_NOT_AUTHORIZED,
                TIMEOUT, "Current invoice must be ins IS_PAYMENT_NOT_AUTHORIZED state");

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_success_payment_url").build()); // safely ignoring response, as we don't need

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getCurrentInvoice().getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                TIMEOUT, "Invoice must be in IS_WAIT_FOR_PAYMENT state");

        authorizePayment(orderId);

        TGetOrderInfoRsp confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CONFIRMED,
                TIMEOUT, "Order must be in OS_CONFIRMED state");
        assertThat(confirmedOrderResponse.getResult().getServiceCount()).isEqualTo(2);
        assertThat(confirmedOrderResponse.getResult().getInvoiceCount()).isEqualTo(2);

        var reservation = getTrainReservation(confirmedOrderResponse, 0);
        assertThat(reservation.getPassengers().get(0).getTicket().getBookedTariffCode()).isEqualTo("full");
        assertThat(reservation.getPassengers().get(0).getTicket().getRawTariffName()).isEqualTo("Полный");
    }

    @Test
    public void testInsuranceAutoReturn() {
        var createOrderReq = createOrderRequest(1);
        createOrderReq.getCreateServicesBuilder(0).getTrainTestContextBuilder()
                .setInsuranceCheckoutConfirmOutcome(ETrainInsuranceCheckoutConfirmOutcome.ICCO_FAILURE);
        TCreateOrderRsp resp = client.createOrder(createOrderReq.build());
        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                TIMEOUT, "Order must be in OS_RESERVED state");

        addInsurance(resp.getNewOrder());
        waitUntilInsuranceIsReserved(orderId);

        client.checkout(TCheckoutReq.newBuilder().setOrderId(orderId).build());
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_return_url").build()); // safely ignoring response, as we don't need payment url

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                TIMEOUT, "Invoice must be in IS_WAIT_FOR_PAYMENT state");
        authorizePayment(orderId);

        TGetOrderInfoRsp confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CONFIRMED,
                TIMEOUT, "Order must be in OS_CONFIRMED state");
        assertThat(confirmedOrderResponse.getResult().getServiceCount()).isEqualTo(2);
        assertThat(getTrainReservationItem(confirmedOrderResponse.getResult()).getServiceInfo().getGenericOrderItemState())
                .isEqualTo(EOrderItemState.IS_CONFIRMED);
        assertThat(getTrainReservation(confirmedOrderResponse, 0).getInsuranceStatus())
                .isEqualTo(InsuranceStatus.AUTO_RETURN);
        assertThat(getTrainReservation(confirmedOrderResponse, 1).getInsuranceStatus())
                .isEqualTo(InsuranceStatus.AUTO_RETURN);
    }

    @Test
    public void testInsuranceImmediateClear() {
        TCreateOrderReq.Builder createOrderReq = createOrderRequest(1);
        TCreateOrderRsp resp = client.createOrder(createOrderReq.build());
        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                TIMEOUT, "Order must be in OS_RESERVED state");

        addInsurance(resp.getNewOrder());
        waitUntilInsuranceIsReserved(orderId);

        client.checkout(TCheckoutReq.newBuilder().setOrderId(orderId).build());
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_return_url").build()); // safely ignoring response, as we don't need payment url

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                TIMEOUT, "Invoice must be in IS_WAIT_FOR_PAYMENT state");
        authorizePayment(orderId);

        TGetOrderInfoRsp confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CONFIRMED
                        && getTrainReservation(rsp1, 0).getInsuranceStatus() == InsuranceStatus.CHECKED_OUT
                        && rsp1.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_CLEARED,
                TIMEOUT,
                "Order must be in OS_CONFIRMED state and insurance in CHECKED_OUT and invoice in IS_CLEARED");
        assertThat(confirmedOrderResponse.getResult().getServiceCount()).isEqualTo(2);
    }

    @Test
    public void testDoubleDiscountWithCancel() {
        var factory = new TrainOrderItemFactory();
        factory.setTariffCode("senior");
        factory.setDocumentNumber("1234123454");
        var trainReservation = factory.createTrainReservation();
        trainReservation.setPartnerMultiOrder(true);
        var trainReservationBack = factory.createTrainReservationBack();
        trainReservationBack.setPartnerItemIndex(1);
        trainReservationBack.setPartnerMultiOrder(true);
        var orderId = createOrderWaitingForPayment(createOrderRequest(trainReservation, trainReservationBack),
                true, true);
        var getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        var getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getPayload(getOrderInfoRsp.getResult()).getPassengers().get(0).isDiscountDenied()).isFalse();
        assertThat(getPayload(getOrderInfoRsp.getResult()).getPassengers().get(0).getTariffCode()).isEqualTo("senior");

        // 1st senior is reserved, try to reserve 2nd senior
        var createOrderReq2 = createOrderRequest(trainReservation, trainReservationBack);
        var resp2 = client.createOrder(createOrderReq2);
        var orderId2 = resp2.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId2).build());
        waitForPredicateOrTimeout(client, orderId2,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                TIMEOUT, "Order must be in OS_RESERVED state");
        TGetOrderInfoReq getOrderInfoRequest2 = TGetOrderInfoReq.newBuilder().setOrderId(orderId2).build();
        TGetOrderInfoRsp getOrderInfoRsp2 = client.getOrderInfo(getOrderInfoRequest2);
        assertThat(getPayload(getOrderInfoRsp2.getResult()).getPassengers().get(0).isDiscountDenied()).isTrue();
        assertThat(getPayload(getOrderInfoRsp2.getResult()).getPassengers().get(0).getTariffCode()).isEqualTo("full");

        // 2nd senior reserved as full, lets cancel 1st senior

        client.startCancellation(TStartCancellationReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CANCELLED,
                TIMEOUT, "Order must be in OS_CANCELLED state");

        // 1st senior refunded, try to reserve 2nd senior again

        createOrderReq2 = createOrderRequest(trainReservation, trainReservationBack);
        resp2 = client.createOrder(createOrderReq2);
        orderId2 = resp2.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId2).build());
        waitForPredicateOrTimeout(client, orderId2,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                TIMEOUT, "Order must be in OS_RESERVED state");
        getOrderInfoRequest2 = TGetOrderInfoReq.newBuilder().setOrderId(orderId2).build();
        getOrderInfoRsp2 = client.getOrderInfo(getOrderInfoRequest2);
        assertThat(getPayload(getOrderInfoRsp2.getResult()).getPassengers().get(0).isDiscountDenied()).isFalse();
        assertThat(getPayload(getOrderInfoRsp2.getResult()).getPassengers().get(0).getTariffCode()).isEqualTo("senior");
    }

    @Test
    public void testOrderRefund() {
        var orderId = createSuccessfulOrder();
        var calculateRefundRsp = client.calculateRefundV2(TCalculateRefundReqV2.newBuilder().setOrderId(orderId)
                .addContext(RefundPartsService.partContextToString(TRefundPartContext.newBuilder()
                        .setType(ERefundPartType.RPT_ORDER)
                        .build()))
                .build());
        assertThat(calculateRefundRsp.getPenaltyAmount().getAmount()).isPositive();
        assertThat(calculateRefundRsp.getExpiresAt().getSeconds()).isGreaterThan(Instant.now().getEpochSecond());

        client.startRefund(TStartRefundReq.newBuilder().setOrderId(orderId)
                .setRefundToken(calculateRefundRsp.getRefundToken()).build());

        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_REFUNDED &&
                        getTrainReservation(rsp1, 0).getPassengers().get(0).getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDED &&
                        getTrainReservation(rsp1, 1).getPassengers().get(0).getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDED,
                TIMEOUT, "Order must be REFUNDED");
    }

    @Test
    public void testOrderExpired() {
        var orderId = createOrderWaitingForPayment(null, false, false);
        transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            order.getOrderItems().get(0).setExpiresAt(Instant.now().minusSeconds(1));
            return null;
        });
        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CANCELLED, TIMEOUT,
                "Order must be in OS_CANCELLED state");
    }

    @Test
    public void testCancelOnConfirmationFailed() {
        var createOrderReq = createOrderRequest(1);
        createOrderReq.getCreateServicesBuilder(0).getTrainTestContextBuilder()
                .setReservationConfirmOutcome(ETrainReservationConfirmOutcome.RCOO_FAILURE);
        createOrderReq.getCreateServicesBuilder(1).getTrainTestContextBuilder()
                .setReservationConfirmOutcome(ETrainReservationConfirmOutcome.RCOO_FAILURE);
        var orderId = createOrderWaitingForPayment(createOrderReq.build(), true, true);

        assertThat(getInvoiceTotal(orderId)).isPositive();
        authorizePayment(orderId);

        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CANCELLED &&
                        rsp1.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_CANCELLED,
                TIMEOUT, "Order must be in OS_CANCELLED state");
        assertThat(getInvoiceTotal(orderId)).isZero();
    }

    @Test
    public void testCancelOrder() {
        var orderId = createOrderWaitingForPayment();
        var startCancellationReq = TStartCancellationReq.newBuilder().setOrderId(orderId).build();

        client.startCancellation(startCancellationReq);

        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CANCELLED,
                TIMEOUT, "Order must be in OS_CANCELLED state");

        authorizePayment(orderId);

        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CANCELLED &&
                        rsp1.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_CANCELLED,
                TIMEOUT,"Order must be in OS_CANCELLED state");
    }

    private String createOrderWaitingForPayment() {
        return createOrderWaitingForPayment(null, true, true);
    }

    private String createOrderWaitingForPayment(TCreateOrderReq createOrderReq, boolean startPayment, boolean checkout) {
        if (createOrderReq == null) {
            createOrderReq = createOrderRequest(1).build();
        }
        TCreateOrderRsp resp = client.createOrder(createOrderReq);

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                TIMEOUT, "Order must be in OS_RESERVED state");
        getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getService(0).getServiceInfo().getGenericOrderItemState())
                .isEqualTo(EOrderItemState.IS_RESERVED);

        if (checkout) {
            client.checkout(TCheckoutReq.newBuilder().setOrderId(orderId).build());
        }

        if (startPayment) {
            client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                    .setSource("desktop")
                    .setOrderId(orderId)
                    .setReturnUrl("some_return_url").build()); // safely ignoring response, as we don't need payment url

            waitForPredicateOrTimeout(client, orderId,
                    rsp2 -> rsp2.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                    TIMEOUT, "Invoice must be in IS_WAIT_FOR_PAYMENT state");
        }

        return orderId;
    }

    private BigDecimal getInvoiceTotal(String orderId) {
        return transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            assertThat(order.getCurrentInvoice()).isNotNull();
            Invoice invoice = order.getCurrentInvoice();
            BigDecimal total = BigDecimal.ZERO;
            for (var i : invoice.getInvoiceItems()) {
                total = total.add(i.getPrice());
            }
            return total;
        });
    }

    private TrainReservation getTrainReservation(TGetOrderInfoRsp response, int serviceIndex) {
        var payloadJson = response.getResult().getService(serviceIndex).getServiceInfo().getPayload();
        return fromTJson(payloadJson, TrainReservation.class);
    }

    private TOrderServiceInfo getTrainReservationItem(TOrderInfo order) {
        return order.getServiceList().stream()
                .filter(s -> s.getServiceType() == EServiceType.PT_TRAIN)
                .findAny()
                .orElse(null);
    }

    private void addInsurance(TOrderInfo order) {
        client.addInsurance(TAddInsuranceReq.newBuilder().setOrderId(order.getOrderId()).build());
    }

    private void waitUntilInsuranceIsReserved(String orderId) {
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED
                        && getTrainReservation(rsp3, 0).getInsuranceStatus() == InsuranceStatus.CHECKED_OUT
                        && getTrainReservation(rsp3, 1).getInsuranceStatus() == InsuranceStatus.CHECKED_OUT,
                TIMEOUT, "Order must be in OS_RESERVED state and Insurance must be " +
                        "CHECKED_OUT");
    }

    private void authorizePayment(String orderId) {
        transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            assertThat(order.getCurrentInvoice()).isNotNull();
            Invoice invoice = order.getCurrentInvoice();
            String purchaseToken = invoice.getPurchaseToken();
            MockTrustClient mockTrustClient = (MockTrustClient) trustClient;
            mockTrustClient.paymentAuthorized(purchaseToken);
            return null;
        });
    }

    private void failPayment(String orderId) {
        transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            assertThat(order.getCurrentInvoice()).isNotNull();
            Invoice invoice = order.getCurrentInvoice();
            MockTrustClient mockTrustClient = (MockTrustClient) trustClient;
            mockTrustClient.paymentNotAuthorized(invoice.getPurchaseToken());
            return null;
        });
    }

    @Test
    public void testOrderReservationCancelled() {
        var createOrderReq = createOrderRequest(1);
        createOrderReq.getCreateServicesBuilder(0).getTrainTestContextBuilder()
                .setReservationCreateOutcome(ETrainReservationCreateOutcome.RCRO_FAILURE);
        TCreateOrderRsp resp = client.createOrder(createOrderReq.build());

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build(); // we'll
        // reuse it
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_CANCELLED,
                TIMEOUT, "Order must be in OS_CANCELLED state");
    }

    @Test
    public void testImValidationError() {
        var createOrderReq = createOrderRequest(1);
        createOrderReq.getOwnerBuilder().setEmail(MockImClient.INVALID_EMAIL);
        TCreateOrderRsp resp = client.createOrder(createOrderReq.build());

        var orderId = resp.getNewOrder().getOrderId();
        var getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        var getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                TIMEOUT, "Order must be in OS_RESERVED state");
        getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getService(0).getServiceInfo().getGenericOrderItemState())
                .isEqualTo(EOrderItemState.IS_RESERVED);

        assertThat(getTrainReservation(getOrderInfoRsp, 0).getPassengers().get(0).isUseEmailForReservation()).isFalse();
    }

    @Test
    public void testChangeRegistrationStatus() {
        var orderId = createSuccessfulOrder();

        TGetOrderInfoRsp orderInfo = waitForPredicateOrTimeout(client, orderId,
                rsp -> getTrainReservation(rsp, 0).getPassengers().get(0).getTicket().getImBlankStatus()
                        == ImBlankStatus.REMOTE_CHECK_IN,
                TIMEOUT, "OrderItem must be in REMOTE_CHECK_IN state");

        int blankId = getTrainReservation(orderInfo, 0).getPassengers().get(0).getTicket().getBlankId();
        TChangeTrainRegistrationStatusReq request = TChangeTrainRegistrationStatusReq.newBuilder()
                .setOrderId(orderId)
                .setEnabled(false)
                .addBlankIds(blankId).build();
        client.changeTrainRegistrationStatus(request);

        waitForPredicateOrTimeout(client, orderId,
                rsp -> getTrainReservation(rsp, 0).getPassengers().get(0).getTicket().getImBlankStatus()
                        == ImBlankStatus.NO_REMOTE_CHECK_IN && !rsp.getResult().getUserActionScheduled(),
                TIMEOUT, "OrderItem must be in NO_REMOTE_CHECK_IN state and not scheduled");
    }

    private String createSuccessfulOrder() {
        return createSuccessfulOrder(createOrderRequest(1).build());
    }

    private String createSuccessfulOrder(TCreateOrderReq createOrderReq) {
        TCreateOrderRsp resp = client.createOrder(createOrderReq);
        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                TIMEOUT, "Order must be in OS_RESERVED state");

        client.checkout(TCheckoutReq.newBuilder().setOrderId(orderId).build());
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_return_url").build()); // safely ignoring response, as we don't need payment url

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                TIMEOUT, "Invoice must be in IS_WAIT_FOR_PAYMENT state");
        authorizePayment(orderId);

        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CONFIRMED,
                TIMEOUT, "Order must be in OS_CONFIRMED state");

        return orderId;
    }

    private TrainReservation getPayload(TOrderInfo orderInfo) {
        return getPayload(orderInfo.getService(0).getServiceInfo().getPayload());
    }

    private TrainReservation getPayload(TJson payloadJson) {
        return ProtoUtils.fromTJson(payloadJson, TrainReservation.class);
    }

    private TCreateOrderReq createOrderRequest(TrainReservation payload, TrainReservation payloadBack) {
        return createOrderRequest(ProtoUtils.toTJson(payload).getValue(), ProtoUtils.toTJson(payloadBack).getValue())
                .build();
    }

    @Override
    protected TCreateServiceReq.Builder getServiceForPayload(String payloadJson) {
        return TCreateServiceReq.newBuilder()
                .setServiceType(EServiceType.PT_TRAIN)
                .setSourcePayload(TJson.newBuilder().setValue(payloadJson).build())
                .setTrainTestContext(TTrainTestContext.newBuilder()
                        .setInsurancePricingOutcome(ETrainInsurancePricingOutcome.IPO_SUCCESS)
                        .setInsuranceCheckoutOutcome(ETrainInsuranceCheckoutOutcome.ICO_SUCCESS)
                        .setInsuranceCheckoutConfirmOutcome(ETrainInsuranceCheckoutConfirmOutcome.ICCO_SUCCESS)
                        .setRefundPricingOutcome(ETrainRefundPricingOutcome.RPO_SUCCESS)
                        .setRefundCheckoutOutcome(ETrainRefundCheckoutOutcome.RCO_SUCCESS)
                        .setReservationCreateOutcome(ETrainReservationCreateOutcome.RCRO_SUCCESS)
                        .setReservationConfirmOutcome(ETrainReservationConfirmOutcome.RCOO_SUCCESS)
                        .setTrainChangeElectronicRegistrationOutcome(ETrainChangeElectronicRegistrationOutcome.CERO_SUCCESS)
                        .build()
                );
    }

    @Override
    protected TCreateOrderReq.Builder createOrderRequest(TrainReservation payload,
                                                         TrainOrderItemFactory factory) {
        payload.setPartnerMultiOrder(true);
        TrainReservation payloadBack = factory.createTrainReservationBack();
        payloadBack.setPartnerMultiOrder(true);
        return createOrderRequest(ProtoUtils.toTJson(payload).getValue(), ProtoUtils.toTJson(payloadBack).getValue());
    }

    @TestConfiguration
    static class IntegrationTestConfiguration {
        @Bean
        @Primary
        public TrustClient trainTrustClient() {
            return new MockTrustClient();
        }

        @Bean
        public TrustClientProvider trustClientProvider(@Autowired TrustClient trainTrustClient) {
            return paymentProfile -> trainTrustClient;
        }

        @Bean
        @Primary
        public TrainTariffInfoDataProvider trainTariffInfoDataProvider() {
            var tariffInfos = Arrays.asList(
                    TTrainTariffInfo.newBuilder().setId(1)
                            .setCode("full")
                            .setTitleRu("Полный")
                            .setImRequestCode("Full")
                            .setImResponseCodes("Full")
                            .setWithoutPlace(false)
                            .setMinAge(0)
                            .setMinAgeIncludesBirthday(false)
                            .setMaxAge(150)
                            .setMaxAgeIncludesBirthday(false)
                            .setNeedDocument(false).build(),

                    TTrainTariffInfo.newBuilder().setId(2)
                            .setCode("senior")
                            .setTitleRu("Сеньор")
                            .setImRequestCode("Senior")
                            .setImResponseCodes("Senior")
                            .setWithoutPlace(false)
                            .setMinAge(0)
                            .setMinAgeIncludesBirthday(false)
                            .setMaxAge(150)
                            .setMaxAgeIncludesBirthday(false)
                            .setNeedDocument(false).build()
            );

            TrainTariffInfoDataProviderProperties config = new TrainTariffInfoDataProviderProperties();
            config.setTablePath("tablePath");
            config.setIndexPath("./train-tariff-index");
            config.setProxy(new ArrayList<>());

            TestLuceneIndexBuilder<TTrainTariffInfo> luceneIndexBuilder = new TestLuceneIndexBuilder<TTrainTariffInfo>()
                    .setLuceneData(tariffInfos);

            return new TrainTariffInfoService(config, luceneIndexBuilder);
        }
    }
}
