package ru.yandex.travel.orders.integration.train;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Comparator;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

import io.grpc.StatusRuntimeException;
import lombok.extern.slf4j.Slf4j;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;
import org.junit.jupiter.api.Assertions;
import org.junit.rules.TestName;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.TestConfiguration;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;
import org.springframework.test.context.TestPropertySource;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.TPrice;
import ru.yandex.travel.dicts.rasp.proto.TTrainTariffInfo;
import ru.yandex.travel.orders.admin.proto.OrdersAdminInterfaceV1Grpc;
import ru.yandex.travel.orders.admin.proto.TByFiscalItemRefund;
import ru.yandex.travel.orders.admin.proto.TManualMoneyRefundReq;
import ru.yandex.travel.orders.admin.proto.TUpdateTrainTicketsReq;
import ru.yandex.travel.orders.commons.proto.ECancellationReason;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.commons.proto.ETrainChangeElectronicRegistrationOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainInsuranceCheckoutConfirmOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainRefundCheckoutOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainReservationConfirmOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainReservationCreateOutcome;
import ru.yandex.travel.orders.commons.proto.TTrainOfficeAction;
import ru.yandex.travel.orders.configurations.TrainTariffInfoDataProviderProperties;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.grpc.OrdersAdminService;
import ru.yandex.travel.orders.integration.IntegrationUtils;
import ru.yandex.travel.orders.integration.train.factories.TrainOrderItemFactory;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.proto.EOrderRefundState;
import ru.yandex.travel.orders.proto.ERefundPartState;
import ru.yandex.travel.orders.proto.ERefundPartType;
import ru.yandex.travel.orders.proto.TAddInsuranceReq;
import ru.yandex.travel.orders.proto.TCalculateRefundReqV2;
import ru.yandex.travel.orders.proto.TChangeTrainRegistrationStatusReq;
import ru.yandex.travel.orders.proto.TCheckoutReq;
import ru.yandex.travel.orders.proto.TCreateOrderReq;
import ru.yandex.travel.orders.proto.TCreateOrderRsp;
import ru.yandex.travel.orders.proto.TDownloadBlankToken;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.proto.TOrderInfo;
import ru.yandex.travel.orders.proto.TOrderServiceInfo;
import ru.yandex.travel.orders.proto.TRefundPartContext;
import ru.yandex.travel.orders.proto.TRefundPartInfo;
import ru.yandex.travel.orders.proto.TReserveReq;
import ru.yandex.travel.orders.proto.TStartCancellationReq;
import ru.yandex.travel.orders.proto.TStartPaymentReq;
import ru.yandex.travel.orders.proto.TStartRefundReq;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.services.cloud.s3.InMemoryS3Object;
import ru.yandex.travel.orders.services.cloud.s3.S3Service;
import ru.yandex.travel.orders.services.mock.MockImClient;
import ru.yandex.travel.orders.services.mock.MockTrustClient;
import ru.yandex.travel.orders.services.orders.RefundPartsService;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.payments.model.TrustCreateRefundRequest;
import ru.yandex.travel.orders.services.payments.model.TrustCreateRefundResponse;
import ru.yandex.travel.orders.services.train.tariffinfo.TrainTariffInfoDataProvider;
import ru.yandex.travel.orders.services.train.tariffinfo.TrainTariffInfoService;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.train.model.TrainTicketRefundStatus;
import ru.yandex.travel.train.partners.im.model.ImBlankStatus;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.yt_lucene_index.TestLuceneIndexBuilder;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyBoolean;
import static org.mockito.ArgumentMatchers.anyLong;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static ru.yandex.travel.commons.proto.ProtoUtils.fromTJson;
import static ru.yandex.travel.orders.integration.IntegrationUtils.waitForPredicateOrTimeout;

@TestPropertySource(properties = {
        "quartz.enabled=true",
        "workflow-processing.pending-workflow-polling-interval=10ms",
        "trust-hotels.clearing-refresh-timeout=1s",
        "trust-hotels.payment-refresh-timeout=1s",
        "single-node.auto-start=true",
        "train-workflow.check-ticket-refund-delay=1s",
        "train-workflow.check-ticket-refund-task-period=10ms",
        "train-workflow.check-ticket-refund-max-tries=4",
        "train-workflow.check-insurance-refund-delay=1s",
        "train-workflow.check-insurance-refund-task-period=10ms",
        "train-workflow.check-insurance-refund-max-tries=50",
        "train-workflow.check-passenger-discounts-enabled=true",
        "train-workflow.check-expiration-task-period=100ms",
        "mock-im-client.enabled=true",
        // needed for testOrderExpired
        "generic-workflow.train-rebooking-enabled=false"
})
@Slf4j
@SuppressWarnings("ResultOfMethodCallIgnored")
public class GenericTrainOrderFlowTests extends AbstractTrainOrderFlowTest {

    private static final Duration TIMEOUT = Duration.ofSeconds(30);
    @Rule
    public TestName testName = new TestName();
    @Autowired
    private OrdersAdminService ordersAdminService;

    private OrdersAdminInterfaceV1Grpc.OrdersAdminInterfaceV1BlockingStub adminClient;

    @Autowired
    private TransactionTemplate transactionTemplate;

    @Autowired
    private OrderRepository orderRepository;

    @Autowired
    private TrustClient trustClient;

    @MockBean
    private S3Service s3Service;

    @Before
    public void setUpCredentialsContext() {
        log.info("Starting the {} test", testName.getMethodName());
        adminClient = IntegrationUtils.createAdminServerAndBlockingStub(cleanupRule, ordersAdminService);
        when(s3Service.readObject(any())).thenReturn(InMemoryS3Object.builder().data(new byte[0]).build());
    }

    @Test
    public void testOrderReserved() {
        TCreateOrderReq.Builder createOrderReq = createOrderRequest(1);
        TCreateOrderRsp resp = client.createOrder(createOrderReq.build());
        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                TIMEOUT, "Order must be in OS_RESERVED state");
        getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getService(0).getServiceInfo().getGenericOrderItemState())
                .isEqualTo(EOrderItemState.IS_RESERVED);
        var ticket = getTrainReservation(getOrderInfoRsp).getPassengers().get(0).getTicket();
        assertThat(ticket.getImBlankStatus()).isNull();
        assertThat(ticket.isPendingElectronicRegistration()).isEqualTo(false);

        addInsurance(resp.getNewOrder());
        waitUntilInsuranceIsReserved(orderId);

        client.checkout(TCheckoutReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_WAITING_PAYMENT,
                TIMEOUT, "Order must be in OS_WAITING_PAYMENT state");

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_failed_payment_url").build()); // safely ignoring response, as we don't need
        // payment url

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getCurrentInvoice().getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                TIMEOUT, "Invoice must be in IS_WAIT_FOR_PAYMENT state");

        failPayment(orderId);

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getCurrentInvoice().getTrustInvoiceState() == ETrustInvoiceState.IS_PAYMENT_NOT_AUTHORIZED,
                TIMEOUT, "Current invoice must be ins IS_PAYMENT_NOT_AUTHORIZED state");

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_success_payment_url").build()); // safely ignoring response, as we don't need

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getCurrentInvoice().getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                TIMEOUT, "Invoice must be in IS_WAIT_FOR_PAYMENT state");

        authorizePayment(orderId);

        TGetOrderInfoRsp confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CONFIRMED,
                TIMEOUT, "Order must be in OS_CONFIRMED state");
        assertThat(confirmedOrderResponse.getResult().getCancellationReason()).isEqualTo(ECancellationReason.CR_UNKNOWN);
        assertThat(confirmedOrderResponse.getResult().getServiceCount()).isEqualTo(1);
        assertThat(confirmedOrderResponse.getResult().getInvoiceCount()).isEqualTo(2);

        var reservation = getTrainReservation(confirmedOrderResponse);
        assertThat(reservation.getPassengers().get(0).getTicket().getBookedTariffCode()).isEqualTo("full");
        assertThat(reservation.getPassengers().get(0).getTicket().getRawTariffName()).isEqualTo("Полный");

        userHasConfirmedOrders(1);
    }

    @Test
    public void testInsuranceAutoReturn() {
        TCreateOrderReq.Builder createOrderReq = createOrderRequest(1);
        createOrderReq.getCreateServicesBuilder(0).getTrainTestContextBuilder()
                .setInsuranceCheckoutConfirmOutcome(ETrainInsuranceCheckoutConfirmOutcome.ICCO_FAILURE);
        TCreateOrderRsp resp = client.createOrder(createOrderReq.build());
        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                TIMEOUT, "Order must be in OS_RESERVED state");

        addInsurance(resp.getNewOrder());
        waitUntilInsuranceIsReserved(orderId);

        client.checkout(TCheckoutReq.newBuilder().setOrderId(orderId).build());
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_return_url").build()); // safely ignoring response, as we don't need payment url

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                TIMEOUT, "Invoice must be in IS_WAIT_FOR_PAYMENT state");
        authorizePayment(orderId);

        TGetOrderInfoRsp confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CONFIRMED,
                TIMEOUT, "Order must be in OS_CONFIRMED state");
        assertThat(confirmedOrderResponse.getResult().getServiceCount()).isEqualTo(1);
        assertThat(getTrainReservationItem(confirmedOrderResponse.getResult()).getServiceInfo().getGenericOrderItemState())
                .isEqualTo(EOrderItemState.IS_CONFIRMED);
        assertThat(getTrainReservation(confirmedOrderResponse).getInsuranceStatus())
                .isEqualTo(InsuranceStatus.AUTO_RETURN);
        assertThat(confirmedOrderResponse.getResult().getWorkflowState()).isEqualTo(EWorkflowState.WS_RUNNING);
    }

    @Test
    public void testInsuranceImmediateClear() {
        TCreateOrderReq.Builder createOrderReq = createOrderRequest(1);
        TCreateOrderRsp resp = client.createOrder(createOrderReq.build());
        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                TIMEOUT, "Order must be in OS_RESERVED state");

        addInsurance(resp.getNewOrder());
        waitUntilInsuranceIsReserved(orderId);

        client.checkout(TCheckoutReq.newBuilder().setOrderId(orderId).build());
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_return_url").build()); // safely ignoring response, as we don't need payment url

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                TIMEOUT, "Invoice must be in IS_WAIT_FOR_PAYMENT state");
        authorizePayment(orderId);

        TGetOrderInfoRsp confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CONFIRMED
                        && getTrainReservation(rsp1).getInsuranceStatus() == InsuranceStatus.CHECKED_OUT
                        && rsp1.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_CLEARED,
                TIMEOUT,
                "Order must be in OS_CONFIRMED state and insurance in CHECKED_OUT and invoice in IS_CLEARED");
        assertThat(confirmedOrderResponse.getResult().getServiceCount()).isEqualTo(1);
        assertThat(confirmedOrderResponse.getResult().getWorkflowState()).isEqualTo(EWorkflowState.WS_RUNNING);
    }

    @Test
    public void testDoubleDiscountWithRefund() {
        var factory = new TrainOrderItemFactory();
        factory.setTariffCode("senior");
        factory.setDocumentNumber("3709123454");
        var trainReservation = factory.createTrainReservation();

        var orderId = createSuccessfulOrder(createOrderRequest(trainReservation).build(), false);
        var getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        var getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getTrainReservation(getOrderInfoRsp).getPassengers().get(0).isDiscountDenied()).isFalse();
        assertThat(getTrainReservation(getOrderInfoRsp).getPassengers().get(0).getTariffCode()).isEqualTo("senior");

        // 1st senior is confirmed, try to reserve 2nd senior

        var createOrderReq2 = createOrderRequest(trainReservation);
        var resp2 = client.createOrder(createOrderReq2.build());
        var orderId2 = resp2.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId2).build());
        waitForPredicateOrTimeout(client, orderId2,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                TIMEOUT, "Order must be in OS_RESERVED state");
        TGetOrderInfoReq getOrderInfoRequest2 = TGetOrderInfoReq.newBuilder().setOrderId(orderId2).build();
        TGetOrderInfoRsp getOrderInfoRsp2 = client.getOrderInfo(getOrderInfoRequest2);
        assertThat(getTrainReservation(getOrderInfoRsp2).getPassengers().get(0).isDiscountDenied()).isTrue();
        assertThat(getTrainReservation(getOrderInfoRsp2).getPassengers().get(0).getTariffCode()).isEqualTo("full");

        // 2nd senior reserved as full, lets refund 1st senior

        var orderInfoRsp = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId)
                .setUpdateOrderOnTheFly(true).build());
        var calculateRefundRsp = client.calculateRefundV2(TCalculateRefundReqV2.newBuilder().setOrderId(orderId)
                .addContext(RefundPartsService.partContextToString(TRefundPartContext.newBuilder()
                        .setServiceId(orderInfoRsp.getResult().getService(0).getServiceId())
                        .setType(ERefundPartType.RPT_SERVICE)
                        .build()))
                .build());
        client.startRefund(TStartRefundReq.newBuilder().setOrderId(orderId)
                .setRefundToken(calculateRefundRsp.getRefundToken()).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_REFUNDED &&
                        getTrainReservation(rsp1).getPassengers().get(0).getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDED,
                TIMEOUT, "Ticket must be REFUNDED");

        // 1st senior refunded, try to reserve 2nd senior again

        createOrderReq2 = createOrderRequest(trainReservation);
        resp2 = client.createOrder(createOrderReq2.build());
        orderId2 = resp2.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId2).build());
        waitForPredicateOrTimeout(client, orderId2,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                TIMEOUT, "Order must be in OS_RESERVED state");
        getOrderInfoRequest2 = TGetOrderInfoReq.newBuilder().setOrderId(orderId2).build();
        getOrderInfoRsp2 = client.getOrderInfo(getOrderInfoRequest2);
        assertThat(getTrainReservation(getOrderInfoRsp2).getPassengers().get(0).isDiscountDenied()).isFalse();
        assertThat(getTrainReservation(getOrderInfoRsp2).getPassengers().get(0).getTariffCode()).isEqualTo("senior");
    }

    @Test
    public void testDoubleDiscountWithCancel() {
        var factory = new TrainOrderItemFactory();
        factory.setTariffCode("senior");
        factory.setDocumentNumber("1234123454");
        var trainReservation = factory.createTrainReservation();
        TCreateOrderReq.Builder createOrderReq = createOrderRequest(trainReservation);
        var orderId = createOrderWaitingForPayment(createOrderReq.build(), true, true);
        var getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        var getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getTrainReservation(getOrderInfoRsp).getPassengers().get(0).isDiscountDenied()).isFalse();
        assertThat(getTrainReservation(getOrderInfoRsp).getPassengers().get(0).getTariffCode()).isEqualTo("senior");

        // 1st senior is reserved, try to reserve 2nd senior
        TCreateOrderReq.Builder createOrderReq2 = createOrderRequest(trainReservation);
        var resp2 = client.createOrder(createOrderReq2.build());
        var orderId2 = resp2.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId2).build());
        waitForPredicateOrTimeout(client, orderId2,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                TIMEOUT, "Order must be in OS_RESERVED state");
        TGetOrderInfoReq getOrderInfoRequest2 = TGetOrderInfoReq.newBuilder().setOrderId(orderId2).build();
        TGetOrderInfoRsp getOrderInfoRsp2 = client.getOrderInfo(getOrderInfoRequest2);
        assertThat(getTrainReservation(getOrderInfoRsp2).getPassengers().get(0).isDiscountDenied()).isTrue();
        assertThat(getTrainReservation(getOrderInfoRsp2).getPassengers().get(0).getTariffCode()).isEqualTo("full");

        // 2nd senior reserved as full, lets cancel 1st senior

        client.startCancellation(TStartCancellationReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CANCELLED,
                TIMEOUT, "Order must be in OS_CANCELLED state");

        // 1st senior refunded, try to reserve 2nd senior again

        createOrderReq2 = createOrderRequest(trainReservation);
        resp2 = client.createOrder(createOrderReq2.build());
        orderId2 = resp2.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId2).build());
        waitForPredicateOrTimeout(client, orderId2,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                TIMEOUT, "Order must be in OS_RESERVED state");
        getOrderInfoRequest2 = TGetOrderInfoReq.newBuilder().setOrderId(orderId2).build();
        getOrderInfoRsp2 = client.getOrderInfo(getOrderInfoRequest2);
        assertThat(getTrainReservation(getOrderInfoRsp2).getPassengers().get(0).isDiscountDenied()).isFalse();
        assertThat(getTrainReservation(getOrderInfoRsp2).getPassengers().get(0).getTariffCode()).isEqualTo("senior");
    }

    @Test
    public void testOrderRefund() {
        var orderId = createSuccessfulOrder(createOrderRequest(1).build(), false);
        var orderInfoRsp = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId)
                .setUpdateOrderOnTheFly(true).build());
        var calculateRefundRsp = client.calculateRefundV2(TCalculateRefundReqV2.newBuilder().setOrderId(orderId)
                .addContext(RefundPartsService.partContextToString(TRefundPartContext.newBuilder()
                        .setServiceId(orderInfoRsp.getResult().getService(0).getServiceId())
                        .setType(ERefundPartType.RPT_SERVICE)
                        .build()))
                .build());
        assertThat(calculateRefundRsp.getRefundAmount().getAmount()).isPositive();
        assertThat(calculateRefundRsp.getPenaltyAmount().getAmount()).isPositive();
        assertThat(calculateRefundRsp.getExpiresAt().getSeconds()).isGreaterThan(Instant.now().getEpochSecond());
        userHasConfirmedOrders(1);

        client.startRefund(TStartRefundReq.newBuilder().setOrderId(orderId)
                .setRefundToken(calculateRefundRsp.getRefundToken()).build());
        orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_REFUNDED &&
                        getTrainReservation(rsp1).getPassengers().get(0).getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDED,
                TIMEOUT, "Ticket must be REFUNDED");
        TRefundPartInfo refundedTicketPartInfo = orderInfoRsp.getResult().getRefundPartsList().stream()
                .filter(x -> x.getState() == ERefundPartState.RPS_REFUNDED && x.getType() == ERefundPartType.RPT_SERVICE_PART)
                .findFirst().orElseThrow();
        assertThat(refundedTicketPartInfo.hasRefund()).isTrue();
        assertThat(refundedTicketPartInfo.getRefund().getRefundBlankToken().getOneOfDownloadBlankParamsCase())
                .isEqualByComparingTo(TDownloadBlankToken.OneOfDownloadBlankParamsCase.TRAINDOWNLOADBLANKPARAMS);
        assertThat(refundedTicketPartInfo.getRefund().getRefundAmount().getAmount()).isPositive();
        userHasConfirmedOrders(0);
    }

    @Test
    public void testOrderRefundOneTicketFailed() {
        TCreateOrderReq.Builder createOrderReq = createOrderRequest(3);
        createOrderReq.getCreateServicesBuilder(0).getTrainTestContextBuilder()
                .setRefundCheckoutOutcome(ETrainRefundCheckoutOutcome.RCO_ODD_REFUND_FAILURE);
        String orderId = createSuccessfulOrder(createOrderReq.build(), false);
        var orderInfoRsp = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId)
                .setUpdateOrderOnTheFly(true).build());
        List<TRefundPartInfo> ticketRefundParts = orderInfoRsp.getResult().getRefundPartsList().stream()
                .filter(x -> x.getType() == ERefundPartType.RPT_SERVICE_PART)
                .sorted(Comparator.comparing(TRefundPartInfo::getKey))
                .collect(Collectors.toList());
        var calculateRefundRsp = client.calculateRefundV2(TCalculateRefundReqV2.newBuilder()
                .setOrderId(orderId)
                .addContext(ticketRefundParts.get(0).getContext())
                .addContext(ticketRefundParts.get(1).getContext())
                .build());
        assertThat(calculateRefundRsp.getRefundAmount().getAmount()).isPositive();
        assertThat(calculateRefundRsp.getPenaltyAmount().getAmount()).isPositive();
        assertThat(calculateRefundRsp.getExpiresAt().getSeconds()).isGreaterThan(Instant.now().getEpochSecond());
        userHasConfirmedOrders(1);

        client.startRefund(TStartRefundReq.newBuilder().setOrderId(orderId)
                .setRefundToken(calculateRefundRsp.getRefundToken()).build());
        orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CONFIRMED &&
                        getTrainReservation(rsp1).getPassengers().stream()
                                .anyMatch(p -> p.getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDED),
                TIMEOUT, "Ticket must be REFUNDED");
        TRefundPartInfo refundedTicketPartInfo = orderInfoRsp.getResult().getRefundPartsList().stream()
                .filter(x -> x.getState() == ERefundPartState.RPS_REFUNDED && x.getType() == ERefundPartType.RPT_SERVICE_PART)
                .findFirst().orElseThrow();
        assertThat(refundedTicketPartInfo.hasRefund()).isTrue();
        assertThat(refundedTicketPartInfo.getRefund().getRefundBlankToken().getOneOfDownloadBlankParamsCase())
                .isEqualByComparingTo(TDownloadBlankToken.OneOfDownloadBlankParamsCase.TRAINDOWNLOADBLANKPARAMS);
        assertThat(refundedTicketPartInfo.getRefund().getRefundAmount().getAmount()).isPositive();

        TRefundPartInfo refundFailedTicketPartInfo = orderInfoRsp.getResult().getRefundPartsList().stream()
                .filter(x -> x.getType() == ERefundPartType.RPT_SERVICE_PART &&
                        x.hasRefund() &&
                        x.getRefund().getState() == EOrderRefundState.RS_FAILED)
                .findFirst().orElseThrow();
        assertThat(refundFailedTicketPartInfo.getState()).isEqualTo(ERefundPartState.RPS_ENABLED);
        // not finalized refunding
        userHasConfirmedOrders(1);

        Assertions.assertThrows(StatusRuntimeException.class,
                () -> client.startRefund(TStartRefundReq.newBuilder().setOrderId(orderId)
                        .setRefundToken(calculateRefundRsp.getRefundToken()).build()),
                "Second call startRefund with the same parameters should throw an exception");
    }

    @Test
    public void testOrderWithInsuranceRefund() {
        var orderId = createSuccessfulOrder(createOrderRequest(1).build(), true);

        userHasConfirmedOrders(1);
        var orderInfoRsp = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId)
                .setUpdateOrderOnTheFly(true).build());
        var calculateRefundRsp = client.calculateRefundV2(TCalculateRefundReqV2.newBuilder().setOrderId(orderId)
                .addContext(RefundPartsService.partContextToString(TRefundPartContext.newBuilder()
                        .setType(ERefundPartType.RPT_ORDER)
                        .build()))
                .build());
        assertThat(calculateRefundRsp.getRefundAmount().getAmount()).isPositive();
        assertThat(calculateRefundRsp.getPenaltyAmount().getAmount()).isPositive();
        assertThat(calculateRefundRsp.getExpiresAt().getSeconds()).isGreaterThan(Instant.now().getEpochSecond());

        client.startRefund(TStartRefundReq.newBuilder().setOrderId(orderId)
                .setRefundToken(calculateRefundRsp.getRefundToken()).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_REFUNDED &&
                        getTrainReservation(rsp1).getPassengers().get(0).getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDED,
                TIMEOUT, "Ticket must be REFUNDED");
        userHasConfirmedOrders(0);
    }

    @Test
    public void testOrderExpired() {
        var orderId = createOrderWaitingForPayment(null, true, true);
        transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            order.getOrderItems().get(0).setExpiresAt(Instant.now().minusSeconds(1));
            return null;
        });
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CANCELLED, TIMEOUT,
                "Order must be in OS_CANCELLED state");
        assertThat(orderInfoRsp.getResult().getWorkflowState()).isEqualTo(EWorkflowState.WS_RUNNING);
        assertThat(orderInfoRsp.getResult().getCancellationReason()).isEqualTo(ECancellationReason.CR_EXPIRED);
    }

    @Test
    public void testCancelOnConfirmationFailed() {
        TCreateOrderReq.Builder createOrderReq = createOrderRequest(1);
        createOrderReq.getCreateServicesBuilder(0).getTrainTestContextBuilder()
                .setReservationConfirmOutcome(ETrainReservationConfirmOutcome.RCOO_FAILURE);
        var orderId = createOrderWaitingForPayment(createOrderReq.build(), true, true);

        assertThat(getInvoiceTotal(orderId)).isPositive();
        authorizePayment(orderId);

        //TODO (syukhno) correct test after fix refunded invoice state
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                // todo(mbobrov): another part of the WaitingRefundAfterCancellationStateHandler
                //  .ensureSingleInvoiceAndMoneyRefunded fix
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CANCELLED,
                TIMEOUT, "Order must be in OS_CANCELLED state");

        assertThat(orderInfoRsp.getResult().getCancellationReason()).isEqualTo(ECancellationReason.CR_CONFIRMATION_FAILED);
        assertThat(orderInfoRsp.getResult().getInvoiceCount()).isEqualTo(1);
        assertThat(orderInfoRsp.getResult().getInvoice(0).getTrustInvoiceState()).isEqualTo(ETrustInvoiceState.IS_CANCELLED);
        assertThat(getInvoiceTotal(orderId)).isZero();
    }

    @Test
    public void testCancelOrder() {
        var orderId = createOrderWaitingForPayment(null, true, true);
        var startCancellationReq = TStartCancellationReq.newBuilder().setOrderId(orderId).build();

        client.startCancellation(startCancellationReq);

        var getOrderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CANCELLED,
                TIMEOUT, "Order must be in OS_CANCELLED state");
        assertThat(getOrderInfoRsp.getResult().getCancellationReason()).isEqualTo(ECancellationReason.CR_USER_CANCELLED);

        authorizePayment(orderId);

        //TODO (mbobrov) Fix invoice state after refund
        //TODO (syukhno) correct test after fix refunded invoice state
//        getOrderInfoRsp = waitForPredicateOrTimeout(client, orderId,
//                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CANCELLED,
//                TIMEOUT, "Order must be in OS_CANCELLED state");
    }

    private String createOrderWaitingForPayment(TCreateOrderReq createOrderReq,
                                                boolean startPayment, boolean checkout) {
        if (createOrderReq == null) {
            createOrderReq = createOrderRequest(1).build();
        }
        TCreateOrderRsp resp = client.createOrder(createOrderReq);

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                TIMEOUT, "Order must be in OS_RESERVED state");
        getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getService(0).getServiceInfo().getGenericOrderItemState())
                .isEqualTo(EOrderItemState.IS_RESERVED);

        if (checkout) {
            client.checkout(TCheckoutReq.newBuilder().setOrderId(orderId).build());
        }

        if (startPayment) {
            client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                    .setSource("desktop")
                    .setOrderId(orderId)
                    .setReturnUrl("some_return_url").build()); // safely ignoring response, as we don't need payment url

            waitForPredicateOrTimeout(client, orderId,
                    rsp2 -> rsp2.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                    TIMEOUT, "Invoice must be in IS_WAIT_FOR_PAYMENT state");
        }

        return orderId;
    }

    private BigDecimal getInvoiceTotal(String orderId) {
        return transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            assertThat(order.getCurrentInvoice()).isNotNull();
            Invoice invoice = order.getCurrentInvoice();
            BigDecimal total = BigDecimal.ZERO;
            for (var i : invoice.getInvoiceItems()) {
                total = total.add(i.getPrice());
            }
            return total;
        });
    }

    private TrainReservation getTrainReservation(TGetOrderInfoRsp response) {
        return getTrainReservation(response.getResult());
    }

    private TrainReservation getTrainReservation(TOrderInfo response) {
        var payloadJson = response.getService(0).getServiceInfo().getPayload();
        return fromTJson(payloadJson, TrainReservation.class);
    }

    private TOrderServiceInfo getTrainReservationItem(TOrderInfo order) {
        return order.getServiceList().stream()
                .filter(s -> s.getServiceType() == EServiceType.PT_TRAIN)
                .findAny()
                .orElse(null);
    }

    private void addInsurance(TOrderInfo order) {
        client.addInsurance(TAddInsuranceReq.newBuilder().setOrderId(order.getOrderId()).build());
    }

    private void waitUntilInsuranceIsReserved(String orderId) {
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED
                        && getTrainReservation(rsp3).getInsuranceStatus() == InsuranceStatus.CHECKED_OUT,
                TIMEOUT, "Order must be in OS_RESERVED state and Insurance must be " +
                        "CHECKED_OUT");
    }

    private void authorizePayment(String orderId) {
        transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            assertThat(order.getCurrentInvoice()).isNotNull();
            Invoice invoice = order.getCurrentInvoice();
            String purchaseToken = invoice.getPurchaseToken();
            MockTrustClient mockTrustClient = (MockTrustClient) trustClient;
            mockTrustClient.paymentAuthorized(purchaseToken);
            return null;
        });
    }

    private void failPayment(String orderId) {
        transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            assertThat(order.getCurrentInvoice()).isNotNull();
            Invoice invoice = order.getCurrentInvoice();
            MockTrustClient mockTrustClient = (MockTrustClient) trustClient;
            mockTrustClient.paymentNotAuthorized(invoice.getPurchaseToken());
            return null;
        });
    }

    private void startRefund(String orderId) {
        transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            assertThat(order.getCurrentInvoice()).isNotNull();
            Invoice invoice = order.getCurrentInvoice();
            String purchaseToken = invoice.getPurchaseToken();
            MockTrustClient mockTrustClient = (MockTrustClient) trustClient;
            TrustCreateRefundRequest refundRequest = new TrustCreateRefundRequest();
            refundRequest.setPurchaseToken(purchaseToken);
            TrustCreateRefundResponse refund = mockTrustClient.createRefund(refundRequest, null);
            mockTrustClient.startRefund(refund.getTrustRefundId(), null);
            return null;
        });
    }

    @Test
    public void testOrderReservationCancelled() {
        TCreateOrderReq.Builder createOrderReq = createOrderRequest(2);
        createOrderReq.getCreateServicesBuilder(0).getTrainTestContextBuilder()
                .setReservationCreateOutcome(ETrainReservationCreateOutcome.RCRO_FAILURE);
        TCreateOrderRsp resp = client.createOrder(createOrderReq.build());

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build(); // we'll
        // reuse it
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        getOrderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_CANCELLED,
                TIMEOUT, "Order must be in OS_CANCELLED state");
        assertThat(getOrderInfoRsp.getResult().getCancellationReason()).isEqualTo(ECancellationReason.CR_RESERVATION_FAILED);
    }

    @Test
    public void testOrderItemCrashed() throws InterruptedException {
        CountDownLatch stCalled = new CountDownLatch(1);
        when(starTrekService.createIssueForGeneralOrderError(any(), anyLong(), any(), any())).then(
                invocation -> {
                    stCalled.countDown();
                    return UUID.randomUUID();
                }
        );

        var createOrderReq = createOrderRequest(1);
        createOrderReq.getCreateServicesBuilder(0).getTrainTestContextBuilder()
                .setReservationCreateOutcome(ETrainReservationCreateOutcome.RCRO_INVALID);
        TCreateOrderRsp resp = client.createOrder(createOrderReq.build());
        String orderId = resp.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());

        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getWorkflowState() == EWorkflowState.WS_CRASHED, TIMEOUT,
                "Order workflow must be in WS_CRASHED");

        stCalled.await(2, TimeUnit.SECONDS);
        verify(starTrekService).createIssueForGeneralOrderError(eq(UUID.fromString(orderId)), anyLong(), any(), any());
    }

    @Test
    public void testImValidationError() {
        var createOrderReq = createOrderRequest(1);
        createOrderReq.getOwnerBuilder().setEmail(MockImClient.INVALID_EMAIL);
        TCreateOrderRsp resp = client.createOrder(createOrderReq.build());

        var orderId = resp.getNewOrder().getOrderId();
        var getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        var getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                TIMEOUT, "Order must be in OS_RESERVED state");
        getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getService(0).getServiceInfo().getGenericOrderItemState())
                .isEqualTo(EOrderItemState.IS_RESERVED);

        assertThat(getTrainReservation(getOrderInfoRsp).getPassengers().get(0).isUseEmailForReservation()).isFalse();
    }

    @Test
    @Ignore("TODO TRAVELBACK-3391 flaky https://paste.yandex-team.ru/8446251")
    public void testServiceInfoRefresh() throws InterruptedException {
        TCreateOrderReq.Builder createOrderReq = createOrderRequest(1);
        createOrderReq.getCreateServicesBuilder(0).getTrainTestContextBuilder()
                .getOfficeActionsBuilder().setAcquire(TTrainOfficeAction.newBuilder()
                        .setDelayInSeconds(1).build());
        String orderId = createSuccessfulOrder(createOrderReq.build(), false);

        Thread.sleep(1000);
        TGetOrderInfoRsp rsp = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build());
        assertThat(getTrainReservation(rsp).getPassengers().get(0).getTicket().getImBlankStatus())
                .isEqualTo(ImBlankStatus.REMOTE_CHECK_IN);
        adminClient.updateTrainTickets(TUpdateTrainTicketsReq.newBuilder()
                .setOrderId(orderId)
                .build());
        waitForPredicateOrTimeout(client, orderId, rsp2 -> getTrainReservation(rsp2).getPassengers().get(0).getTicket()
                .getImBlankStatus() == ImBlankStatus.STRICT_BOARDING_PASS, TIMEOUT, "Tickets must be updated");
    }

    @Test
    public void testManualRefund() {
        var orderId = createSuccessfulOrder(createOrderRequest(1).build(), false);
        var fiscalItemId = transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            TrustInvoice invoice = (TrustInvoice) order.getInvoices().get(0);
            invoice.setState(ETrustInvoiceState.IS_CLEARED);
            return invoice.getInvoiceItems().get(0).getFiscalItemId();
        });
        userHasConfirmedOrders(1);

        //noinspection ConstantConditions
        adminClient.manualMoneyRefund(TManualMoneyRefundReq.newBuilder()
                .setOrderId(orderId)
                .setReason("Just because we can")
                .setByFiscalItemRefund(TByFiscalItemRefund.newBuilder()
                        .putTargetFiscalItems(fiscalItemId, TPrice.newBuilder()
                                .setAmount(826)
                                .setCurrency(ECurrency.C_RUB)
                                .build())).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getGenericOrderState() == EOrderState.OS_REFUNDING,
                TIMEOUT, "Order must be in OS_MANUAL_PROCESSING state");
        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_REFUNDING,
                TIMEOUT, "Invoice must be in IS_REFUNDING state");
        startRefund(orderId);
        var confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                rsp4 -> !rsp4.getResult().getUserActionScheduled(),
                TIMEOUT, "Order must have UserActionScheduled flag unset");
        assertThat(confirmedOrderResponse.getResult().getWorkflowState()).isEqualTo(EWorkflowState.WS_RUNNING);

        // refund is not finished, counter didn't yet counted the refunding action
        userHasConfirmedOrders(1);
    }

    @Test
    public void testChangeRegistrationStatus() {
        var orderId = createSuccessfulOrder(createOrderRequest(1).build(), false);
        TGetOrderInfoRsp orderInfoRsp = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build());
        int blankId = getTrainReservation(orderInfoRsp).getPassengers().get(0).getTicket().getBlankId();
        var request = TChangeTrainRegistrationStatusReq.newBuilder()
                .setOrderId(orderId)
                .setEnabled(false)
                .addBlankIds(blankId).build();

        waitForPredicateOrTimeout(client, orderId,
                rsp -> getTrainReservation(rsp).getPassengers().get(0).getTicket().getImBlankStatus()
                        == ImBlankStatus.REMOTE_CHECK_IN,
                TIMEOUT, "OrderItem must be in REMOTE_CHECK_IN state");

        client.changeTrainRegistrationStatus(request);

        var confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                rsp -> getTrainReservation(rsp).getPassengers().get(0).getTicket().getImBlankStatus()
                        == ImBlankStatus.NO_REMOTE_CHECK_IN && !rsp.getResult().getUserActionScheduled(),
                TIMEOUT, "OrderItem must be in NO_REMOTE_CHECK_IN state and not scheduled");

        assertThat(confirmedOrderResponse.getResult().getWorkflowState()).isEqualTo(EWorkflowState.WS_RUNNING);
    }

    @Test
    public void testChangeRegistrationStatusPending() {
        TCreateOrderReq.Builder createOrderReq = createOrderRequest(1);
        createOrderReq.getCreateServicesBuilder(0).getTrainTestContextBuilder()
                .setTrainChangeElectronicRegistrationOutcome(ETrainChangeElectronicRegistrationOutcome.CERO_FAILURE);
        var orderId = createSuccessfulOrder(createOrderReq.build(), false);
        TGetOrderInfoRsp orderInfoRsp = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).build());
        int blankId = getTrainReservation(orderInfoRsp).getPassengers().get(0).getTicket().getBlankId();
        TChangeTrainRegistrationStatusReq request = TChangeTrainRegistrationStatusReq.newBuilder()
                .setOrderId(orderId)
                .setEnabled(false)
                .addBlankIds(blankId).build();

        waitForPredicateOrTimeout(client, orderId,
                rsp -> getTrainReservation(rsp).getPassengers().get(0).getTicket().getImBlankStatus()
                        == ImBlankStatus.REMOTE_CHECK_IN,
                TIMEOUT, "OrderItem must be in REMOTE_CHECK_IN state");

        client.changeTrainRegistrationStatus(request);

        TGetOrderInfoRsp confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                rsp -> getTrainReservation(rsp).getPassengers().get(0).getTicket().isPendingElectronicRegistration()
                        && !rsp.getResult().getUserActionScheduled(),
                TIMEOUT, "OrderItem must be pending electronic registration and not scheduled");

        assertThat(confirmedOrderResponse.getResult().getWorkflowState()).isEqualTo(EWorkflowState.WS_RUNNING);
    }

    @Test(expected = StatusRuntimeException.class)
    public void testChangeRegistrationStatusBlankNotValid() {
        var orderId = createSuccessfulOrder(createOrderRequest(1).build(), false);

        waitForPredicateOrTimeout(client, orderId,
                rsp -> getTrainReservation(rsp).getPassengers().get(0).getTicket().getImBlankStatus()
                        == ImBlankStatus.REMOTE_CHECK_IN,
                TIMEOUT, "OrderItem must be in REMOTE_CHECK_IN state");

        var request = TChangeTrainRegistrationStatusReq.newBuilder()
                .setOrderId(orderId)
                .setEnabled(true)
                .addBlankIds(20000001).build();

        client.changeTrainRegistrationStatus(request);
    }

    private String createSuccessfulOrder(TCreateOrderReq createOrderReq, boolean withInsurance) {
        TCreateOrderRsp resp = client.createOrder(createOrderReq);
        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                TIMEOUT, "Order must be in OS_RESERVED state");

        if (withInsurance) {
            addInsurance(resp.getNewOrder());
            waitUntilInsuranceIsReserved(orderId);
        }

        client.checkout(TCheckoutReq.newBuilder().setOrderId(orderId).build());
        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_return_url").build()); // safely ignoring response, as we don't need payment url

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                TIMEOUT, "Invoice must be in IS_WAIT_FOR_PAYMENT state");
        authorizePayment(orderId);

        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CONFIRMED,
                TIMEOUT, "Order must be in OS_CONFIRMED state");

        return orderId;
    }

    @TestConfiguration
    static class IntegrationTestConfiguration {
        @Bean
        @Primary
        public TrustClient trainTrustClient() {
            return new MockTrustClient();
        }

        @Bean
        public TrustClientProvider trustClientProvider(@Autowired TrustClient trainTrustClient) {
            return paymentProfile -> trainTrustClient;
        }

        @Bean
        @Primary
        public TrainTariffInfoDataProvider trainTariffInfoDataProvider() {
            var tariffInfos = Arrays.asList(
                    TTrainTariffInfo.newBuilder().setId(1)
                            .setCode("full")
                            .setTitleRu("Полный")
                            .setImRequestCode("Full")
                            .setImResponseCodes("Full")
                            .setWithoutPlace(false)
                            .setMinAge(0)
                            .setMinAgeIncludesBirthday(false)
                            .setMaxAge(150)
                            .setMaxAgeIncludesBirthday(false)
                            .setNeedDocument(false).build(),

                    TTrainTariffInfo.newBuilder().setId(2)
                            .setCode("senior")
                            .setTitleRu("Сеньор")
                            .setImRequestCode("Senior")
                            .setImResponseCodes("Senior")
                            .setWithoutPlace(false)
                            .setMinAge(0)
                            .setMinAgeIncludesBirthday(false)
                            .setMaxAge(150)
                            .setMaxAgeIncludesBirthday(false)
                            .setNeedDocument(false).build()
            );

            TrainTariffInfoDataProviderProperties config = new TrainTariffInfoDataProviderProperties();
            config.setTablePath("tablePath");
            config.setIndexPath("./train-tariff-index");
            config.setProxy(new ArrayList<>());

            TestLuceneIndexBuilder<TTrainTariffInfo> luceneIndexBuilder = new TestLuceneIndexBuilder<TTrainTariffInfo>()
                    .setLuceneData(tariffInfos);

            return new TrainTariffInfoService(config, luceneIndexBuilder);
        }
    }
}
