package ru.yandex.travel.orders.integration.train;

import java.time.Duration;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;

import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.TestConfiguration;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;
import org.springframework.test.context.TestPropertySource;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TJson;
import ru.yandex.travel.dicts.rasp.proto.TTrainTariffInfo;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.commons.proto.ETrainChangeElectronicRegistrationOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainInsuranceCheckoutConfirmOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainInsuranceCheckoutOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainInsurancePricingOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainRefundCheckoutOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainRefundPricingOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainReservationConfirmOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainReservationCreateOutcome;
import ru.yandex.travel.orders.commons.proto.TTrainTestContext;
import ru.yandex.travel.orders.configurations.TrainTariffInfoDataProviderProperties;
import ru.yandex.travel.orders.entities.FiscalItemType;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderRefund;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.proto.EOrderRefundState;
import ru.yandex.travel.orders.proto.ERefundPartState;
import ru.yandex.travel.orders.proto.ERefundPartType;
import ru.yandex.travel.orders.proto.TAddInsuranceReq;
import ru.yandex.travel.orders.proto.TCalculateRefundReqV2;
import ru.yandex.travel.orders.proto.TCheckoutReq;
import ru.yandex.travel.orders.proto.TCreateOrderReq;
import ru.yandex.travel.orders.proto.TCreateOrderRsp;
import ru.yandex.travel.orders.proto.TCreateServiceReq;
import ru.yandex.travel.orders.proto.TDownloadBlankToken;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.proto.TOrderInfo;
import ru.yandex.travel.orders.proto.TRefundPartContext;
import ru.yandex.travel.orders.proto.TRefundPartInfo;
import ru.yandex.travel.orders.proto.TReserveReq;
import ru.yandex.travel.orders.proto.TStartPaymentReq;
import ru.yandex.travel.orders.proto.TStartRefundReq;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.services.mock.MockTrustClient;
import ru.yandex.travel.orders.services.orders.RefundPartsService;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.train.tariffinfo.TrainTariffInfoDataProvider;
import ru.yandex.travel.orders.services.train.tariffinfo.TrainTariffInfoService;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.train.proto.ETrainOrderState;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.train.model.TrainTicketRefundStatus;
import ru.yandex.travel.train.partners.im.model.ImBlankStatus;
import ru.yandex.travel.yt_lucene_index.TestLuceneIndexBuilder;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyBoolean;
import static org.mockito.Mockito.when;
import static ru.yandex.travel.commons.proto.ProtoUtils.fromTJson;
import static ru.yandex.travel.orders.integration.IntegrationUtils.waitForPredicateOrTimeout;

@SuppressWarnings("ResultOfMethodCallIgnored")
@TestPropertySource(properties = {
        "quartz.enabled=true",
        "workflow-processing.pending-workflow-polling-interval=100ms",
        "trust-hotels.clearing-refresh-timeout=1s",
        "trust-hotels.payment-refresh-timeout=1s",
        "single-node.auto-start=true",
        "train-workflow.check-ticket-refund-delay=1s",
        "train-workflow.check-ticket-refund-task-period=200ms",
        "train-workflow.check-ticket-refund-max-tries=50",
        "train-workflow.check-passenger-discounts-enabled=true",
        "train-workflow.insurance-enabled=true",
        "mock-im-client.enabled=true",
})
public class MockImTrainOrderFlowTests extends AbstractTrainOrderFlowTest {

    @Autowired
    private TransactionTemplate transactionTemplate;

    @Autowired
    private OrderRepository orderRepository;

    @Autowired
    private TrustClient trustClient;

    @Test
    public void testOrderReserved() {
        TCreateOrderReq.Builder createOrderReq = createOrderRequest(1);
        TCreateOrderRsp resp = client.createOrder(createOrderReq.build());

        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        var orderInfoReq = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        var orderInfoResp = client.getOrderInfo(orderInfoReq);
        assertThat(orderInfoResp.getResult().getTrainOrderState()).isEqualTo(ETrainOrderState.OS_NEW);

        var reserveReq = TReserveReq.newBuilder().setOrderId(orderId).build();
        client.reserve(reserveReq);
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT,
                Duration.ofSeconds(10), "Order must be in OS_WAITING_PAYMENT state");
        orderInfoResp = client.getOrderInfo(orderInfoReq);
        assertThat(orderInfoResp.getResult().getService(0).getServiceInfo().getGenericOrderItemState())
                .isEqualTo(EOrderItemState.IS_RESERVED);

        var ticket = ProtoUtils.fromTJson(
                orderInfoResp.getResult().getService(0).getServiceInfo().getPayload(),
                TrainReservation.class).getPassengers().get(0).getTicket();
        assertThat(ticket.getImBlankStatus()).isNull();
        assertThat(ticket.isPendingElectronicRegistration()).isEqualTo(false);

        client.addInsurance(TAddInsuranceReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT
                        && getTrainReservation(rsp3).getInsuranceStatus() == InsuranceStatus.CHECKED_OUT,
                Duration.ofSeconds(10), "Order must be in OS_WAITING_PAYMENT state and InsuranceStatus must be " +
                        "CHECKED_OUT");

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_success_payment_url").build()); // safely ignoring response, as we don't need it

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getCurrentInvoice().getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                Duration.ofSeconds(10), "Invoice must be in IS_WAIT_FOR_PAYMENT state");

        authorizePayment(orderId);

        var confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getTrainOrderState() == ETrainOrderState.OS_CONFIRMED,
                Duration.ofSeconds(10), "Order must be in OS_CONFIRMED state");
        assertThat(confirmedOrderResponse.getResult().getServiceCount()).isEqualTo(1);
        assertThat(confirmedOrderResponse.getResult().getInvoiceCount()).isEqualTo(1);
        assertThat(getTrainReservation(confirmedOrderResponse).getReservationNumber()).isNotNull();
        assert (hasInsuranceFiscalItem(confirmedOrderResponse));
    }

    @Test
    public void testGenericOrderReserved() {
        TCreateOrderReq.Builder createOrderReq = createOrderRequest(1);
        createOrderReq = createOrderReq.setOrderType(EOrderType.OT_GENERIC);
        TCreateOrderRsp resp = client.createOrder(createOrderReq.build());

        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        var orderInfoReq = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        var orderInfoResp = client.getOrderInfo(orderInfoReq);
        assertThat(orderInfoResp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        var reserveReq = TReserveReq.newBuilder().setOrderId(orderId).build();
        client.reserve(reserveReq);
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                Duration.ofSeconds(10), "Order must be in OS_RESERVED state");
        orderInfoResp = client.getOrderInfo(orderInfoReq);
        assertThat(orderInfoResp.getResult().getService(0).getServiceInfo().getGenericOrderItemState())
                .isEqualTo(EOrderItemState.IS_RESERVED);

        var ticket = ProtoUtils.fromTJson(
                orderInfoResp.getResult().getService(0).getServiceInfo().getPayload(),
                TrainReservation.class).getPassengers().get(0).getTicket();
        assertThat(ticket.getImBlankStatus()).isNull();
        assertThat(ticket.isPendingElectronicRegistration()).isEqualTo(false);

        addInsurance(resp.getNewOrder());
        waitUntilInsuranceIsReserved(orderId);

        client.checkout(TCheckoutReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_WAITING_PAYMENT,
                Duration.ofSeconds(10), "Order must be in OS_WAITING_PAYMENT state");

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_success_payment_url").build()); // safely ignoring response, as we don't need it

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getCurrentInvoice().getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                Duration.ofSeconds(10), "Invoice must be in IS_WAIT_FOR_PAYMENT state");

        authorizePayment(orderId);

        TGetOrderInfoRsp confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CONFIRMED,
                Duration.ofSeconds(10), "Order must be in OS_CONFIRMED state");
        assertThat(confirmedOrderResponse.getResult().getServiceCount()).isEqualTo(1);
        assertThat(confirmedOrderResponse.getResult().getInvoiceCount()).isEqualTo(1);
        assert (hasInsuranceFiscalItem(confirmedOrderResponse));
    }

    @Test
    public void testGenericOrderOfficeRefund() {
        var createOrderReqBuilder = createOrderRequest(1).setOrderType(EOrderType.OT_GENERIC);
        var actionsBuilder = createOrderReqBuilder.getCreateServicesBuilder(0)
                .getTrainTestContextBuilder()
                .getOfficeActionsBuilder();
        actionsBuilder.addRefundsBuilder().setDelayInSeconds(2).setAmount(2000);
        actionsBuilder.addRefundsBuilder().setDelayInSeconds(10).setAmount(1000);
        TCreateOrderRsp resp = client.createOrder(createOrderReqBuilder.build());

        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        var orderInfoReq = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        var orderInfoResp = client.getOrderInfo(orderInfoReq);
        assertThat(orderInfoResp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        var reserveReq = TReserveReq.newBuilder().setOrderId(orderId).build();
        client.reserve(reserveReq);
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                Duration.ofSeconds(10), "Order must be in OS_RESERVED state");
        orderInfoResp = client.getOrderInfo(orderInfoReq);
        assertThat(orderInfoResp.getResult().getService(0).getServiceInfo().getGenericOrderItemState())
                .isEqualTo(EOrderItemState.IS_RESERVED);

        var ticket = ProtoUtils.fromTJson(
                orderInfoResp.getResult().getService(0).getServiceInfo().getPayload(),
                TrainReservation.class).getPassengers().get(0).getTicket();
        assertThat(ticket.getImBlankStatus()).isNull();
        assertThat(ticket.isPendingElectronicRegistration()).isEqualTo(false);

        client.checkout(TCheckoutReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_WAITING_PAYMENT,
                Duration.ofSeconds(10), "Order must be in OS_WAITING_PAYMENT state");

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_success_payment_url").build()); // safely ignoring response, as we don't need it

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getCurrentInvoice().getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                Duration.ofSeconds(10), "Invoice must be in IS_WAIT_FOR_PAYMENT state");

        authorizePayment(orderId);

        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CONFIRMED,
                Duration.ofSeconds(10), "Order must be in OS_CONFIRMED state");

        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_REFUNDED &&
                        getTrainReservation(rsp1).getPassengers().get(0).getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDED,
                Duration.ofSeconds(10), "Ticket must be REFUNDED");

        waitForPredicateOrTimeout(
                () -> {
                    var refundStates = getOrderRefundStates(orderId);
                    return refundStates.size() == 2 && refundStates.stream().allMatch(x -> x.equals(EOrderRefundState.RS_REFUNDED));
                },
                Duration.ofSeconds(10), "Two refunds must be processed"
        );

        var orderInfoRsp = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId)
                .setUpdateOrderOnTheFly(true).build());
        assertThat(getTrainReservation(orderInfoRsp).getPassengers().get(0).getTicket().getImBlankStatus()).isEqualTo(ImBlankStatus.REFUNDED);
    }

    @Test
    public void testOrderRefund() {
        TCreateOrderReq.Builder createOrderReq = createOrderRequest(1);
        createOrderReq = createOrderReq.setOrderType(EOrderType.OT_GENERIC);
        TCreateOrderRsp resp = client.createOrder(createOrderReq.build());

        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        var orderInfoReq = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        var orderInfoResp = client.getOrderInfo(orderInfoReq);
        assertThat(orderInfoResp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        var reserveReq = TReserveReq.newBuilder().setOrderId(orderId).build();
        client.reserve(reserveReq);
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                Duration.ofSeconds(10), "Order must be in OS_RESERVED state");
        orderInfoResp = client.getOrderInfo(orderInfoReq);
        assertThat(orderInfoResp.getResult().getService(0).getServiceInfo().getGenericOrderItemState())
                .isEqualTo(EOrderItemState.IS_RESERVED);

        var ticket = ProtoUtils.fromTJson(
                orderInfoResp.getResult().getService(0).getServiceInfo().getPayload(),
                TrainReservation.class).getPassengers().get(0).getTicket();
        assertThat(ticket.getImBlankStatus()).isNull();
        assertThat(ticket.isPendingElectronicRegistration()).isEqualTo(false);

        client.checkout(TCheckoutReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_WAITING_PAYMENT,
                Duration.ofSeconds(10), "Order must be in OS_WAITING_PAYMENT state");

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_success_payment_url").build()); // safely ignoring response, as we don't need it

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getCurrentInvoice().getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                Duration.ofSeconds(10), "Invoice must be in IS_WAIT_FOR_PAYMENT state");

        authorizePayment(orderId);

        TGetOrderInfoRsp confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CONFIRMED,
                Duration.ofSeconds(10), "Order must be in OS_CONFIRMED state");

        var orderInfoRsp = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId)
                .setUpdateOrderOnTheFly(true).build());
        var calculateRefundRsp = client.calculateRefundV2(TCalculateRefundReqV2.newBuilder().setOrderId(orderId)
                .addContext(RefundPartsService.partContextToString(TRefundPartContext.newBuilder()
                        .setServiceId(orderInfoRsp.getResult().getService(0).getServiceId())
                        .setType(ERefundPartType.RPT_SERVICE)
                        .build()))
                .build());
        assertThat(calculateRefundRsp.getRefundAmount().getAmount()).isPositive();
        assertThat(calculateRefundRsp.getPenaltyAmount().getAmount()).isPositive();
        assertThat(calculateRefundRsp.getExpiresAt().getSeconds()).isGreaterThan(Instant.now().getEpochSecond());

        client.startRefund(TStartRefundReq.newBuilder().setOrderId(orderId)
                .setRefundToken(calculateRefundRsp.getRefundToken()).build());

        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_REFUNDED &&
                        getTrainReservation(rsp1).getPassengers().get(0).getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDED,
                Duration.ofSeconds(10), "Ticket must be REFUNDED");

        orderInfoRsp = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId)
                .setUpdateOrderOnTheFly(true).build());
        assertThat(getTrainReservation(orderInfoRsp).getPassengers().get(0).getTicket().getImBlankStatus()).isEqualTo(ImBlankStatus.REFUNDED);
        TRefundPartInfo refundedTicketPartInfo = orderInfoRsp.getResult().getRefundPartsList().stream()
                .filter(x -> x.getState() == ERefundPartState.RPS_REFUNDED && x.getType() == ERefundPartType.RPT_SERVICE_PART)
                .findFirst().orElseThrow();
        assertThat(refundedTicketPartInfo.hasRefund()).isTrue();
        assertThat(refundedTicketPartInfo.getRefund().getRefundBlankToken().getOneOfDownloadBlankParamsCase())
                .isEqualByComparingTo(TDownloadBlankToken.OneOfDownloadBlankParamsCase.TRAINDOWNLOADBLANKPARAMS);
        assertThat(refundedTicketPartInfo.getRefund().getRefundAmount().getAmount()).isPositive();
    }

    private void addInsurance(TOrderInfo order) {
        client.addInsurance(TAddInsuranceReq.newBuilder().setOrderId(order.getOrderId()).build());
    }

    private void waitUntilInsuranceIsReserved(String orderId) {
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED
                        && getTrainReservation(rsp3).getInsuranceStatus() == InsuranceStatus.CHECKED_OUT,
                Duration.ofSeconds(10), "Order must be in OS_RESERVED state and Insurance must be " +
                        "CHECKED_OUT");
    }

    private boolean hasInsuranceFiscalItem(TGetOrderInfoRsp orderInfoRsp) {
        var paymentBreakdown = orderInfoRsp.getResult().getService(0).getServiceInfo().getPaymentBreakdown();
        var fiscalItemsCount = paymentBreakdown.getFiscalItemsCount();
        for (int i = 0; i < fiscalItemsCount; ++i) {
            if (paymentBreakdown.getFiscalItems(i).getType().equals(FiscalItemType.TRAIN_INSURANCE.getValue())) {
                return true;
            }
        }
        return false;
    }

    private void authorizePayment(String orderId) {
        transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            assertThat(order.getCurrentInvoice()).isNotNull();
            Invoice invoice = order.getCurrentInvoice();
            String purchaseToken = invoice.getPurchaseToken();
            MockTrustClient mockTrustClient = (MockTrustClient) trustClient;
            mockTrustClient.paymentAuthorized(purchaseToken);
            return null;
        });
    }

    private List<EOrderRefundState> getOrderRefundStates(String orderId) {
        return transactionTemplate.execute(i -> {
            var order = orderRepository.getOne(UUID.fromString(orderId));
            var refunds = order.getOrderRefunds();
            return refunds.stream().map(OrderRefund::getState).collect(Collectors.toList());
        });
    }

    @Override
    protected EOrderType orderTypeForOrderReq() {
        return EOrderType.OT_TRAIN;
    }

    @Override
    protected TCreateServiceReq.Builder getServiceForPayload(String payloadJson) {
        return TCreateServiceReq.newBuilder()
                .setServiceType(EServiceType.PT_TRAIN)
                .setTrainTestContext(
                        TTrainTestContext.newBuilder()
                                .setInsurancePricingOutcome(ETrainInsurancePricingOutcome.IPO_SUCCESS)
                                .setInsuranceCheckoutOutcome(ETrainInsuranceCheckoutOutcome.ICO_SUCCESS)
                                .setInsuranceCheckoutConfirmOutcome(ETrainInsuranceCheckoutConfirmOutcome.ICCO_SUCCESS)
                                .setRefundPricingOutcome(ETrainRefundPricingOutcome.RPO_SUCCESS)
                                .setRefundCheckoutOutcome(ETrainRefundCheckoutOutcome.RCO_SUCCESS)
                                .setReservationCreateOutcome(ETrainReservationCreateOutcome.RCRO_SUCCESS)
                                .setReservationConfirmOutcome(ETrainReservationConfirmOutcome.RCOO_SUCCESS)
                                .setTrainChangeElectronicRegistrationOutcome(ETrainChangeElectronicRegistrationOutcome.CERO_SUCCESS)
                                .build())
                .setSourcePayload(TJson.newBuilder().setValue(payloadJson).build());
    }

    private TrainReservation getTrainReservation(TGetOrderInfoRsp response) {
        var payloadJson = response.getResult().getService(0).getServiceInfo().getPayload();
        return fromTJson(payloadJson, TrainReservation.class);
    }

    @TestConfiguration
    static class IntegrationTestConfiguration {
        @Bean
        @Primary
        public TrustClient trainTrustClient() {
            return new MockTrustClient();
        }

        @Bean
        public TrustClientProvider trustClientProvider(@Autowired TrustClient trainTrustClient) {
            return paymentProfile -> trainTrustClient;
        }

        @Bean
        @Primary
        public TrainTariffInfoDataProvider trainTariffInfoDataProvider() {
            var tariffInfos = Arrays.asList(
                    TTrainTariffInfo.newBuilder().setId(1)
                            .setCode("full")
                            .setTitleRu("Полный")
                            .setImRequestCode("Full")
                            .setImResponseCodes("Full")
                            .setWithoutPlace(false)
                            .setMinAge(0)
                            .setMinAgeIncludesBirthday(false)
                            .setMaxAge(150)
                            .setMaxAgeIncludesBirthday(false)
                            .setNeedDocument(false).build(),

                    TTrainTariffInfo.newBuilder().setId(2)
                            .setCode("senior")
                            .setTitleRu("Сеньор")
                            .setImRequestCode("Senior")
                            .setImResponseCodes("Senior")
                            .setWithoutPlace(false)
                            .setMinAge(0)
                            .setMinAgeIncludesBirthday(false)
                            .setMaxAge(150)
                            .setMaxAgeIncludesBirthday(false)
                            .setNeedDocument(false).build()
            );

            TrainTariffInfoDataProviderProperties config = new TrainTariffInfoDataProviderProperties();
            config.setTablePath("tablePath");
            config.setIndexPath("./train-tariff-index");
            config.setProxy(new ArrayList<>());

            TestLuceneIndexBuilder<TTrainTariffInfo> luceneIndexBuilder = new TestLuceneIndexBuilder<TTrainTariffInfo>()
                    .setLuceneData(tariffInfos);

            return new TrainTariffInfoService(config, luceneIndexBuilder);
        }
    }
}
