package ru.yandex.travel.orders.integration.train;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicReference;

import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.TestConfiguration;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TJson;
import ru.yandex.travel.dicts.rasp.proto.TTrainTariffInfo;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.configurations.TrainTariffInfoDataProviderProperties;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.integration.train.factories.ImOrderInfoResponseFactory;
import ru.yandex.travel.orders.integration.train.factories.ImReservationCreateResponseFactory;
import ru.yandex.travel.orders.integration.train.factories.TrainOrderItemFactory;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.proto.EOrderRefundState;
import ru.yandex.travel.orders.proto.EOrderRefundType;
import ru.yandex.travel.orders.proto.ERefundPartType;
import ru.yandex.travel.orders.proto.TAddInsuranceReq;
import ru.yandex.travel.orders.proto.TCalculateRefundReqV2;
import ru.yandex.travel.orders.proto.TChangeTrainRegistrationStatusReq;
import ru.yandex.travel.orders.proto.TCheckoutReq;
import ru.yandex.travel.orders.proto.TCreateOrderReq;
import ru.yandex.travel.orders.proto.TCreateOrderRsp;
import ru.yandex.travel.orders.proto.TCreateServiceReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.proto.TReserveReq;
import ru.yandex.travel.orders.proto.TStartPaymentReq;
import ru.yandex.travel.orders.proto.TStartRefundReq;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.services.mock.MockTrustClient;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.train.tariffinfo.TrainTariffInfoDataProvider;
import ru.yandex.travel.orders.services.train.tariffinfo.TrainTariffInfoService;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.train.model.DocumentType;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.train.model.PassengerCategory;
import ru.yandex.travel.train.model.TariffType;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.train.model.TrainTicketRefundStatus;
import ru.yandex.travel.train.partners.im.ImClient;
import ru.yandex.travel.train.partners.im.model.AutoReturnRequest;
import ru.yandex.travel.train.partners.im.model.AutoReturnResponse;
import ru.yandex.travel.train.partners.im.model.ElectronicRegistrationResponse;
import ru.yandex.travel.train.partners.im.model.ImBlankStatus;
import ru.yandex.travel.train.partners.im.model.RailwayAutoReturnResponse;
import ru.yandex.travel.train.partners.im.model.RailwayReturnAmountResponse;
import ru.yandex.travel.train.partners.im.model.RailwayReturnBlankResponse;
import ru.yandex.travel.train.partners.im.model.ReservationCreateResponse;
import ru.yandex.travel.train.partners.im.model.ReturnAmountResponse;
import ru.yandex.travel.train.partners.im.model.insurance.InsuranceCheckoutResponse;
import ru.yandex.travel.train.partners.im.model.insurance.InsurancePricingResponse;
import ru.yandex.travel.train.partners.im.model.insurance.RailwayTravelPricingResult;
import ru.yandex.travel.train.partners.im.model.insurance.RailwayTravelProductPricingInfo;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationStatus;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderInfoResponse;
import ru.yandex.travel.train.service.TrainPartKeyService;
import ru.yandex.travel.yt_lucene_index.TestLuceneIndexBuilder;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyBoolean;
import static org.mockito.ArgumentMatchers.anyInt;
import static org.mockito.Mockito.atLeast;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static ru.yandex.travel.commons.proto.ProtoUtils.fromTJson;
import static ru.yandex.travel.orders.integration.IntegrationUtils.waitForPredicateOrTimeout;

@SuppressWarnings("ResultOfMethodCallIgnored")
public class TrainOrderFlowBabyTests extends AbstractTrainOrderFlowTest {

    @MockBean
    private ImClient imClient;

    @Autowired
    private TransactionTemplate transactionTemplate;

    @Autowired
    private OrderRepository orderRepository;

    @Autowired
    private TrustClient trustClient;

    @Test
    public void testOrderReserved() {
        ReservationCreateResponse reservationResponse = createReservationCreateResponse();
        when(imClient.reservationCreate(any(), any())).thenReturn(reservationResponse);
        var imOrderInfoResponse = createImOrderInfoResponse(ImOperationStatus.OK, ImOperationStatus.OK);
        when(imClient.orderInfo(anyInt())).thenReturn(imOrderInfoResponse);
        when(imClient.orderInfo(anyInt(), any())).thenReturn(imOrderInfoResponse);
        when(imClient.insurancePricing(any())).thenReturn(createInsurancePricingResponse());
        when(imClient.insuranceCheckout(any())).thenReturn(createInsuranceCheckoutResponse());
        TCreateOrderReq createOrderReq = createOrderRequest();
        TCreateOrderRsp resp = client.createOrder(createOrderReq);
        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                Duration.ofSeconds(10), "Order must be in OS_RESERVED state");
        getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getService(0).getServiceInfo().getGenericOrderItemState())
                .isEqualTo(EOrderItemState.IS_RESERVED);
        var ticket = ProtoUtils.fromTJson(
                getOrderInfoRsp.getResult().getService(0).getServiceInfo().getPayload(),
                TrainReservation.class).getPassengers().get(0).getTicket();
        assertThat(ticket.getImBlankStatus()).isNull();
        assertThat(ticket.isPendingElectronicRegistration()).isEqualTo(false);

        verify(imClient).insurancePricing(any());
        verify(imClient).reservationCreate(any(), any());

        client.addInsurance(TAddInsuranceReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED
                        && getTrainReservation(rsp3).getInsuranceStatus() == InsuranceStatus.CHECKED_OUT,
                Duration.ofSeconds(10), "Order must be in OS_RESERVED state");

        client.checkout(TCheckoutReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_WAITING_PAYMENT
                        && getTrainReservation(rsp3).getInsuranceStatus() == InsuranceStatus.CHECKED_OUT,
                Duration.ofSeconds(10), "Order must be in OS_WAITING_PAYMENT state and InsuranceStatus must be " +
                        "CHECKED_OUT");

        verify(imClient, times(3)).insuranceCheckout(any());

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_success_payment_url").build()); // safely ignoring response, as we don't need

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getCurrentInvoice().getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                Duration.ofSeconds(10), "Invoice must be in IS_WAIT_FOR_PAYMENT state");

        authorizePayment(orderId);

        TGetOrderInfoRsp confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CONFIRMED,
                Duration.ofSeconds(10), "Order must be in OS_CONFIRMED state");
        assertThat(confirmedOrderResponse.getResult().getServiceCount()).isEqualTo(1);
        assertThat(confirmedOrderResponse.getResult().getInvoiceCount()).isEqualTo(1);
    }

    private ReservationCreateResponse createReservationCreateResponse() {
        var factory = new TrainOrderItemFactory();
        TrainOrderItem orderItem = factory.createTrainOrderItem();
        TrainReservation payload = orderItem.getPayload();
        payload.getPassengers().add(factory.createTrainPassenger());
        factory.setCategory(PassengerCategory.BABY);
        factory.setTariffCode("baby");
        factory.setTariffType(TariffType.BABY);
        factory.setDocumentType(DocumentType.BIRTH_CERTIFICATE);
        payload.getPassengers().add(factory.createTrainPassenger());
        var reservationResponseFactory = ImReservationCreateResponseFactory.createForOrder(orderItem, null);
        return reservationResponseFactory.createReservationCreateResponse();
    }

    @Test
    public void testInsuranceAutoReturn() {
        ReservationCreateResponse reservationResponse = createReservationCreateResponse();
        when(imClient.reservationCreate(any(), any())).thenReturn(reservationResponse);
        var imOrderInfoResponse = createImOrderInfoResponse(ImOperationStatus.OK, ImOperationStatus.FAILED);
        when(imClient.orderInfo(anyInt())).thenReturn(imOrderInfoResponse);
        when(imClient.insurancePricing(any())).thenReturn(createInsurancePricingResponse());
        when(imClient.insuranceCheckout(any())).thenReturn(createInsuranceCheckoutResponse());
        TCreateOrderReq createOrderReq = createOrderRequest();
        TCreateOrderRsp resp = client.createOrder(createOrderReq);
        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                Duration.ofSeconds(10), "Order must be in OS_RESERVED state");

        client.addInsurance(TAddInsuranceReq.newBuilder().setOrderId(orderId).build());

        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED
                        && getTrainReservation(rsp3).getInsuranceStatus() == InsuranceStatus.CHECKED_OUT,
                Duration.ofSeconds(10), "Order must be in OS_RESERVED state");
        client.checkout(TCheckoutReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_WAITING_PAYMENT
                        && getTrainReservation(rsp3).getInsuranceStatus() == InsuranceStatus.CHECKED_OUT,
                Duration.ofSeconds(10), "Order must be in OS_WAITING_PAYMENT state and InsuranceStatus must be " +
                        "CHECKED_OUT");

        verify(imClient, atLeast(1)).insuranceCheckout(any());

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_return_url").build()); // safely ignoring response, as we don't need payment url

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                Duration.ofSeconds(10), "Invoice must be in IS_WAIT_FOR_PAYMENT state");
        authorizePayment(orderId);

        TGetOrderInfoRsp confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CONFIRMED
                        && getTrainReservation(rsp1).getInsuranceStatus() == InsuranceStatus.AUTO_RETURN,
                Duration.ofSeconds(10), "Order must be in OS_CONFIRMED state and insurance in AUTO_RETURN");
        assertThat(confirmedOrderResponse.getResult().getServiceCount()).isEqualTo(1);

        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getRefundCount() == 1 &&
                        rsp1.getResult().getRefund(0).getRefundType() == EOrderRefundType.RT_TRAIN_INSURANCE_AUTO_RETURN &&
                        rsp1.getResult().getRefund(0).getState() == EOrderRefundState.RS_REFUNDED,
                Duration.ofSeconds(5), "Order insurance refund must be in RS_REFUNDED state");
    }

    @Test
    public void testOrderRefund() {
        AtomicReference<AutoReturnRequest> autoReturnRequestRef = new AtomicReference<>();
        AtomicReference<OrderInfoResponse> orderInfoResponseRef = new AtomicReference<>();
        var nonRefundedOrderInfoResponse = createImOrderInfoResponse(ImOperationStatus.OK, ImOperationStatus.OK);
        orderInfoResponseRef.set(nonRefundedOrderInfoResponse);
        when(imClient.orderInfo(anyInt())).thenAnswer(invocation -> orderInfoResponseRef.get());
        when(imClient.orderInfo(anyInt(), any())).thenAnswer(invocation -> orderInfoResponseRef.get());
        when(imClient.getReturnAmount(any())).thenReturn(createReturnAmountResponse());
        var autoReturnRsp = createAutoReturnResponse();
        when(imClient.autoReturn(any())).thenAnswer(invocation -> {
            autoReturnRequestRef.set(invocation.getArgument(0));
            return autoReturnRsp;
        });
        var orderId = createSuccessfulOrder();
        var orderInfoRsp = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId)
                .setUpdateOrderOnTheFly(true)
                .build());

        var partKey = TrainPartKeyService.getPassengerKey(
                UUID.fromString(orderInfoRsp.getResult().getService(0).getServiceId()),
                getTrainReservation(orderInfoRsp).getPassengers().get(1).getCustomerId());
        var refundPart = orderInfoRsp.getResult().getRefundPartsList().stream()
                .filter(x -> x.getKey().equals(partKey)).findFirst().get();
        var calculateRefundRsp = client.calculateRefundV2(TCalculateRefundReqV2.newBuilder()
                .setOrderId(orderId)
                .addContext(refundPart.getContext())
                .build());

        client.startRefund(TStartRefundReq.newBuilder().setOrderId(orderId)
                .setRefundToken(calculateRefundRsp.getRefundToken()).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> getTrainReservation(rsp1).getPassengers().get(1).getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDING &&
                        autoReturnRequestRef.get() != null,
                Duration.ofSeconds(10), "Ticket must be REFUNDING");

        orderInfoResponseRef.set(createImOrderInfoResponseWithRefund(
                autoReturnRequestRef.get().getServiceAutoReturnRequest().getAgentReferenceId(), false));

        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CONFIRMED &&
                        getTrainReservation(rsp1).getPassengers().get(1).getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDED &&
                        getTrainReservation(rsp1).getPassengers().get(2).getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDED,
                Duration.ofSeconds(10), "Ticket must be REFUNDED");
    }

    @Test
    public void testRefundBabyBlank() {
        var childBlankId = 99999999;
        AtomicReference<AutoReturnRequest> autoReturnRequestRef = new AtomicReference<>();
        AtomicReference<OrderInfoResponse> imOrderInfoResponseRef = new AtomicReference<>();
        var factory = new ImOrderInfoResponseFactory();
        factory.setPassengers(2);
        factory.setBuyTicketStatus(ImOperationStatus.OK);
        factory.setBuyInsuranceStatus(ImOperationStatus.OK);
        var imOrderInfoResponse = factory.create();
        var imChildOrderInfoBlank = factory.createBlank(childBlankId);
        imChildOrderInfoBlank.setAmount(BigDecimal.ZERO);
        imOrderInfoResponse.getOrderItems().get(0).getOrderItemBlanks().add(imChildOrderInfoBlank);
        imOrderInfoResponseRef.set(imOrderInfoResponse);

        ReservationCreateResponse imReservationResponse = createReservationCreateResponse();
        imReservationResponse.getReservationResults().get(0).getPassengers().get(2).setOrderItemBlankId(childBlankId);
        var imChildReserveBlank = ImReservationCreateResponseFactory.create()
                .createReservationBlankResponse(childBlankId, "FreeChild");
        imChildReserveBlank.setAmount(BigDecimal.ZERO);
        imChildReserveBlank.setBaseFare(BigDecimal.ZERO);
        imChildReserveBlank.setAdditionalPrice(BigDecimal.ZERO);
        imChildReserveBlank.setServicePrice(BigDecimal.ZERO);
        imChildReserveBlank.setVatRateValues(new ArrayList<>());
        imReservationResponse.getReservationResults().get(0).getBlanks().add(imChildReserveBlank);
        when(imClient.reservationCreate(any(), any())).thenReturn(imReservationResponse);

        when(imClient.orderInfo(anyInt())).thenAnswer(invocation -> imOrderInfoResponseRef.get());
        when(imClient.orderInfo(anyInt(), any())).thenAnswer(invocation -> imOrderInfoResponseRef.get());
        ReturnAmountResponse imReturnAmountRsp = createReturnAmountResponse();
        imReturnAmountRsp.getServiceReturnResponse().getBlanks().get(0).setPurchaseOrderItemBlankId(childBlankId);
        imReturnAmountRsp.getServiceReturnResponse().getBlanks().get(0).setAmount(BigDecimal.ZERO);
        imReturnAmountRsp.getServiceReturnResponse().getBlanks().get(0).setServicePrice(BigDecimal.ZERO);

        when(imClient.getReturnAmount(any())).thenReturn(imReturnAmountRsp);
        var autoReturnRsp = createAutoReturnResponse();
        when(imClient.autoReturn(any())).thenAnswer(invocation -> {
            autoReturnRequestRef.set(invocation.getArgument(0));
            return autoReturnRsp;
        });
        var orderId = createSuccessfulOrder(imReservationResponse);
        var orderInfoRsp = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId)
                .setUpdateOrderOnTheFly(true)
                .build());

        assertThat(getTrainReservation(orderInfoRsp).getPassengers().get(2).getCategory()).isEqualTo(PassengerCategory.BABY);
        assertThat(getTrainReservation(orderInfoRsp).getPassengers().get(2).getTicket().getBlankId()).isEqualTo(childBlankId);

        var partKey = TrainPartKeyService.getPassengerKey(
                UUID.fromString(orderInfoRsp.getResult().getService(0).getServiceId()),
                getTrainReservation(orderInfoRsp).getPassengers().get(2).getCustomerId());
        var childRefundPart = orderInfoRsp.getResult().getRefundPartsList().stream()
                .filter(x -> x.getKey().equals(partKey)).findFirst().get();
        var calculateRefundRsp = client.calculateRefundV2(TCalculateRefundReqV2.newBuilder()
                .setOrderId(orderId)
                .addContext(childRefundPart.getContext())
                .build());

        client.startRefund(TStartRefundReq.newBuilder().setOrderId(orderId)
                .setRefundToken(calculateRefundRsp.getRefundToken()).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> getTrainReservation(rsp1).getPassengers().get(2).getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDING &&
                        autoReturnRequestRef.get() != null,
                Duration.ofSeconds(10), "Ticket must be REFUNDING");

        var imRefundOrderItem = factory.createRefundOrderItem();
        imRefundOrderItem.setAgentReferenceId(autoReturnRequestRef.get().getServiceAutoReturnRequest().getAgentReferenceId());
        imRefundOrderItem.getOrderItemBlanks().get(0).setPreviousOrderItemBlankId(childBlankId);
        imRefundOrderItem.getOrderItemBlanks().get(0).setAmount(BigDecimal.ZERO);
        imOrderInfoResponse.getOrderItems().add(imRefundOrderItem);
        imOrderInfoResponseRef.set(imOrderInfoResponse);

        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CONFIRMED &&
                        getTrainReservation(rsp1).getPassengers().get(0).getTicket().getRefundStatus() == null &&
                        getTrainReservation(rsp1).getPassengers().get(1).getTicket().getRefundStatus() == null &&
                        getTrainReservation(rsp1).getPassengers().get(2).getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDED,
                Duration.ofSeconds(10), "Ticket must be REFUNDED");
    }

    @Test
    public void testRefundAllWithBabyBlank() {
        var childBlankId = 99999999;
        AtomicReference<AutoReturnRequest> autoReturnRequestRef = new AtomicReference<>();
        AtomicReference<OrderInfoResponse> imOrderInfoResponseRef = new AtomicReference<>();
        var factory = new ImOrderInfoResponseFactory();
        factory.setPassengers(2);
        factory.setBuyTicketStatus(ImOperationStatus.OK);
        factory.setBuyInsuranceStatus(ImOperationStatus.OK);
        var imOrderInfoResponse = factory.create();
        var imChildOrderInfoBlank = factory.createBlank(childBlankId);
        imChildOrderInfoBlank.setAmount(BigDecimal.ZERO);
        imOrderInfoResponse.getOrderItems().get(0).getOrderItemBlanks().add(imChildOrderInfoBlank);
        imOrderInfoResponseRef.set(imOrderInfoResponse);

        ReservationCreateResponse imReservationResponse = createReservationCreateResponse();
        imReservationResponse.getReservationResults().get(0).getPassengers().get(2).setOrderItemBlankId(childBlankId);
        var imChildReserveBlank = ImReservationCreateResponseFactory.create()
                .createReservationBlankResponse(childBlankId, "FreeChild");
        imChildReserveBlank.setAmount(BigDecimal.ZERO);
        imChildReserveBlank.setBaseFare(BigDecimal.ZERO);
        imChildReserveBlank.setAdditionalPrice(BigDecimal.ZERO);
        imChildReserveBlank.setServicePrice(BigDecimal.ZERO);
        imChildReserveBlank.setVatRateValues(new ArrayList<>());
        imReservationResponse.getReservationResults().get(0).getBlanks().add(imChildReserveBlank);
        when(imClient.reservationCreate(any(), any())).thenReturn(imReservationResponse);

        when(imClient.orderInfo(anyInt())).thenAnswer(invocation -> imOrderInfoResponseRef.get());
        when(imClient.orderInfo(anyInt(), any())).thenAnswer(invocation -> imOrderInfoResponseRef.get());
        ReturnAmountResponse imReturnAmountRsp = new ReturnAmountResponse();
        imReturnAmountRsp.setServiceReturnResponse(new RailwayReturnAmountResponse());
        var blank1 = new RailwayReturnBlankResponse();
        blank1.setAmount(BigDecimal.valueOf(350));
        blank1.setServicePrice(BigDecimal.valueOf(156));
        blank1.setPurchaseOrderItemBlankId(20000001);
        var blank2 = new RailwayReturnBlankResponse();
        blank2.setAmount(BigDecimal.valueOf(350));
        blank2.setServicePrice(BigDecimal.valueOf(156));
        blank2.setPurchaseOrderItemBlankId(20000002);
        var blank3 = new RailwayReturnBlankResponse();
        blank3.setAmount(BigDecimal.ZERO);
        blank3.setServicePrice(BigDecimal.ZERO);
        blank3.setPurchaseOrderItemBlankId(childBlankId);
        imReturnAmountRsp.getServiceReturnResponse().setBlanks(List.of(blank1, blank2, blank3));
        when(imClient.getReturnAmount(any())).thenReturn(imReturnAmountRsp);
        var autoReturnRsp = createAutoReturnResponse();
        when(imClient.autoReturn(any())).thenAnswer(invocation -> {
            autoReturnRequestRef.set(invocation.getArgument(0));
            return autoReturnRsp;
        });
        var orderId = createSuccessfulOrder(imReservationResponse);
        var orderInfoRsp = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId)
                .setUpdateOrderOnTheFly(true)
                .build());

        var refundPart = orderInfoRsp.getResult().getRefundPartsList().stream()
                .filter(x -> x.getType() == ERefundPartType.RPT_ORDER).findFirst().get();
        var calculateRefundRsp = client.calculateRefundV2(TCalculateRefundReqV2.newBuilder()
                .setOrderId(orderId)
                .addContext(refundPart.getContext())
                .build());

        client.startRefund(TStartRefundReq.newBuilder().setOrderId(orderId)
                .setRefundToken(calculateRefundRsp.getRefundToken()).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> getTrainReservation(rsp1).getPassengers().get(2).getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDING &&
                        autoReturnRequestRef.get() != null,
                Duration.ofSeconds(10), "Ticket must be REFUNDING");

        var imRefundOrderItem = factory.createRefundOrderItem();
        imRefundOrderItem.setAgentReferenceId(autoReturnRequestRef.get().getServiceAutoReturnRequest().getAgentReferenceId());
        var imRefundBlank1 = factory.createRefundBlank(20000001);
        var imRefundBlank2 = factory.createRefundBlank(20000002);
        var imRefundBlank3 = factory.createRefundBlank(childBlankId);
        imRefundBlank3.setPreviousOrderItemBlankId(childBlankId);
        imRefundBlank3.setAmount(BigDecimal.ZERO);
        imRefundOrderItem.setOrderItemBlanks(List.of(imRefundBlank1, imRefundBlank2, imRefundBlank3));
        imOrderInfoResponse.getOrderItems().add(imRefundOrderItem);
        imOrderInfoResponseRef.set(imOrderInfoResponse);

        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_REFUNDED &&
                        getTrainReservation(rsp1).getPassengers().get(0).getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDED &&
                        getTrainReservation(rsp1).getPassengers().get(1).getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDED &&
                        getTrainReservation(rsp1).getPassengers().get(2).getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDED,
                Duration.ofSeconds(10), "Ticket must be REFUNDED");
    }


    private TrainReservation getTrainReservation(TGetOrderInfoRsp rsp3) {
        var payloadJson = rsp3.getResult().getService(0).getServiceInfo().getPayload();
        return fromTJson(payloadJson, TrainReservation.class);
    }

    private InsuranceCheckoutResponse createInsuranceCheckoutResponse() {
        var response = new InsuranceCheckoutResponse();
        response.setAmount(BigDecimal.valueOf(100));
        response.setOrderCustomerId(10000001);
        response.setOrderItemId(30000002);
        return response;
    }

    private InsurancePricingResponse createInsurancePricingResponse() {
        return createInsurancePricingResponse(3);
    }

    private InsurancePricingResponse createInsurancePricingResponse(int passengers) {
        var response = new InsurancePricingResponse();
        var result = new RailwayTravelPricingResult();
        response.setPricingResult(result);
        result.setProductPricingInfoList(new ArrayList<>());
        for (int i = 0; i < passengers; i++) {
            var product = new RailwayTravelProductPricingInfo();
            result.getProductPricingInfoList().add(product);
            product.setCompensation(BigDecimal.valueOf(100500));
            product.setAmount(BigDecimal.valueOf(100));
            product.setOrderCustomerId(10000001 + i);
            product.setCompany("CCC");
            product.setProductPackage("random");
            product.setProvider("PPP");
        }
        return response;
    }

    private OrderInfoResponse createImOrderInfoResponse(ImOperationStatus buyTicketStatus,
                                                        ImOperationStatus buyInsuranceStatus) {
        var factory = new ImOrderInfoResponseFactory();
        factory.setPassengers(2);
        factory.setBuyTicketStatus(buyTicketStatus);
        factory.setBuyInsuranceStatus(buyInsuranceStatus);
        return factory.create();
    }

    private OrderInfoResponse createImOrderInfoResponseWithRefund(String refundReferenceId,
                                                                  boolean refundIsExternallyLoaded) {
        var factory = new ImOrderInfoResponseFactory();
        factory.setPassengers(2);
        factory.setRefundReferenceId(refundReferenceId);
        factory.setRefundIsExternallyLoaded(refundIsExternallyLoaded);
        var result = factory.create();
        result.getOrderItems().get(result.getOrderItems().size() - 1)
                .getOrderItemBlanks().get(0).setPreviousOrderItemBlankId(20000002);
        return result;
    }

    private ReturnAmountResponse createReturnAmountResponse() {
        var rsp = new ReturnAmountResponse();
        rsp.setServiceReturnResponse(new RailwayReturnAmountResponse());
        var blank1 = new RailwayReturnBlankResponse();
        rsp.getServiceReturnResponse().setBlanks(List.of(blank1));
        blank1.setAmount(BigDecimal.valueOf(350));
        blank1.setServicePrice(BigDecimal.valueOf(156));
        blank1.setPurchaseOrderItemBlankId(20000002);
        return rsp;
    }

    private AutoReturnResponse createAutoReturnResponse() {
        var rsp = new AutoReturnResponse();
        rsp.setServiceReturnResponse(new RailwayAutoReturnResponse());
        rsp.getServiceReturnResponse().setAgentReferenceId("");
        var blank1 = new RailwayReturnBlankResponse();
        rsp.getServiceReturnResponse().setBlanks(List.of(blank1));
        blank1.setAmount(BigDecimal.valueOf(350));
        blank1.setServicePrice(BigDecimal.valueOf(156));
        blank1.setPurchaseOrderItemBlankId(20000002);
        return rsp;
    }

    private void authorizePayment(String orderId) {
        transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            assertThat(order.getCurrentInvoice()).isNotNull();
            Invoice invoice = order.getCurrentInvoice();
            String purchaseToken = invoice.getPurchaseToken();
            MockTrustClient mockTrustClient = (MockTrustClient) trustClient;
            mockTrustClient.paymentAuthorized(purchaseToken);
            return null;
        });
    }

    @Test
    public void testChangeRegistrationStatus() {
        var electronicRegistrationResponse = new ElectronicRegistrationResponse();
        electronicRegistrationResponse.setExpirationElectronicRegistrationDateTime(LocalDateTime.now().plusMinutes(60));
        when(imClient.orderInfo(anyInt())).thenReturn(createImOrderInfoResponse(ImOperationStatus.OK,
                ImOperationStatus.OK));
        when(imClient.changeElectronicRegistration(any())).thenReturn(electronicRegistrationResponse);
        var orderId = createSuccessfulOrder();
        var request = TChangeTrainRegistrationStatusReq.newBuilder()
                .setOrderId(orderId)
                .setEnabled(false)
                .addBlankIds(20000001).addBlankIds(20000002).build();

        waitForPredicateOrTimeout(client, orderId,
                rsp -> getTrainReservation(rsp).getPassengers().get(0).getTicket().getImBlankStatus()
                        == ImBlankStatus.REMOTE_CHECK_IN,
                Duration.ofSeconds(3), "OrderItem must be in REMOTE_CHECK_IN state");

        client.changeTrainRegistrationStatus(request);

        waitForPredicateOrTimeout(client, orderId,
                rsp -> getTrainReservation(rsp).getPassengers().get(0).getTicket().getImBlankStatus()
                        == ImBlankStatus.NO_REMOTE_CHECK_IN && !rsp.getResult().getUserActionScheduled(),
                Duration.ofSeconds(3), "OrderItem must be in NO_REMOTE_CHECK_IN state and not scheduled");

        verify(imClient).changeElectronicRegistration(any());
    }

    private String createSuccessfulOrder() {
        return createSuccessfulOrder(1234567890);
    }

    private String createSuccessfulOrder(Integer imOrderId) {
        ReservationCreateResponse reservationResponse = createReservationCreateResponse();
        reservationResponse.setOrderId(imOrderId);
        return createSuccessfulOrder(reservationResponse);
    }

    private String createSuccessfulOrder(ReservationCreateResponse reservationResponse) {
        when(imClient.reservationCreate(any(), any())).thenReturn(reservationResponse);
        when(imClient.insurancePricing(any())).thenReturn(createInsurancePricingResponse());
        when(imClient.insuranceCheckout(any())).thenReturn(createInsuranceCheckoutResponse());
        TCreateOrderReq createOrderReq = createOrderRequest();
        TCreateOrderRsp resp = client.createOrder(createOrderReq);
        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getGenericOrderState()).isEqualTo(EOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_RESERVED,
                Duration.ofSeconds(10), "Order must be in OS_RESERVED state");
        client.checkout(TCheckoutReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getGenericOrderState() == EOrderState.OS_WAITING_PAYMENT,
                Duration.ofSeconds(10), "Order must be in OS_WAITING_PAYMENT state");

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_return_url").build()); // safely ignoring response, as we don't need payment url

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                Duration.ofSeconds(10), "Invoice must be in IS_WAIT_FOR_PAYMENT state");
        authorizePayment(orderId);

        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_CONFIRMED,
                Duration.ofSeconds(10), "Order must be in OS_CONFIRMED state");

        return orderId;
    }

    protected TCreateOrderReq createOrderRequest() {
        var factory = new TrainOrderItemFactory();
        TrainOrderItem orderItem = factory.createTrainOrderItem();
        TrainReservation payload = orderItem.getPayload();
        payload.getPassengers().add(factory.createTrainPassenger());
        factory.setCategory(PassengerCategory.BABY);
        factory.setTariffCode("baby");
        factory.setTariffType(TariffType.BABY);
        factory.setDocumentType(DocumentType.BIRTH_CERTIFICATE);
        payload.getPassengers().add(factory.createTrainPassenger());
        payload.getPassengers().forEach(x -> x.setRequestedPlaces(null));
        return createOrderRequest(ProtoUtils.toTJson(payload).getValue()).build();
    }

    @Override
    protected EOrderType orderTypeForOrderReq() {
        return EOrderType.OT_GENERIC;
    }

    @Override
    protected TCreateServiceReq.Builder getServiceForPayload(String payloadJson) {
        return TCreateServiceReq.newBuilder()
                .setServiceType(EServiceType.PT_TRAIN)
                .setSourcePayload(TJson.newBuilder().setValue(payloadJson).build());
    }

    @TestConfiguration
    static class IntegrationTestConfiguration {
        @Bean
        @Primary
        public TrustClient trainTrustClient() {
            return new MockTrustClient();
        }

        @Bean
        public TrustClientProvider trustClientProvider(@Autowired TrustClient trainTrustClient) {
            return paymentProfile -> trainTrustClient;
        }

        @Bean
        @Primary
        public TrainTariffInfoDataProvider trainTariffInfoDataProvider() {
            var tariffInfos = Arrays.asList(
                    TTrainTariffInfo.newBuilder()
                            .setId(1)
                            .setCode("full")
                            .setTitleRu("full")
                            .setImRequestCode("Full")
                            .setImResponseCodes("Full")
                            .setWithoutPlace(false)
                            .setMinAge(0)
                            .setMinAgeIncludesBirthday(false)
                            .setMaxAge(150)
                            .setMaxAgeIncludesBirthday(false)
                            .setNeedDocument(false).build(),
                    TTrainTariffInfo.newBuilder()
                            .setId(2)
                            .setCode("baby")
                            .setTitleRu("baby")
                            .setImRequestCode("Full")
                            .setImResponseCodes("FreeChild")
                            .setWithoutPlace(true)
                            .setMinAge(0)
                            .setMinAgeIncludesBirthday(false)
                            .setMaxAge(5)
                            .setMaxAgeIncludesBirthday(true)
                            .setNeedDocument(false).build(),
                    TTrainTariffInfo.newBuilder()
                            .setId(3)
                            .setCode("child")
                            .setTitleRu("child")
                            .setImRequestCode("Full")
                            .setImResponseCodes("Child")
                            .setWithoutPlace(false)
                            .setMinAge(0)
                            .setMinAgeIncludesBirthday(false)
                            .setMaxAge(10)
                            .setMaxAgeIncludesBirthday(true)
                            .setNeedDocument(false).build());

            TrainTariffInfoDataProviderProperties config = new TrainTariffInfoDataProviderProperties();
            config.setTablePath("tablePath");
            config.setIndexPath("./train-tariff-index");
            config.setProxy(new ArrayList<>());

            TestLuceneIndexBuilder<TTrainTariffInfo> luceneIndexBuilder = new TestLuceneIndexBuilder<TTrainTariffInfo>()
                    .setLuceneData(tariffInfos);

            return new TrainTariffInfoService(config, luceneIndexBuilder);
        }
    }
}
