package ru.yandex.travel.orders.integration.train;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

import org.junit.Test;
import org.mockito.Mockito;
import org.mockito.stubbing.Answer;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.TestConfiguration;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;
import org.springframework.test.context.TestPropertySource;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TJson;
import ru.yandex.travel.dicts.rasp.proto.TTrainTariffInfo;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.configurations.TrainTariffInfoDataProviderProperties;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.integration.train.factories.ImOrderInfoResponseFactory;
import ru.yandex.travel.orders.integration.train.factories.ImReservationCreateResponseFactory;
import ru.yandex.travel.orders.integration.train.factories.TrainOrderItemFactory;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.proto.TAddInsuranceReq;
import ru.yandex.travel.orders.proto.TCalculateRefundReq;
import ru.yandex.travel.orders.proto.TCreateOrderReq;
import ru.yandex.travel.orders.proto.TCreateOrderRsp;
import ru.yandex.travel.orders.proto.TCreateServiceReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.proto.TOrderInfo;
import ru.yandex.travel.orders.proto.TReserveReq;
import ru.yandex.travel.orders.proto.TStartPaymentReq;
import ru.yandex.travel.orders.proto.TStartRefundReq;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.services.train.tariffinfo.TrainTariffInfoDataProvider;
import ru.yandex.travel.orders.services.train.tariffinfo.TrainTariffInfoService;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.train.proto.ETrainOrderState;
import ru.yandex.travel.orders.workflow.train.proto.TTrainCalculateRefundReqInfo;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.train.model.TrainTicketRefundStatus;
import ru.yandex.travel.train.partners.im.ImClient;
import ru.yandex.travel.train.partners.im.model.AutoReturnRequest;
import ru.yandex.travel.train.partners.im.model.AutoReturnResponse;
import ru.yandex.travel.train.partners.im.model.ImBlankStatus;
import ru.yandex.travel.train.partners.im.model.RailwayAutoReturnResponse;
import ru.yandex.travel.train.partners.im.model.RailwayReturnAmountResponse;
import ru.yandex.travel.train.partners.im.model.RailwayReturnBlankResponse;
import ru.yandex.travel.train.partners.im.model.ReturnAmountResponse;
import ru.yandex.travel.train.partners.im.model.insurance.InsuranceCheckoutResponse;
import ru.yandex.travel.train.partners.im.model.insurance.InsurancePricingResponse;
import ru.yandex.travel.train.partners.im.model.insurance.RailwayTravelPricingResult;
import ru.yandex.travel.train.partners.im.model.insurance.RailwayTravelProductPricingInfo;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationStatus;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderInfoResponse;
import ru.yandex.travel.yt_lucene_index.TestLuceneIndexBuilder;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyBoolean;
import static org.mockito.ArgumentMatchers.anyInt;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static ru.yandex.travel.commons.proto.ProtoUtils.fromTJson;
import static ru.yandex.travel.orders.integration.IntegrationUtils.waitForPredicateOrTimeout;

@TestPropertySource(properties = {
        "trust-db-mock.enabled=true",
        "quartz.enabled=true",
        "workflow-processing.pending-workflow-polling-interval=100ms",
        "trust-hotels.clearing-refresh-timeout=1s",
        "trust-hotels.payment-refresh-timeout=1s",
        "single-node.auto-start=true",
        "train-workflow.check-ticket-refund-delay=1s",
        "train-workflow.check-ticket-refund-task-period=200ms",
        "train-workflow.check-ticket-refund-max-tries=50"
})
@SuppressWarnings("ResultOfMethodCallIgnored")
public class TrainOrderFlowMockPaymentTests extends AbstractTrainOrderFlowTest {

    @MockBean
    private ImClient imClient;

    @Autowired
    private TransactionTemplate transactionTemplate;

    @Autowired
    private OrderRepository orderRepository;

    private static TrainReservation obtainPayload(TGetOrderInfoRsp response) {
        return ProtoUtils.fromTJson(
                response.getResult().getService(0).getServiceInfo().getPayload(), TrainReservation.class);
    }

    @Test
    public void testOrderReserved() {
        var reservationResponseFactory = ImReservationCreateResponseFactory.create();
        var reservationResponse = reservationResponseFactory.createReservationCreateResponse();
        when(imClient.reservationCreate(any(), any())).thenReturn(reservationResponse);
        var imOrderInfoResponse = createImOrderInfoResponse(ImOperationStatus.OK, ImOperationStatus.OK);
        when(imClient.orderInfo(anyInt())).thenReturn(imOrderInfoResponse);
        when(imClient.orderInfo(anyInt(), any())).thenReturn(imOrderInfoResponse);
        when(imClient.insurancePricing(any())).thenReturn(createInsurancePricingResponse());
        when(imClient.insuranceCheckout(any())).thenReturn(createInsuranceCheckoutResponse());
        TCreateOrderReq createOrderReq = createOrderRequest();
        TCreateOrderRsp resp = client.createOrder(createOrderReq);
        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getTrainOrderState()).isEqualTo(ETrainOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT,
                Duration.ofSeconds(10), "Order must be in OS_WAITING_PAYMENT state");
        getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getService(0).getServiceInfo().getGenericOrderItemState())
                .isEqualTo(EOrderItemState.IS_RESERVED);
        var ticket = ProtoUtils.fromTJson(
                getOrderInfoRsp.getResult().getService(0).getServiceInfo().getPayload(),
                TrainReservation.class).getPassengers().get(0).getTicket();
        assertThat(ticket.getImBlankStatus()).isNull();
        assertThat(ticket.isPendingElectronicRegistration()).isEqualTo(false);

        verify(imClient).insurancePricing(any());
        verify(imClient).reservationCreate(any(), any());

        client.addInsurance(TAddInsuranceReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT
                        && getTrainReservation(rsp3).getInsuranceStatus() == InsuranceStatus.CHECKED_OUT,
                Duration.ofSeconds(10), "Order must be in OS_WAITING_PAYMENT state and InsuranceStatus must be " +
                        "CHECKED_OUT");

        verify(imClient).insuranceCheckout(any());

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_failed_payment_url").build()); // safely ignoring response, as we don't need
        // payment url

        TGetOrderInfoRsp confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getTrainOrderState() == ETrainOrderState.OS_CONFIRMED,
                Duration.ofSeconds(10), "Order must be in OS_CONFIRMED state");
        assertThat(confirmedOrderResponse.getResult().getServiceCount()).isEqualTo(1);

        var blankToUpdate = imOrderInfoResponse.findBuyRailwayItems().get(0).getOrderItemBlanks().get(0);
        blankToUpdate.setBlankStatus(ImBlankStatus.REFUNDED);
        var getOrderInfoRspUpdated = client.getOrderInfo(
                TGetOrderInfoReq.newBuilder().setOrderId(orderId).setUpdateOrderOnTheFly(true).build());

        waitForPredicateOrTimeout(client, orderId,
                rsp4 -> obtainPayload(rsp4).getPassengers().get(0).getTicket().getImBlankStatus()
                        == ImBlankStatus.REFUNDED,
                Duration.ofSeconds(10), "Ticket must be in REFUNDED ImBlankStatus");

        waitForPredicateOrTimeout(() -> Mockito.mockingDetails(yaSmsService).getInvocations().size() > 0,
                Duration.ofSeconds(5), "Test notifications should be sent in 5 seconds at most");

        verify(yaSmsService).sendSms(any(), eq("+79111111111"));
    }

    @Test
    public void testInsuranceAutoReturn() {
        var reservationResponseFactory = ImReservationCreateResponseFactory.create();
        var reservationResponse = reservationResponseFactory.createReservationCreateResponse();
        when(imClient.reservationCreate(any(), any())).thenReturn(reservationResponse);
        var imOrderInfoResponse = createImOrderInfoResponse(ImOperationStatus.OK, ImOperationStatus.FAILED);
        when(imClient.orderInfo(anyInt())).thenReturn(imOrderInfoResponse);
        when(imClient.insurancePricing(any())).thenReturn(createInsurancePricingResponse());
        when(imClient.insuranceCheckout(any())).thenReturn(createInsuranceCheckoutResponse());
        TCreateOrderReq createOrderReq = createOrderRequest();
        TCreateOrderRsp resp = client.createOrder(createOrderReq);
        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getTrainOrderState()).isEqualTo(ETrainOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT,
                Duration.ofSeconds(10), "Order must be in OS_WAITING_PAYMENT state");

        client.addInsurance(TAddInsuranceReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT
                        && getTrainReservation(rsp3).getInsuranceStatus() == InsuranceStatus.CHECKED_OUT,
                Duration.ofSeconds(10), "Order must be in OS_WAITING_PAYMENT state and InsuranceStatus must be " +
                        "CHECKED_OUT");

        verify(imClient).insuranceCheckout(any());

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_return_url").build()); // safely ignoring response, as we don't need payment url

        TGetOrderInfoRsp confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getTrainOrderState() == ETrainOrderState.OS_CONFIRMED
                        && getTrainReservation(rsp1).getInsuranceStatus() == InsuranceStatus.AUTO_RETURN,
                Duration.ofSeconds(10), "Order must be in OS_CONFIRMED state and insurance in AUTO_RETURN");
        assertThat(confirmedOrderResponse.getResult().getServiceCount()).isEqualTo(1);
    }

    @Test
    public void testInsuranceImmediateClear() {
        var reservationResponseFactory = ImReservationCreateResponseFactory.create();
        var reservationResponse = reservationResponseFactory.createReservationCreateResponse();
        when(imClient.reservationCreate(any(), any())).thenReturn(reservationResponse);
        var imOrderInfoResponse = createImOrderInfoResponse(ImOperationStatus.OK, ImOperationStatus.OK);
        when(imClient.orderInfo(anyInt())).thenReturn(imOrderInfoResponse);
        when(imClient.insurancePricing(any())).thenReturn(createInsurancePricingResponse());
        when(imClient.insuranceCheckout(any())).thenReturn(createInsuranceCheckoutResponse());
        TCreateOrderReq createOrderReq = createOrderRequest();
        TCreateOrderRsp resp = client.createOrder(createOrderReq);
        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getTrainOrderState()).isEqualTo(ETrainOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT,
                Duration.ofSeconds(10), "Order must be in OS_WAITING_PAYMENT state");

        client.addInsurance(TAddInsuranceReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT
                        && getTrainReservation(rsp3).getInsuranceStatus() == InsuranceStatus.CHECKED_OUT,
                Duration.ofSeconds(10), "Order must be in OS_WAITING_PAYMENT state and InsuranceStatus must be " +
                        "CHECKED_OUT");

        verify(imClient).insuranceCheckout(any());

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_return_url").build()); // safely ignoring response, as we don't need payment url

        TGetOrderInfoRsp confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getTrainOrderState() == ETrainOrderState.OS_CONFIRMED
                        && getTrainReservation(rsp1).getInsuranceStatus() == InsuranceStatus.CHECKED_OUT
                        && rsp1.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_CLEARED,
                Duration.ofSeconds(5),
                "Order must be in OS_CONFIRMED state and insurance in CHECKED_OUT and invoice in IS_CLEARED");
        assertThat(confirmedOrderResponse.getResult().getServiceCount()).isEqualTo(1);
    }

    @Test
    public void testOrderRefund() throws InterruptedException {
        AtomicReference<AutoReturnRequest> autoReturnRequestRef = new AtomicReference<>();

        CountDownLatch autoReturnCalled = new CountDownLatch(1);
        AtomicReference<AutoReturnRequest> autoReturnRequest = new AtomicReference<>();

        AtomicReference<OrderInfoResponse> orderInfoResponseRef = new AtomicReference<>();

        var nonRefundedOrderInfoResponse = new ImOrderInfoResponseFactory(ImOperationStatus.OK, null, 1,
                null, false, LocalDateTime.now().plusDays(1)).create();
        orderInfoResponseRef.set(nonRefundedOrderInfoResponse);
        when(imClient.orderInfo(anyInt())).thenAnswer((Answer<OrderInfoResponse>) invocation -> orderInfoResponseRef.get());
        when(imClient.orderInfo(anyInt(), any())).thenAnswer((Answer<OrderInfoResponse>) invocation -> orderInfoResponseRef.get());
        when(imClient.autoReturn(any())).thenAnswer(invocation -> {
            autoReturnRequest.set(invocation.getArgument(0));
            autoReturnCalled.countDown();
            return new AutoReturnResponse();
        });

        var orderId = createSuccessfulOrder();

        var orderInfoRsp = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId)
                .setUpdateOrderOnTheFly(true).build());
        var trainCalculateRefundInfo = TTrainCalculateRefundReqInfo.newBuilder()
                .addBlankIds(getTrainReservation(orderInfoRsp).getPassengers().get(0).getTicket().getBlankId()).build();
        var calculateRefundRsp = client.calculateRefund(TCalculateRefundReq.newBuilder().setOrderId(orderId)
                .setTrainCalculateRefundInfo(trainCalculateRefundInfo).build());

        client.startRefund(TStartRefundReq.newBuilder().setOrderId(orderId)
                .setRefundToken(calculateRefundRsp.getRefundToken()).build());

        autoReturnCalled.await(2, TimeUnit.SECONDS);

        orderInfoResponseRef.set(
                createImOrderInfoResponseWithRefund(
                        autoReturnRequest.get().getServiceAutoReturnRequest().getAgentReferenceId(),
                        false
                )
        );

        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getTrainOrderState() == ETrainOrderState.OS_REFUNDED &&
                        getTrainReservation(rsp1).getPassengers().get(0).getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDED,
                Duration.ofSeconds(10), "Ticket must be REFUNDED");
    }

    @Test
    public void testCancelOnConfirmationFailed() {
        String orderId = createPayedOrder();

        var imOrderInfoResponse = createImOrderInfoResponse(ImOperationStatus.FAILED, null);
        when(imClient.orderInfo(anyInt())).thenReturn(imOrderInfoResponse);

        assertThat(getInvoiceTotal(orderId)).isPositive();

        //TODO (syukhno) correct test after fix refunded invoice state
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                // todo(mbobrov): another part of the WaitingRefundAfterCancellationStateHandler
                //  .ensureSingleInvoiceAndMoneyRefunded fix
                rsp1 -> rsp1.getResult().getTrainOrderState() == ETrainOrderState.OS_CANCELLED,
                Duration.ofSeconds(10), "Order must be in OS_CANCELLED state");

        assertThat(orderInfoRsp.getResult().getInvoiceCount()).isEqualTo(1);
        assertThat(orderInfoRsp.getResult().getInvoice(0).getTrustInvoiceState()).isEqualTo(ETrustInvoiceState.IS_CANCELLED);
        assertThat(getInvoiceTotal(orderId)).isZero();
    }

    private String createPayedOrder() {
        var payload = new TrainOrderItemFactory().createTrainReservation();
        TCreateOrderReq createOrderReq = createOrderRequest(ProtoUtils.toTJson(payload).getValue()).build();
        TCreateOrderRsp resp = client.createOrder(createOrderReq);
        var passengers = getPayload(resp.getNewOrder()).getPassengers().size();
        var reservationResponseFactory = ImReservationCreateResponseFactory.create(false, passengers);
        var reservationResponse = reservationResponseFactory.createReservationCreateResponse();
        when(imClient.reservationCreate(any(), any())).thenReturn(reservationResponse);
        var imOrderInfoResponse = new ImOrderInfoResponseFactory(ImOperationStatus.OK, null,
                passengers, null, false, null).create();
        when(imClient.orderInfo(anyInt())).thenReturn(imOrderInfoResponse);
        when(imClient.insurancePricing(any())).thenReturn(createInsurancePricingResponse(passengers));

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getTrainOrderState()).isEqualTo(ETrainOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT,
                Duration.ofSeconds(10), "Order must be in OS_WAITING_PAYMENT state");
        getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getService(0).getServiceInfo().getGenericOrderItemState())
                .isEqualTo(EOrderItemState.IS_RESERVED);

        verify(imClient).insurancePricing(any());
        verify(imClient).reservationCreate(any(), any());

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_return_url").build()); // safely ignoring response, as we don't need payment url

        return orderId;
    }

    private BigDecimal getInvoiceTotal(String orderId) {
        return transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            assertThat(order.getCurrentInvoice()).isNotNull();
            Invoice invoice = order.getCurrentInvoice();
            BigDecimal total = BigDecimal.ZERO;
            for (var i : invoice.getInvoiceItems()) {
                total = total.add(i.getPrice());
            }
            return total;
        });
    }

    private TrainReservation getTrainReservation(TGetOrderInfoRsp rsp3) {
        var payloadJson = rsp3.getResult().getService(0).getServiceInfo().getPayload();
        return fromTJson(payloadJson, TrainReservation.class);
    }

    private InsuranceCheckoutResponse createInsuranceCheckoutResponse() {
        var response = new InsuranceCheckoutResponse();
        response.setAmount(BigDecimal.valueOf(100));
        response.setOrderCustomerId(10000001);
        response.setOrderItemId(30000002);
        return response;
    }

    private InsurancePricingResponse createInsurancePricingResponse() {
        return createInsurancePricingResponse(1);
    }

    private InsurancePricingResponse createInsurancePricingResponse(int passengers) {
        var response = new InsurancePricingResponse();
        var result = new RailwayTravelPricingResult();
        response.setPricingResult(result);
        result.setProductPricingInfoList(new ArrayList<>());
        for (int i = 0; i < passengers; i++) {
            var product = new RailwayTravelProductPricingInfo();
            result.getProductPricingInfoList().add(product);
            product.setCompensation(BigDecimal.valueOf(100500));
            product.setAmount(BigDecimal.valueOf(100));
            product.setOrderCustomerId(10000001 + i);
            product.setCompany("CCC");
            product.setProductPackage("random");
            product.setProvider("PPP");
        }
        return response;
    }

    private OrderInfoResponse createImOrderInfoResponse(ImOperationStatus buyTicketStatus,
                                                        ImOperationStatus buyInsuranceStatus) {
        var factory = new ImOrderInfoResponseFactory();
        factory.setBuyTicketStatus(buyTicketStatus);
        factory.setBuyInsuranceStatus(buyInsuranceStatus);
        return factory.create();
    }

    private OrderInfoResponse createImOrderInfoResponseWithRefund(String refundReferenceId,
                                                                  boolean refundIsExternallyLoaded) {
        var factory = new ImOrderInfoResponseFactory();
        factory.setRefundReferenceId(refundReferenceId);
        factory.setRefundIsExternallyLoaded(refundIsExternallyLoaded);
        return factory.create();
    }

    private ReturnAmountResponse createReturnAmountResponse() {
        var rsp = new ReturnAmountResponse();
        rsp.setServiceReturnResponse(new RailwayReturnAmountResponse());
        var blank1 = new RailwayReturnBlankResponse();
        rsp.getServiceReturnResponse().setBlanks(List.of(blank1));
        blank1.setAmount(BigDecimal.valueOf(350));
        blank1.setServicePrice(BigDecimal.valueOf(156));
        blank1.setPurchaseOrderItemBlankId(20000001);
        return rsp;
    }

    private AutoReturnResponse createAutoReturnResponse() {
        var rsp = new AutoReturnResponse();
        rsp.setServiceReturnResponse(new RailwayAutoReturnResponse());
        rsp.getServiceReturnResponse().setAgentReferenceId("");
        var blank1 = new RailwayReturnBlankResponse();
        rsp.getServiceReturnResponse().setBlanks(List.of(blank1));
        blank1.setAmount(BigDecimal.valueOf(350));
        blank1.setServicePrice(BigDecimal.valueOf(156));
        blank1.setPurchaseOrderItemBlankId(20000001);
        return rsp;
    }

    private String createSuccessfulOrder() {
        return createSuccessfulOrder(1234567890);
    }

    private String createSuccessfulOrder(Integer imOrderId) {
        var reservationResponseFactory = ImReservationCreateResponseFactory.create();
        var reservationResponse = reservationResponseFactory.createReservationCreateResponse();
        reservationResponse.setOrderId(imOrderId);
        when(imClient.reservationCreate(any(), any())).thenReturn(reservationResponse);
        when(imClient.insurancePricing(any())).thenReturn(createInsurancePricingResponse());
        when(imClient.insuranceCheckout(any())).thenReturn(createInsuranceCheckoutResponse());
        when(imClient.getReturnAmount(any())).thenReturn(createReturnAmountResponse());
        TCreateOrderReq createOrderReq = createOrderRequest();
        TCreateOrderRsp resp = client.createOrder(createOrderReq);
        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getTrainOrderState()).isEqualTo(ETrainOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT,
                Duration.ofSeconds(10), "Order must be in OS_WAITING_PAYMENT state");

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_return_url").build()); // safely ignoring response, as we don't need payment url

        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getTrainOrderState() == ETrainOrderState.OS_CONFIRMED,
                Duration.ofSeconds(10), "Order must be in OS_CONFIRMED state");

        return orderId;
    }

    private TrainReservation getPayload(TOrderInfo orderInfo) {
        return getPayload(orderInfo.getService(0).getServiceInfo().getPayload());
    }

    private TrainReservation getPayload(TJson payloadJson) {
        return ProtoUtils.fromTJson(payloadJson, TrainReservation.class);
    }


    protected TCreateOrderReq createOrderRequest() {
        var factory = new TrainOrderItemFactory();
        TrainReservation payload = factory.createTrainReservation();
        return createOrderRequest(ProtoUtils.toTJson(payload).getValue()).build();
    }


    @Override
    protected TCreateOrderReq.Builder createOrderRequest(String... payloads) {
        return super.createOrderRequest(payloads)
                .setOrderType(EOrderType.OT_TRAIN)
                .setMockPayment(true);
    }

    @Override
    protected TCreateServiceReq.Builder getServiceForPayload(String payloadJson) {
        return TCreateServiceReq.newBuilder()
                .setServiceType(EServiceType.PT_TRAIN)
                .setSourcePayload(TJson.newBuilder().setValue(payloadJson).build());
    }

    @TestConfiguration
    static class IntegrationTestConfiguration {

        @Bean
        @Primary
        public TrainTariffInfoDataProvider trainTariffInfoDataProvider() {
            var tariffInfos = List.of(
                    TTrainTariffInfo.newBuilder().setId(1)
                            .setCode("full")
                            .setTitleRu("full")
                            .setImRequestCode("Full")
                            .setImResponseCodes("Full,Senior")
                            .setWithoutPlace(false)
                            .setMinAge(0)
                            .setMinAgeIncludesBirthday(false)
                            .setMaxAge(150)
                            .setMaxAgeIncludesBirthday(false)
                            .setNeedDocument(false).build());

            TrainTariffInfoDataProviderProperties config = new TrainTariffInfoDataProviderProperties();
            config.setTablePath("tablePath");
            config.setIndexPath("./train-tariff-index");
            config.setProxy(new ArrayList<>());

            TestLuceneIndexBuilder<TTrainTariffInfo> luceneIndexBuilder = new TestLuceneIndexBuilder<TTrainTariffInfo>()
                    .setLuceneData(tariffInfos);

            return new TrainTariffInfoService(config, luceneIndexBuilder);
        }
    }
}
