package ru.yandex.travel.orders.integration.train;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.UUID;

import org.junit.Test;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.TestConfiguration;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TJson;
import ru.yandex.travel.dicts.rasp.proto.TTrainTariffInfo;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.commons.proto.ETrainChangeElectronicRegistrationOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainInsuranceCheckoutConfirmOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainInsuranceCheckoutOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainInsurancePricingOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainRefundCheckoutOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainRefundPricingOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainReservationConfirmOutcome;
import ru.yandex.travel.orders.commons.proto.ETrainReservationCreateOutcome;
import ru.yandex.travel.orders.commons.proto.TTrainTestContext;
import ru.yandex.travel.orders.configurations.TrainTariffInfoDataProviderProperties;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.proto.TAddInsuranceReq;
import ru.yandex.travel.orders.proto.TCreateOrderReq;
import ru.yandex.travel.orders.proto.TCreateOrderRsp;
import ru.yandex.travel.orders.proto.TCreateServiceReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.proto.TReserveReq;
import ru.yandex.travel.orders.proto.TStartPaymentReq;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.repository.mock.MockTrainOrderItemRepository;
import ru.yandex.travel.orders.repository.mock.MockTrainOrderRepository;
import ru.yandex.travel.orders.services.mock.MockImClient;
import ru.yandex.travel.orders.services.mock.MockImResponseGenerator;
import ru.yandex.travel.orders.services.mock.MockTrustClient;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.train.ImClientProvider;
import ru.yandex.travel.orders.services.train.tariffinfo.TrainTariffInfoDataProvider;
import ru.yandex.travel.orders.services.train.tariffinfo.TrainTariffInfoService;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.train.proto.ETrainOrderState;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.workflow.single_operation.SingleOperationService;
import ru.yandex.travel.yt_lucene_index.TestLuceneIndexBuilder;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyBoolean;
import static org.mockito.Mockito.when;
import static ru.yandex.travel.commons.proto.ProtoUtils.fromTJson;
import static ru.yandex.travel.orders.integration.IntegrationUtils.waitForPredicateOrTimeout;

@SuppressWarnings("ResultOfMethodCallIgnored")
public class TrainOrderFlowMockTests extends AbstractTrainOrderFlowTest {
    @Autowired
    private OrderRepository orderRepository;
    @Autowired
    private TransactionTemplate transactionTemplate;
    @Autowired
    private TrustClient trustClient;

    @Test
    public void testOrderReserved() {
        TCreateOrderReq createOrderReq = createOrderRequest(1).build();
        TCreateOrderRsp resp = client.createOrder(createOrderReq);
        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getTrainOrderState()).isEqualTo(ETrainOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT,
                Duration.ofSeconds(10), "Order must be in OS_WAITING_PAYMENT state");
        getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getService(0).getServiceInfo().getGenericOrderItemState())
                .isEqualTo(EOrderItemState.IS_RESERVED);
        var ticket = ProtoUtils.fromTJson(
                getOrderInfoRsp.getResult().getService(0).getServiceInfo().getPayload(),
                TrainReservation.class).getPassengers().get(0).getTicket();
        assertThat(ticket.getImBlankStatus()).isNull();
        assertThat(ticket.isPendingElectronicRegistration()).isEqualTo(false);

//        verify(imClient).insurancePricing(any());
//        verify(imClient).reservationCreate(any(), any());

        client.addInsurance(TAddInsuranceReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT
                        && getTrainReservation(rsp3).getInsuranceStatus() == InsuranceStatus.CHECKED_OUT,
                Duration.ofSeconds(10), "Order must be in OS_WAITING_PAYMENT state and InsuranceStatus must be " +
                        "CHECKED_OUT");

//        verify(imClient).insuranceCheckout(any());

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_failed_payment_url").build()); // safely ignoring response, as we don't need
        // payment url

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getCurrentInvoice().getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                Duration.ofSeconds(10), "Invoice must be in IS_WAIT_FOR_PAYMENT state");

        failPayment(orderId);

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getCurrentInvoice().getTrustInvoiceState() == ETrustInvoiceState.IS_PAYMENT_NOT_AUTHORIZED,
                Duration.ofSeconds(10), "Current invoice must be ins IS_PAYMENT_NOT_AUTHORIZED state");

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_success_payment_url").build()); // safely ignoring response, as we don't need

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getCurrentInvoice().getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                Duration.ofSeconds(10), "Invoice must be in IS_WAIT_FOR_PAYMENT state");

        authorizePayment(orderId);

        TGetOrderInfoRsp confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getTrainOrderState() == ETrainOrderState.OS_CONFIRMED,
                Duration.ofSeconds(10), "Order must be in OS_CONFIRMED state");
        assertThat(confirmedOrderResponse.getResult().getServiceCount()).isEqualTo(1);
        assertThat(confirmedOrderResponse.getResult().getInvoiceCount()).isEqualTo(2);

        var reservation = getTrainReservation(confirmedOrderResponse);
        assertThat(reservation.getPassengers().get(0).getTicket().getBookedTariffCode()).isEqualTo("full");
        assertThat(reservation.getPassengers().get(0).getTicket().getRawTariffName()).isEqualTo("Полный");

//        var blankToUpdate = imOrderInfoResponse.findBuyRailwayItem().getOrderItemBlanks().get(0);
//        blankToUpdate.setBlankStatus(ImBlankStatus.REFUNDED);
//        var getOrderInfoRspUpdated = client.getOrderInfo(
//                TGetOrderInfoReq.newBuilder().setOrderId(orderId).setUpdateOrderOnTheFly(true).build());
//
//        waitForPredicateOrTimeout(client, orderId,
//                rsp4 -> obtainPayload(rsp4).getPassengers().get(0).getTicket().getImBlankStatus()
//                        == ImBlankStatus.REFUNDED,
//                Duration.ofSeconds(10), "Ticket must be in REFUNDED ImBlankStatus");
//
//        waitForPredicateOrTimeout(() -> Mockito.mockingDetails(yaSmsService).getInvocations().size() > 0,
//                Duration.ofSeconds(5), "Test notifications should be sent in 5 seconds at most");
//
//        verify(yaSmsService).sendSms(any(), eq("+79111111111"));
    }

    protected TCreateServiceReq.Builder getServiceForPayload(String payloadJson) {
        return TCreateServiceReq.newBuilder()
                .setServiceType(EServiceType.PT_TRAIN)
                .setTrainTestContext(
                        TTrainTestContext.newBuilder()
                                .setInsurancePricingOutcome(ETrainInsurancePricingOutcome.IPO_SUCCESS)
                                .setInsuranceCheckoutOutcome(ETrainInsuranceCheckoutOutcome.ICO_SUCCESS)
                                .setInsuranceCheckoutConfirmOutcome(ETrainInsuranceCheckoutConfirmOutcome.ICCO_SUCCESS)
                                .setRefundPricingOutcome(ETrainRefundPricingOutcome.RPO_SUCCESS)
                                .setRefundCheckoutOutcome(ETrainRefundCheckoutOutcome.RCO_SUCCESS)
                                .setReservationCreateOutcome(ETrainReservationCreateOutcome.RCRO_SUCCESS)
                                .setReservationConfirmOutcome(ETrainReservationConfirmOutcome.RCOO_SUCCESS)
                                .setTrainChangeElectronicRegistrationOutcome(ETrainChangeElectronicRegistrationOutcome.CERO_SUCCESS)
                                .build())
                .setSourcePayload(TJson.newBuilder().setValue(payloadJson).build());
    }

    @Override
    protected EOrderType orderTypeForOrderReq() {
        return EOrderType.OT_TRAIN;
    }

    private TrainReservation getTrainReservation(TGetOrderInfoRsp response) {
        var payloadJson = response.getResult().getService(0).getServiceInfo().getPayload();
        return fromTJson(payloadJson, TrainReservation.class);
    }

    private void authorizePayment(String orderId) {
        transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            assertThat(order.getCurrentInvoice()).isNotNull();
            Invoice invoice = order.getCurrentInvoice();
            String purchaseToken = invoice.getPurchaseToken();
            MockTrustClient mockTrustClient = (MockTrustClient) trustClient;
            mockTrustClient.paymentAuthorized(purchaseToken);
            return null;
        });
    }

    private void failPayment(String orderId) {
        transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            assertThat(order.getCurrentInvoice()).isNotNull();
            Invoice invoice = order.getCurrentInvoice();
            MockTrustClient mockTrustClient = (MockTrustClient) trustClient;
            mockTrustClient.paymentNotAuthorized(invoice.getPurchaseToken());
            return null;
        });
    }

    @TestConfiguration
    static class IntegrationTestConfiguration {
        @Bean
        public MockImClient mockImClient(MockTrainOrderRepository mockTrainOrderRepository,
                                         MockTrainOrderItemRepository mockTrainOrderItemRepository,
                                         TransactionTemplate transactionTemplate,
                                         SingleOperationService singleOperationService,
                                         MockImResponseGenerator mockImResponseGenerator) {
            return new MockImClient(mockTrainOrderRepository, mockTrainOrderItemRepository, transactionTemplate,
                    singleOperationService, mockImResponseGenerator);
        }

        @Bean
        @Primary
        public ImClientProvider imClientProvider(MockImClient mockImClient) {
            return orderItem -> mockImClient;
        }

        @Bean
        @Primary
        public TrustClient trainTrustClient() {
            return new MockTrustClient();
        }

        @Bean
        public TrustClientProvider trustClientProvider(@Autowired TrustClient trainTrustClient) {
            return paymentProfile -> trainTrustClient;
        }

        @Bean
        @Primary
        public TrainTariffInfoDataProvider trainTariffInfoDataProvider() {
            var tariffInfos = Arrays.asList(
                    TTrainTariffInfo.newBuilder().setId(1)
                            .setCode("full")
                            .setTitleRu("Полный")
                            .setImRequestCode("Full")
                            .setImResponseCodes("Full")
                            .setWithoutPlace(false)
                            .setMinAge(0)
                            .setMinAgeIncludesBirthday(false)
                            .setMaxAge(150)
                            .setMaxAgeIncludesBirthday(false)
                            .setNeedDocument(false).build(),

                    TTrainTariffInfo.newBuilder().setId(2)
                            .setCode("senior")
                            .setTitleRu("Сеньор")
                            .setImRequestCode("Senior")
                            .setImResponseCodes("Senior")
                            .setWithoutPlace(false)
                            .setMinAge(0)
                            .setMinAgeIncludesBirthday(false)
                            .setMaxAge(150)
                            .setMaxAgeIncludesBirthday(false)
                            .setNeedDocument(false).build()
            );

            TrainTariffInfoDataProviderProperties config = new TrainTariffInfoDataProviderProperties();
            config.setTablePath("tablePath");
            config.setIndexPath("./train-tariff-index");
            config.setProxy(new ArrayList<>());

            TestLuceneIndexBuilder<TTrainTariffInfo> luceneIndexBuilder = new TestLuceneIndexBuilder<TTrainTariffInfo>()
                    .setLuceneData(tariffInfos);

            return new TrainTariffInfoService(config, luceneIndexBuilder);
        }
    }
}
