package ru.yandex.travel.orders.integration.train;

import java.io.IOException;
import java.math.BigDecimal;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;

import io.grpc.StatusRuntimeException;
import lombok.extern.slf4j.Slf4j;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.TestName;
import org.mockito.Mockito;
import org.mockito.stubbing.Answer;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.TestConfiguration;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Primary;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.commons.proto.ECurrency;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TJson;
import ru.yandex.travel.commons.proto.TPrice;
import ru.yandex.travel.dicts.rasp.proto.TTrainTariffInfo;
import ru.yandex.travel.orders.admin.proto.OrdersAdminInterfaceV1Grpc;
import ru.yandex.travel.orders.admin.proto.TByFiscalItemRefund;
import ru.yandex.travel.orders.admin.proto.TManualMoneyRefundReq;
import ru.yandex.travel.orders.admin.proto.TUpdateTrainTicketsReq;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.configurations.TrainTariffInfoDataProviderProperties;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.grpc.OrdersAdminService;
import ru.yandex.travel.orders.integration.IntegrationUtils;
import ru.yandex.travel.orders.integration.train.factories.ImOrderInfoResponseFactory;
import ru.yandex.travel.orders.integration.train.factories.ImReservationCreateResponseFactory;
import ru.yandex.travel.orders.integration.train.factories.TrainOrderItemFactory;
import ru.yandex.travel.orders.proto.EInvoiceType;
import ru.yandex.travel.orders.proto.OrderInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.TAddInsuranceReq;
import ru.yandex.travel.orders.proto.TCalculateRefundReq;
import ru.yandex.travel.orders.proto.TChangeTrainRegistrationStatusReq;
import ru.yandex.travel.orders.proto.TCreateOrderReq;
import ru.yandex.travel.orders.proto.TCreateOrderRsp;
import ru.yandex.travel.orders.proto.TCreateServiceReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TGetOrderInfoRsp;
import ru.yandex.travel.orders.proto.TOrderInfo;
import ru.yandex.travel.orders.proto.TReserveReq;
import ru.yandex.travel.orders.proto.TStartCancellationReq;
import ru.yandex.travel.orders.proto.TStartPaymentReq;
import ru.yandex.travel.orders.proto.TStartRefundReq;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.services.UrlShortenerService;
import ru.yandex.travel.orders.services.mock.MockTrustClient;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.payments.model.TrustCreateRefundRequest;
import ru.yandex.travel.orders.services.payments.model.TrustCreateRefundResponse;
import ru.yandex.travel.orders.services.train.tariffinfo.TrainTariffInfoDataProvider;
import ru.yandex.travel.orders.services.train.tariffinfo.TrainTariffInfoService;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.train.proto.ETrainOrderState;
import ru.yandex.travel.orders.workflow.train.proto.TTrainCalculateRefundReqInfo;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.train.model.TrainTicketRefundStatus;
import ru.yandex.travel.train.partners.im.ImClient;
import ru.yandex.travel.train.partners.im.ImClientException;
import ru.yandex.travel.train.partners.im.ImClientInvalidPassengerEmailException;
import ru.yandex.travel.train.partners.im.model.AutoReturnRequest;
import ru.yandex.travel.train.partners.im.model.AutoReturnResponse;
import ru.yandex.travel.train.partners.im.model.ElectronicRegistrationResponse;
import ru.yandex.travel.train.partners.im.model.ImBlankStatus;
import ru.yandex.travel.train.partners.im.model.PendingElectronicRegistration;
import ru.yandex.travel.train.partners.im.model.RailwayAutoReturnResponse;
import ru.yandex.travel.train.partners.im.model.RailwayBlankInfo;
import ru.yandex.travel.train.partners.im.model.RailwayReturnAmountResponse;
import ru.yandex.travel.train.partners.im.model.RailwayReturnBlankResponse;
import ru.yandex.travel.train.partners.im.model.ReservationCreateResponse;
import ru.yandex.travel.train.partners.im.model.ReturnAmountResponse;
import ru.yandex.travel.train.partners.im.model.UpdateBlanksResponse;
import ru.yandex.travel.train.partners.im.model.insurance.InsuranceCheckoutResponse;
import ru.yandex.travel.train.partners.im.model.insurance.InsurancePricingResponse;
import ru.yandex.travel.train.partners.im.model.insurance.RailwayTravelPricingResult;
import ru.yandex.travel.train.partners.im.model.insurance.RailwayTravelProductPricingInfo;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationStatus;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderInfoResponse;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.yt_lucene_index.TestLuceneIndexBuilder;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyBoolean;
import static org.mockito.ArgumentMatchers.anyInt;
import static org.mockito.ArgumentMatchers.anyLong;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.atLeastOnce;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static ru.yandex.travel.commons.proto.ProtoUtils.fromTJson;
import static ru.yandex.travel.orders.integration.IntegrationUtils.waitForPredicateOrTimeout;

@SuppressWarnings("ResultOfMethodCallIgnored")
@Slf4j
public class TrainOrderFlowTests extends AbstractTrainOrderFlowTest {

    private static final Duration TIMEOUT = Duration.ofSeconds(10);
    @Rule
    public TestName testName = new TestName();

    @Autowired
    private OrdersAdminService ordersAdminService;
    @MockBean
    private ImClient imClient;

    private OrdersAdminInterfaceV1Grpc.OrdersAdminInterfaceV1BlockingStub adminClient;

    @Autowired
    private TransactionTemplate transactionTemplate;

    @Autowired
    private OrderRepository orderRepository;

    @Autowired
    private TrustClient trustClient;

    @Before
    public void setUpCredentialsContext() throws IOException {
        log.info("Starting the {} test", testName.getMethodName());
        adminClient = IntegrationUtils.createAdminServerAndBlockingStub(cleanupRule, ordersAdminService);
    }

    static TrainReservation getTrainReservation(TGetOrderInfoRsp response) {
        var payloadJson = response.getResult().getService(0).getServiceInfo().getPayload();
        return fromTJson(payloadJson, TrainReservation.class);
    }

    private static InsuranceCheckoutResponse createInsuranceCheckoutResponse() {
        var response = new InsuranceCheckoutResponse();
        response.setAmount(BigDecimal.valueOf(100));
        response.setOrderCustomerId(10000001);
        response.setOrderItemId(30000002);
        return response;
    }

    private static InsurancePricingResponse createInsurancePricingResponse() {
        return createInsurancePricingResponse(1);
    }

    private static InsurancePricingResponse createInsurancePricingResponse(int passengers) {
        var response = new InsurancePricingResponse();
        var result = new RailwayTravelPricingResult();
        response.setPricingResult(result);
        result.setProductPricingInfoList(new ArrayList<>());
        for (int i = 0; i < passengers; i++) {
            var product = new RailwayTravelProductPricingInfo();
            result.getProductPricingInfoList().add(product);
            product.setCompensation(BigDecimal.valueOf(100500));
            product.setAmount(BigDecimal.valueOf(100));
            product.setOrderCustomerId(10000001 + i);
            product.setCompany("CCC");
            product.setProductPackage("random");
            product.setProvider("PPP");
        }
        return response;
    }

    static OrderInfoResponse createImOrderInfoResponseWithRefund(String refundReferenceId,
                                                                 boolean refundIsExternallyLoaded) {
        var factory = new ImOrderInfoResponseFactory();
        factory.setRefundReferenceId(refundReferenceId);
        factory.setRefundIsExternallyLoaded(refundIsExternallyLoaded);
        return factory.create();
    }

    private static ReturnAmountResponse createReturnAmountResponse() {
        var rsp = new ReturnAmountResponse();
        rsp.setServiceReturnResponse(new RailwayReturnAmountResponse());
        var blank1 = new RailwayReturnBlankResponse();
        rsp.getServiceReturnResponse().setBlanks(List.of(blank1));
        blank1.setAmount(BigDecimal.valueOf(350));
        blank1.setServicePrice(BigDecimal.valueOf(156));
        blank1.setPurchaseOrderItemBlankId(20000001);
        return rsp;
    }

    static AutoReturnResponse createAutoReturnResponse() {
        var rsp = new AutoReturnResponse();
        rsp.setServiceReturnResponse(new RailwayAutoReturnResponse());
        rsp.getServiceReturnResponse().setAgentReferenceId("");
        var blank1 = new RailwayReturnBlankResponse();
        rsp.getServiceReturnResponse().setBlanks(List.of(blank1));
        blank1.setAmount(BigDecimal.valueOf(350));
        blank1.setServicePrice(BigDecimal.valueOf(156));
        blank1.setPurchaseOrderItemBlankId(20000001);
        return rsp;
    }

    private static void authorizePayment(TransactionTemplate transactionTemplate, OrderRepository orderRepository,
                                         TrustClient trustClient, String orderId) {
        transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            assertThat(order.getCurrentInvoice()).isNotNull();
            Invoice invoice = order.getCurrentInvoice();
            String purchaseToken = invoice.getPurchaseToken();
            MockTrustClient mockTrustClient = (MockTrustClient) trustClient;
            mockTrustClient.paymentAuthorized(purchaseToken);
            return null;
        });
    }

    static String createSuccessfulOrder(ImClient imClient,
                                        OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub client,
                                        UrlShortenerService urlShortenerService,
                                        TransactionTemplate transactionTemplate,
                                        OrderRepository orderRepository,
                                        TrustClient trustClient,
                                        TCreateOrderReq createOrderReq) {
        var reservationResponseFactory = ImReservationCreateResponseFactory.create();
        var reservationResponse = reservationResponseFactory.createReservationCreateResponse();
        reservationResponse.setOrderId(1234567890);
        when(imClient.reservationCreate(any(), any())).thenReturn(reservationResponse);
        when(imClient.insurancePricing(any())).thenReturn(createInsurancePricingResponse());
        when(imClient.insuranceCheckout(any())).thenReturn(createInsuranceCheckoutResponse());
        when(imClient.getReturnAmount(any())).thenReturn(createReturnAmountResponse());
        when(imClient.autoReturn(any())).thenReturn(createAutoReturnResponse());
        TCreateOrderRsp resp = client.createOrder(createOrderReq);
        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getTrainOrderState()).isEqualTo(ETrainOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT,
                TIMEOUT, "Order must be in OS_WAITING_PAYMENT state");

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_return_url").build()); // safely ignoring response, as we don't need payment url

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                TIMEOUT, "Invoice must be in IS_WAIT_FOR_PAYMENT state");
        authorizePayment(transactionTemplate, orderRepository, trustClient, orderId);

        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getTrainOrderState() == ETrainOrderState.OS_CONFIRMED,
                TIMEOUT, "Order must be in OS_CONFIRMED state");

        return orderId;
    }

    private static TrainReservation obtainPayload(TGetOrderInfoRsp response) {
        return ProtoUtils.fromTJson(
                response.getResult().getService(0).getServiceInfo().getPayload(), TrainReservation.class);
    }

    @Test
    public void testOrderReserved() {
        var reservationResponseFactory = ImReservationCreateResponseFactory.create();
        var reservationResponse = reservationResponseFactory.createReservationCreateResponse();
        when(imClient.reservationCreate(any(), any())).thenReturn(reservationResponse);
        var imOrderInfoResponse = createImOrderInfoResponse(ImOperationStatus.OK, ImOperationStatus.OK);
        when(imClient.orderInfo(anyInt())).thenReturn(imOrderInfoResponse);
        when(imClient.orderInfo(anyInt(), any())).thenReturn(imOrderInfoResponse);
        when(imClient.insurancePricing(any())).thenReturn(createInsurancePricingResponse());
        when(imClient.insuranceCheckout(any())).thenReturn(createInsuranceCheckoutResponse());
        TCreateOrderReq.Builder createOrderReq = createOrderRequest(1);
        TCreateOrderRsp resp = client.createOrder(createOrderReq.build());
        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getTrainOrderState()).isEqualTo(ETrainOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT,
                TIMEOUT, "Order must be in OS_WAITING_PAYMENT state");
        getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getService(0).getServiceInfo().getGenericOrderItemState())
                .isEqualTo(EOrderItemState.IS_RESERVED);
        var ticket = ProtoUtils.fromTJson(
                getOrderInfoRsp.getResult().getService(0).getServiceInfo().getPayload(),
                TrainReservation.class).getPassengers().get(0).getTicket();
        assertThat(ticket.getImBlankStatus()).isNull();
        assertThat(ticket.isPendingElectronicRegistration()).isEqualTo(false);

        verify(imClient).insurancePricing(any());
        verify(imClient).reservationCreate(any(), any());

        client.addInsurance(TAddInsuranceReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT
                        && getTrainReservation(rsp3).getInsuranceStatus() == InsuranceStatus.CHECKED_OUT,
                TIMEOUT, "Order must be in OS_WAITING_PAYMENT state and InsuranceStatus must be " +
                        "CHECKED_OUT");

        verify(imClient).insuranceCheckout(any());

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_failed_payment_url").build()); // safely ignoring response, as we don't need
        // payment url

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getCurrentInvoice().getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                TIMEOUT, "Invoice must be in IS_WAIT_FOR_PAYMENT state");

        failPayment(orderId);

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getCurrentInvoice().getTrustInvoiceState() == ETrustInvoiceState.IS_PAYMENT_NOT_AUTHORIZED,
                TIMEOUT, "Current invoice must be ins IS_PAYMENT_NOT_AUTHORIZED state");

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_success_payment_url").build()); // safely ignoring response, as we don't need

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getCurrentInvoice().getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                TIMEOUT, "Invoice must be in IS_WAIT_FOR_PAYMENT state");

        authorizePayment(orderId);

        TGetOrderInfoRsp confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getTrainOrderState() == ETrainOrderState.OS_CONFIRMED,
                TIMEOUT, "Order must be in OS_CONFIRMED state");
        assertThat(confirmedOrderResponse.getResult().getServiceCount()).isEqualTo(1);
        assertThat(confirmedOrderResponse.getResult().getInvoiceCount()).isEqualTo(2);

        var reservation = getTrainReservation(confirmedOrderResponse);
        assertThat(reservation.getPassengers().get(0).getTicket().getBookedTariffCode()).isEqualTo("full");
        assertThat(reservation.getPassengers().get(0).getTicket().getRawTariffName()).isEqualTo("Полный");

        var blankToUpdate = imOrderInfoResponse.findBuyRailwayItems().get(0).getOrderItemBlanks().get(0);
        blankToUpdate.setBlankStatus(ImBlankStatus.REFUNDED);
        var getOrderInfoRspUpdated = client.getOrderInfo(
                TGetOrderInfoReq.newBuilder().setOrderId(orderId).setUpdateOrderOnTheFly(true).build());

        waitForPredicateOrTimeout(client, orderId,
                rsp4 -> obtainPayload(rsp4).getPassengers().get(0).getTicket().getImBlankStatus()
                        == ImBlankStatus.REFUNDED,
                TIMEOUT, "Ticket must be in REFUNDED ImBlankStatus");

        waitForPredicateOrTimeout(() -> Mockito.mockingDetails(yaSmsService).getInvocations().size() > 0,
                TIMEOUT, "Test notifications should be sent in 5 seconds at most");

        verify(yaSmsService).sendSms(any(), eq("+79111111111"));
    }

    @Test
    public void testInsuranceAutoReturn() {
        var reservationResponseFactory = ImReservationCreateResponseFactory.create();
        var reservationResponse = reservationResponseFactory.createReservationCreateResponse();
        when(imClient.reservationCreate(any(), any())).thenReturn(reservationResponse);
        var imOrderInfoResponse = createImOrderInfoResponse(ImOperationStatus.OK, ImOperationStatus.FAILED);
        when(imClient.orderInfo(anyInt())).thenReturn(imOrderInfoResponse);
        when(imClient.orderInfo(anyInt(), any())).thenReturn(imOrderInfoResponse);
        when(imClient.insurancePricing(any())).thenReturn(createInsurancePricingResponse());
        when(imClient.insuranceCheckout(any())).thenReturn(createInsuranceCheckoutResponse());
        TCreateOrderReq.Builder createOrderReq = createOrderRequest(1);
        TCreateOrderRsp resp = client.createOrder(createOrderReq.build());
        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getTrainOrderState()).isEqualTo(ETrainOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT,
                TIMEOUT, "Order must be in OS_WAITING_PAYMENT state");

        client.addInsurance(TAddInsuranceReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT
                        && getTrainReservation(rsp3).getInsuranceStatus() == InsuranceStatus.CHECKED_OUT,
                TIMEOUT, "Order must be in OS_WAITING_PAYMENT state and InsuranceStatus must be " +
                        "CHECKED_OUT");

        verify(imClient).insuranceCheckout(any());

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_return_url").build()); // safely ignoring response, as we don't need payment url

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                TIMEOUT, "Invoice must be in IS_WAIT_FOR_PAYMENT state");
        authorizePayment(orderId);

        TGetOrderInfoRsp confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getTrainOrderState() == ETrainOrderState.OS_CONFIRMED
                        && getTrainReservation(rsp1).getInsuranceStatus() == InsuranceStatus.AUTO_RETURN,
                TIMEOUT, "Order must be in OS_CONFIRMED state and insurance in AUTO_RETURN");
        assertThat(confirmedOrderResponse.getResult().getServiceCount()).isEqualTo(1);
    }

    @Test
    @Ignore("TODO flaky. remove in TRAINS-6367")
    public void testInsuranceImmediateClear() {
        var reservationResponseFactory = ImReservationCreateResponseFactory.create();
        var reservationResponse = reservationResponseFactory.createReservationCreateResponse();
        when(imClient.reservationCreate(any(), any())).thenReturn(reservationResponse);
        var imOrderInfoResponse = createImOrderInfoResponse(ImOperationStatus.OK, ImOperationStatus.OK);
        when(imClient.orderInfo(anyInt())).thenReturn(imOrderInfoResponse);
        when(imClient.insurancePricing(any())).thenReturn(createInsurancePricingResponse());
        when(imClient.insuranceCheckout(any())).thenReturn(createInsuranceCheckoutResponse());
        TCreateOrderReq.Builder createOrderReq = createOrderRequest(1);
        TCreateOrderRsp resp = client.createOrder(createOrderReq.build());
        when(urlShortenerService.shorten(any(), anyBoolean())).thenReturn("http://ya.ru/veryshorturl");

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getTrainOrderState()).isEqualTo(ETrainOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT,
                TIMEOUT, "Order must be in OS_WAITING_PAYMENT state");

        client.addInsurance(TAddInsuranceReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT
                        && getTrainReservation(rsp3).getInsuranceStatus() == InsuranceStatus.CHECKED_OUT,
                TIMEOUT, "Order must be in OS_WAITING_PAYMENT state and InsuranceStatus must be " +
                        "CHECKED_OUT");

        verify(imClient).insuranceCheckout(any());

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_return_url").build()); // safely ignoring response, as we don't need payment url

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                TIMEOUT, "Invoice must be in IS_WAIT_FOR_PAYMENT state");
        authorizePayment(orderId);

        TGetOrderInfoRsp confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getTrainOrderState() == ETrainOrderState.OS_CONFIRMED
                        && getTrainReservation(rsp1).getInsuranceStatus() == InsuranceStatus.CHECKED_OUT
                        && rsp1.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_CLEARED,
                TIMEOUT,
                "Order must be in OS_CONFIRMED state and insurance in CHECKED_OUT and invoice in IS_CLEARED");
        assertThat(confirmedOrderResponse.getResult().getServiceCount()).isEqualTo(1);
    }

    @Test
    public void testDoubleDiscountWithRefund() {
        var factory = new TrainOrderItemFactory();
        factory.setTariffCode("senior");
        factory.setDocumentNumber("3709123454");
        var trainReservation = factory.createTrainReservation();
        AtomicReference<AutoReturnRequest> autoReturnRequestRef = new AtomicReference<>();
        AtomicReference<OrderInfoResponse> orderInfoResponseRef = new AtomicReference<>();
        var nonRefundedOrderInfoResponse = new ImOrderInfoResponseFactory(ImOperationStatus.OK, null, 1,
                null, false, LocalDateTime.now().plusDays(1)).create();
        orderInfoResponseRef.set(nonRefundedOrderInfoResponse);
        when(imClient.orderInfo(anyInt())).thenAnswer((Answer<OrderInfoResponse>) invocation -> orderInfoResponseRef.get());
        when(imClient.orderInfo(anyInt(), any())).thenAnswer((Answer<OrderInfoResponse>) invocation -> orderInfoResponseRef.get());
        var orderId = createSuccessfulOrder(createOrderRequest(trainReservation).build());
        var autoReturnRsp = createAutoReturnResponse();
        when(imClient.autoReturn(any())).thenAnswer(invocation -> {
            autoReturnRequestRef.set(invocation.getArgument(0));
            return autoReturnRsp;
        });
        var getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        var getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getPayload(getOrderInfoRsp.getResult()).getPassengers().get(0).isDiscountDenied()).isFalse();
        assertThat(getPayload(getOrderInfoRsp.getResult()).getPassengers().get(0).getTariffCode()).isEqualTo("senior");

        // 1st senior is confirmed, try to reserve 2nd senior

        var createOrderReq2 = createOrderRequest(trainReservation);
        var resp2 = client.createOrder(createOrderReq2.build());
        var orderId2 = resp2.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId2).build());
        waitForPredicateOrTimeout(client, orderId2,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT,
                TIMEOUT, "Order must be in OS_WAITING_PAYMENT state");
        TGetOrderInfoReq getOrderInfoRequest2 = TGetOrderInfoReq.newBuilder().setOrderId(orderId2).build();
        TGetOrderInfoRsp getOrderInfoRsp2 = client.getOrderInfo(getOrderInfoRequest2);
        assertThat(getPayload(getOrderInfoRsp2.getResult()).getPassengers().get(0).isDiscountDenied()).isTrue();
        assertThat(getPayload(getOrderInfoRsp2.getResult()).getPassengers().get(0).getTariffCode()).isEqualTo("full");

        // 2nd senior reserved as full, lets refund 1st senior

        var orderInfoRsp = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId)
                .setUpdateOrderOnTheFly(true).build());
        var trainCalculateRefundInfo = TTrainCalculateRefundReqInfo.newBuilder()
                .addBlankIds(getTrainReservation(orderInfoRsp).getPassengers().get(0).getTicket().getBlankId()).build();
        var calculateRefundRsp = client.calculateRefund(TCalculateRefundReq.newBuilder().setOrderId(orderId)
                .setTrainCalculateRefundInfo(trainCalculateRefundInfo).build());
        client.startRefund(TStartRefundReq.newBuilder().setOrderId(orderId)
                .setRefundToken(calculateRefundRsp.getRefundToken()).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> getTrainReservation(rsp1).getPassengers().get(0).getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDING &&
                        autoReturnRequestRef.get() != null,
                TIMEOUT, "Ticket must be REFUNDING");
        orderInfoResponseRef.set(
                createImOrderInfoResponseWithRefund(
                        autoReturnRequestRef.get().getServiceAutoReturnRequest().getAgentReferenceId(), false));
        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getTrainOrderState() == ETrainOrderState.OS_REFUNDED &&
                        getTrainReservation(rsp1).getPassengers().get(0).getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDED,
                TIMEOUT, "Ticket must be REFUNDED");

        // 1st senior refunded, try to reserve 2nd senior again

        createOrderReq2 = createOrderRequest(trainReservation);
        resp2 = client.createOrder(createOrderReq2.build());
        orderId2 = resp2.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId2).build());
        waitForPredicateOrTimeout(client, orderId2,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT,
                TIMEOUT, "Order must be in OS_WAITING_PAYMENT state");
        getOrderInfoRequest2 = TGetOrderInfoReq.newBuilder().setOrderId(orderId2).build();
        getOrderInfoRsp2 = client.getOrderInfo(getOrderInfoRequest2);
        assertThat(getPayload(getOrderInfoRsp2.getResult()).getPassengers().get(0).isDiscountDenied()).isFalse();
        assertThat(getPayload(getOrderInfoRsp2.getResult()).getPassengers().get(0).getTariffCode()).isEqualTo("senior");
    }

    @Test
    public void testDoubleDiscountWithCancel() {
        var factory = new TrainOrderItemFactory();
        factory.setTariffCode("senior");
        factory.setDocumentNumber("1234123454");
        var trainReservation = factory.createTrainReservation();
        var orderId = createOrderWaitingForPayment(createOrderRequest(trainReservation).build(), null, true);
        var getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        var getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getPayload(getOrderInfoRsp.getResult()).getPassengers().get(0).isDiscountDenied()).isFalse();
        assertThat(getPayload(getOrderInfoRsp.getResult()).getPassengers().get(0).getTariffCode()).isEqualTo("senior");

        // 1st senior is reserved, try to reserve 2nd senior

        var createOrderReq2 = createOrderRequest(trainReservation);
        var resp2 = client.createOrder(createOrderReq2.build());
        var orderId2 = resp2.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId2).build());
        waitForPredicateOrTimeout(client, orderId2,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT,
                TIMEOUT, "Order must be in OS_WAITING_PAYMENT state");
        TGetOrderInfoReq getOrderInfoRequest2 = TGetOrderInfoReq.newBuilder().setOrderId(orderId2).build();
        TGetOrderInfoRsp getOrderInfoRsp2 = client.getOrderInfo(getOrderInfoRequest2);
        assertThat(getPayload(getOrderInfoRsp2.getResult()).getPassengers().get(0).isDiscountDenied()).isTrue();
        assertThat(getPayload(getOrderInfoRsp2.getResult()).getPassengers().get(0).getTariffCode()).isEqualTo("full");

        // 2nd senior reserved as full, lets cancel 1st senior

        client.startCancellation(TStartCancellationReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_REFUND_AFTER_CANCELLATION,
                TIMEOUT, "Order must be in OS_WAITING_REFUND_AFTER_CANCELLATION state");

        // 1st senior refunded, try to reserve 2nd senior again

        createOrderReq2 = createOrderRequest(trainReservation);
        resp2 = client.createOrder(createOrderReq2.build());
        orderId2 = resp2.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId2).build());
        waitForPredicateOrTimeout(client, orderId2,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT,
                TIMEOUT, "Order must be in OS_WAITING_PAYMENT state");
        getOrderInfoRequest2 = TGetOrderInfoReq.newBuilder().setOrderId(orderId2).build();
        getOrderInfoRsp2 = client.getOrderInfo(getOrderInfoRequest2);
        assertThat(getPayload(getOrderInfoRsp2.getResult()).getPassengers().get(0).isDiscountDenied()).isFalse();
        assertThat(getPayload(getOrderInfoRsp2.getResult()).getPassengers().get(0).getTariffCode()).isEqualTo("senior");
    }

    @Test
    public void testOrderRefund() {
        AtomicReference<AutoReturnRequest> autoReturnRequestRef = new AtomicReference<>();
        AtomicReference<OrderInfoResponse> orderInfoResponseRef = new AtomicReference<>();

        var nonRefundedOrderInfoResponse = new ImOrderInfoResponseFactory(ImOperationStatus.OK, null, 1,
                null, false, LocalDateTime.now().plusDays(1)).create();
        orderInfoResponseRef.set(nonRefundedOrderInfoResponse);
        when(imClient.orderInfo(anyInt())).thenAnswer((Answer<OrderInfoResponse>) invocation -> orderInfoResponseRef.get());
        when(imClient.orderInfo(anyInt(), any())).thenAnswer((Answer<OrderInfoResponse>) invocation -> orderInfoResponseRef.get());

        var orderId = createSuccessfulOrder();
        var autoReturnRsp = createAutoReturnResponse();
        when(imClient.autoReturn(any())).thenAnswer(invocation -> {
            autoReturnRequestRef.set(invocation.getArgument(0));
            return autoReturnRsp;
        });

        var orderInfoRsp = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId)
                .setUpdateOrderOnTheFly(true).build());
        var trainCalculateRefundInfo = TTrainCalculateRefundReqInfo.newBuilder()
                .addBlankIds(getTrainReservation(orderInfoRsp).getPassengers().get(0).getTicket().getBlankId()).build();
        var calculateRefundRsp = client.calculateRefund(TCalculateRefundReq.newBuilder().setOrderId(orderId)
                .setTrainCalculateRefundInfo(trainCalculateRefundInfo).build());

        client.startRefund(TStartRefundReq.newBuilder().setOrderId(orderId)
                .setRefundToken(calculateRefundRsp.getRefundToken()).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> getTrainReservation(rsp1).getPassengers().get(0).getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDING &&
                        autoReturnRequestRef.get() != null,
                TIMEOUT, "Ticket must be REFUNDING");

        orderInfoResponseRef.set(
                createImOrderInfoResponseWithRefund(
                        autoReturnRequestRef.get().getServiceAutoReturnRequest().getAgentReferenceId(), false));

        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getTrainOrderState() == ETrainOrderState.OS_REFUNDED &&
                        getTrainReservation(rsp1).getPassengers().get(0).getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDED,
                TIMEOUT, "Ticket must be REFUNDED");
    }

    @Test
    public void testOrderExpired() {
        var trainOrderItem = new TrainOrderItemFactory().createTrainOrderItem();
        var reservationResponseFactory = ImReservationCreateResponseFactory.createForOrder(trainOrderItem, null);
        var orderId = createOrderWaitingForPayment(null, reservationResponseFactory.createReservationCreateResponse()
                , false);
        transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            order.getOrderItems().get(0).setExpiresAt(Instant.now().minusSeconds(1));
            return null;
        });
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getTrainOrderState() == ETrainOrderState.OS_CANCELLED, TIMEOUT,
                "Order must be in OS_CANCELLED state");
        verify(imClient).reservationCancel(anyInt());
    }

    @Test
    public void testCancelOnConfirmationFailed() {
        String orderId = createOrderWaitingForPayment();

        var imOrderInfoResponse = createImOrderInfoResponse(ImOperationStatus.FAILED, null);
        when(imClient.orderInfo(anyInt())).thenReturn(imOrderInfoResponse);

        assertThat(getInvoiceTotal(orderId)).isPositive();
        authorizePayment(orderId);

        //TODO (syukhno) correct test after fix refunded invoice state
        TGetOrderInfoRsp orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                // todo(mbobrov): another part of the WaitingRefundAfterCancellationStateHandler
                //  .ensureSingleInvoiceAndMoneyRefunded fix
                rsp1 -> rsp1.getResult().getTrainOrderState() == ETrainOrderState.OS_CANCELLED,
                TIMEOUT, "Order must be in OS_CANCELLED state");

        assertThat(orderInfoRsp.getResult().getInvoiceCount()).isEqualTo(1);
        assertThat(orderInfoRsp.getResult().getInvoice(0).getTrustInvoiceState()).isEqualTo(ETrustInvoiceState.IS_CANCELLED);
        assertThat(getInvoiceTotal(orderId)).isZero();
    }

    @Test
    public void testGetOrderInfoWithUpdate() {
        prepareOrderCreation();

        var imOrderInfoResponse = createImOrderInfoResponse(ImOperationStatus.OK, null);
        when(imClient.updateBlanks(anyInt())).thenReturn(updateBlanksResponse());
        when(imClient.orderInfo(anyInt())).thenReturn(imOrderInfoResponse);
        when(imClient.orderInfo(anyInt(), any())).thenReturn(imOrderInfoResponse);

        var orderId = client.createOrder(createOrderRequest(1).build()).getNewOrder().getOrderId();
        var getOrderInfoRspWithoutUpdateBlanks = client.getOrderInfo(
                TGetOrderInfoReq.newBuilder().setOrderId(orderId).build());
        var payloadWithoutTickets = ProtoUtils.fromTJson(
                getOrderInfoRspWithoutUpdateBlanks.getResult().getService(0).getServiceInfo().getPayload(),
                TrainReservation.class);

        verify(imClient, never()).updateBlanks(anyInt());
        verify(imClient, never()).orderInfo(anyInt());
        assertThat(payloadWithoutTickets.getPassengers().size()).isEqualTo(1);
        assertThat(payloadWithoutTickets.getPassengers().get(0).getTicket()).isNull();

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());

        waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT,
                TIMEOUT, "Order must be in OS_WAITING_PAYMENT state");

        client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                .setSource("desktop")
                .setOrderId(orderId)
                .setReturnUrl("some_payment_url").build());

        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getCurrentInvoice().getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                TIMEOUT, "Invoice must be in IS_WAIT_FOR_PAYMENT state");

        authorizePayment(orderId);

        TGetOrderInfoRsp confirmedOrderResponse = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getTrainOrderState() == ETrainOrderState.OS_CONFIRMED,
                TIMEOUT, "Order must be in OS_CONFIRMED state");

        var getOrderInfoRspWithUpdateBlanks = client.getOrderInfo(TGetOrderInfoReq.newBuilder()
                .setOrderId(orderId).setUpdateOrderOnTheFly(true).build());
        var payloadWithTickets = ProtoUtils.fromTJson(
                getOrderInfoRspWithUpdateBlanks.getResult().getService(0).getServiceInfo().getPayload(),
                TrainReservation.class);

        var ticket = payloadWithTickets.getPassengers().get(0).getTicket();
        assertThat(ticket.getImBlankStatus()).isEqualByComparingTo(ImBlankStatus.REMOTE_CHECK_IN);
        assertThat(ticket.isPendingElectronicRegistration()).isEqualTo(false);
        verify(imClient, atLeastOnce()).updateBlanks(anyInt(), any());
        verify(imClient, atLeastOnce()).orderInfo(anyInt(), any());

        var blankToUpdate = imOrderInfoResponse.findBuyRailwayItems().get(0).getOrderItemBlanks().get(0);
        blankToUpdate.setBlankStatus(ImBlankStatus.INTERRUPTED);
        blankToUpdate.setPendingElectronicRegistration(PendingElectronicRegistration.TO_CANCEL);

        getOrderInfoRspWithUpdateBlanks = client.getOrderInfo(TGetOrderInfoReq.newBuilder()
                .setOrderId(orderId).setUpdateOrderOnTheFly(true).build());
        var payloadWithTicketsUpdated = ProtoUtils.fromTJson(
                getOrderInfoRspWithUpdateBlanks.getResult().getService(0).getServiceInfo().getPayload(),
                TrainReservation.class);

        var ticketUpdated = payloadWithTicketsUpdated.getPassengers().get(0).getTicket();
        assertThat(ticketUpdated.getImBlankStatus()).isEqualByComparingTo(ImBlankStatus.INTERRUPTED);
        assertThat(ticketUpdated.isPendingElectronicRegistration()).isEqualTo(true);

        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getTrainOrderState() == ETrainOrderState.OS_CONFIRMED &&
                        getTrainReservation(rsp1).getPassengers().get(0).getTicket().getImBlankStatus() == ImBlankStatus.INTERRUPTED,
                TIMEOUT, "Ticket blank status must be updated to INTERRUPTED");
    }

    @Test
    public void testGetOrderInfoWithSkipUpdate() {
        prepareOrderCreation();
        var imOrderInfoResponse = createImOrderInfoResponse(ImOperationStatus.OK, null);
        when(imClient.updateBlanks(anyInt())).thenReturn(updateBlanksResponse());
        when(imClient.orderInfo(anyInt())).thenReturn(imOrderInfoResponse);
        var orderId = client.createOrder(createOrderRequest(1).build()).getNewOrder().getOrderId();

        client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId).setUpdateOrderOnTheFly(true).build());

        verify(imClient, never()).updateBlanks(anyInt());
        verify(imClient, never()).orderInfo(anyInt());
    }

    @Test
    public void testCancelOrder() {
        var orderId = createOrderWaitingForPayment();
        var startCancellationReq = TStartCancellationReq.newBuilder().setOrderId(orderId).build();

        client.startCancellation(startCancellationReq);

        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_REFUND_AFTER_CANCELLATION,
                TIMEOUT, "Order must be in OS_WAITING_REFUND_AFTER_CANCELLATION state");

        verify(imClient).reservationCancel(anyInt());
        authorizePayment(orderId);

        //TODO (mbobrov) Fix invoice state after refund
        //TODO (syukhno) correct test after fix refunded invoice state
        //waitForPredicateOrTimeout(client, orderId,
        //        rsp1 -> rsp1.getResult().getTrainOrderState() == ETrainOrderState.OS_CANCELLED,
        //        TIMEOUT,"Order must be in OS_CANCELLED state");

    }

    private void prepareOrderCreation() {
        var reservationResponseFactory = ImReservationCreateResponseFactory.create();
        var reservationResponse = reservationResponseFactory.createReservationCreateResponse();
        when(imClient.reservationCreate(any(), any())).thenReturn(reservationResponse);
        when(imClient.insurancePricing(any())).thenReturn(createInsurancePricingResponse());
        when(imClient.insuranceCheckout(any())).thenReturn(createInsuranceCheckoutResponse());
    }

    private String createOrderWaitingForPayment() {
        return createOrderWaitingForPayment(null, null, true);
    }

    private String createOrderWaitingForPayment(TCreateOrderReq createOrderReq,
                                                ReservationCreateResponse reservationResponse,
                                                boolean startPayment) {
        if (createOrderReq == null) {
            createOrderReq = createOrderRequest(1).build();
        }
        if (reservationResponse == null) {
            var reservationResponseFactory = ImReservationCreateResponseFactory.create();
            reservationResponse = reservationResponseFactory.createReservationCreateResponse();
        }
        TCreateOrderRsp resp = client.createOrder(createOrderReq);
        var passengers = getPayload(resp.getNewOrder()).getPassengers().size();
        when(imClient.reservationCreate(any(), any())).thenReturn(reservationResponse);
        var imOrderInfoResponse = new ImOrderInfoResponseFactory(ImOperationStatus.OK, null,
                passengers, null, false, null).create();
        when(imClient.orderInfo(anyInt())).thenReturn(imOrderInfoResponse);
        when(imClient.insurancePricing(any())).thenReturn(createInsurancePricingResponse(passengers));

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getTrainOrderState()).isEqualTo(ETrainOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT,
                TIMEOUT, "Order must be in OS_WAITING_PAYMENT state");
        getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getService(0).getServiceInfo().getGenericOrderItemState())
                .isEqualTo(EOrderItemState.IS_RESERVED);

        verify(imClient).insurancePricing(any());
        verify(imClient).reservationCreate(any(), any());

        if (startPayment) {
            client.startPayment(TStartPaymentReq.newBuilder().setInvoiceType(EInvoiceType.IT_TRUST)
                    .setSource("desktop")
                    .setOrderId(orderId)
                    .setReturnUrl("some_return_url").build()); // safely ignoring response, as we don't need payment url

            waitForPredicateOrTimeout(client, orderId,
                    rsp2 -> rsp2.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_WAIT_FOR_PAYMENT,
                    TIMEOUT, "Invoice must be in IS_WAIT_FOR_PAYMENT state");
        }

        return orderId;
    }

    private BigDecimal getInvoiceTotal(String orderId) {
        return transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            assertThat(order.getCurrentInvoice()).isNotNull();
            Invoice invoice = order.getCurrentInvoice();
            BigDecimal total = BigDecimal.ZERO;
            for (var i : invoice.getInvoiceItems()) {
                total = total.add(i.getPrice());
            }
            return total;
        });
    }

    private OrderInfoResponse createImOrderInfoResponse(ImOperationStatus buyTicketStatus,
                                                        ImOperationStatus buyInsuranceStatus) {
        var factory = new ImOrderInfoResponseFactory();
        factory.setBuyTicketStatus(buyTicketStatus);
        factory.setBuyInsuranceStatus(buyInsuranceStatus);
        return factory.create();
    }

    private void authorizePayment(String orderId) {
        authorizePayment(transactionTemplate, orderRepository, trustClient, orderId);
    }

    private void failPayment(String orderId) {
        transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            assertThat(order.getCurrentInvoice()).isNotNull();
            Invoice invoice = order.getCurrentInvoice();
            MockTrustClient mockTrustClient = (MockTrustClient) trustClient;
            mockTrustClient.paymentNotAuthorized(invoice.getPurchaseToken());
            return null;
        });
    }

    private void startRefund(String orderId) {
        transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            assertThat(order.getCurrentInvoice()).isNotNull();
            Invoice invoice = order.getCurrentInvoice();
            String purchaseToken = invoice.getPurchaseToken();
            MockTrustClient mockTrustClient = (MockTrustClient) trustClient;
            TrustCreateRefundRequest refundRequest = new TrustCreateRefundRequest();
            refundRequest.setPurchaseToken(purchaseToken);
            TrustCreateRefundResponse refund = mockTrustClient.createRefund(refundRequest, null);
            mockTrustClient.startRefund(refund.getTrustRefundId(), null);
            return null;
        });
    }

    @Test
    public void testOrderReservationCancelled() {
        when(imClient.reservationCreate(any(), any())).thenThrow(new ImClientException(5, "Твой поезд ушел"));

        TCreateOrderReq createOrderReq = createOrderRequest(1).build();

        TCreateOrderRsp resp = client.createOrder(createOrderReq);

        String orderId = resp.getNewOrder().getOrderId();
        TGetOrderInfoReq getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build(); // we'll
        // reuse it
        TGetOrderInfoRsp getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getTrainOrderState()).isEqualTo(ETrainOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_CANCELLED,
                TIMEOUT, "Order must be in OS_CANCELLED state");
    }

    @Test
    public void testOrderItemCrashed() throws InterruptedException {
        CountDownLatch stCalled = new CountDownLatch(1);
        when(starTrekService.createIssueForGeneralOrderError(any(), anyLong(), any(), any())).then(
                invocation -> {
                    stCalled.countDown();
                    return UUID.randomUUID();
                }
        );
        when(imClient.reservationCreate(any(), any())).thenThrow(new RuntimeException("Unexpected error test"));

        TCreateOrderReq createOrderReq = createOrderRequest(1).build();
        TCreateOrderRsp resp = client.createOrder(createOrderReq);
        String orderId = resp.getNewOrder().getOrderId();
        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());

        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getWorkflowState() == EWorkflowState.WS_CRASHED, TIMEOUT,
                "Order workflow must be in WS_CRASHED");

        stCalled.await(2, TimeUnit.SECONDS);
        verify(starTrekService).createIssueForGeneralOrderError(eq(UUID.fromString(orderId)), anyLong(), any(), any());
    }

    @Test
    public void testImValidationError() {
        var originalError = new ImClientInvalidPassengerEmailException(1386, "Invalid passenger's email", null, 0);
        var reservationResponseFactory = ImReservationCreateResponseFactory.create();
        var reservationResponse = reservationResponseFactory.createReservationCreateResponse();
        when(imClient.reservationCreate(any(), any())).thenThrow(originalError).thenReturn(reservationResponse);
        var imOrderInfoResponse = createImOrderInfoResponse(ImOperationStatus.OK, ImOperationStatus.OK);
        when(imClient.orderInfo(anyInt())).thenReturn(imOrderInfoResponse);
        when(imClient.insurancePricing(any())).thenReturn(createInsurancePricingResponse());
        TCreateOrderReq.Builder createOrderReq = createOrderRequest(1);
        TCreateOrderRsp resp = client.createOrder(createOrderReq.build());

        var orderId = resp.getNewOrder().getOrderId();
        var getOrderInfoRequest = TGetOrderInfoReq.newBuilder().setOrderId(orderId).build();
        var getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getTrainOrderState()).isEqualTo(ETrainOrderState.OS_NEW);

        client.reserve(TReserveReq.newBuilder().setOrderId(orderId).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp3 -> rsp3.getResult().getTrainOrderState() == ETrainOrderState.OS_WAITING_PAYMENT,
                TIMEOUT, "Order must be in OS_WAITING_PAYMENT state");
        getOrderInfoRsp = client.getOrderInfo(getOrderInfoRequest);
        assertThat(getOrderInfoRsp.getResult().getService(0).getServiceInfo().getGenericOrderItemState())
                .isEqualTo(EOrderItemState.IS_RESERVED);

        assertThat(getTrainReservation(getOrderInfoRsp).getPassengers().get(0).isUseEmailForReservation()).isFalse();
    }

    @Test
    public void testServiceInfoRefresh() {
        var imOrderInfoResponse = createImOrderInfoResponse(ImOperationStatus.OK, null);
        when(imClient.orderInfo(anyInt())).thenReturn(imOrderInfoResponse);
        when(imClient.updateBlanks(anyInt())).thenReturn(updateBlanksResponse());
        String orderId = createSuccessfulOrder();

        adminClient.updateTrainTickets(TUpdateTrainTicketsReq.newBuilder()
                .setOrderId(orderId)
                .build());
        waitForPredicateOrTimeout(client, orderId, rsp3 -> !rsp3.getResult().getUserActionScheduled(),
                TIMEOUT, "User scheduled action couldn't be completed in provided duration");
        verify(imClient).updateBlanks(anyInt());
    }

    @Test
    public void testManualRefundLeadsToManualProcessing() {
        var response = new ImOrderInfoResponseFactory(ImOperationStatus.OK, null, 1,
                null, false, LocalDateTime.now().plusDays(1)).create();
        when(imClient.orderInfo(anyInt())).thenReturn(response);
        String orderId = createSuccessfulOrder();
        var fiscalItemId = transactionTemplate.execute(ignored -> {
            Order order = orderRepository.getOne(UUID.fromString(orderId));
            TrustInvoice invoice = (TrustInvoice) order.getInvoices().get(0);
            invoice.setState(ETrustInvoiceState.IS_CLEARED);
            return invoice.getInvoiceItems().get(0).getFiscalItemId();
        });

        //noinspection ConstantConditions
        adminClient.manualMoneyRefund(TManualMoneyRefundReq.newBuilder()
                .setOrderId(orderId)
                .setReason("Just because we can")
                .setByFiscalItemRefund(TByFiscalItemRefund.newBuilder()
                        .putTargetFiscalItems(fiscalItemId, TPrice.newBuilder()
                                .setAmount(826)
                                .setCurrency(ECurrency.C_RUB)
                                .build())).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp -> rsp.getResult().getTrainOrderState() == ETrainOrderState.OS_MANUAL_PROCESSING,
                TIMEOUT, "Order must be in OS_MANUAL_PROCESSING state");
        waitForPredicateOrTimeout(client, orderId,
                rsp2 -> rsp2.getResult().getInvoice(0).getTrustInvoiceState() == ETrustInvoiceState.IS_REFUNDING,
                TIMEOUT, "Invoice must be in IS_REFUNDING state");
        startRefund(orderId);
        waitForPredicateOrTimeout(client, orderId,
                rsp4 -> !rsp4.getResult().getUserActionScheduled(),
                TIMEOUT, "Order must have UserActionScheduled flag unset");
    }

    @Test
    public void testChangeRegistrationStatus() {
        var electronicRegistrationResponse = new ElectronicRegistrationResponse();
        electronicRegistrationResponse.setExpirationElectronicRegistrationDateTime(LocalDateTime.now().plusMinutes(60));

        var response = new ImOrderInfoResponseFactory(ImOperationStatus.OK, null, 1,
                null, false, LocalDateTime.now().plusHours(2 * 24)).create();
        when(imClient.orderInfo(anyInt())).thenReturn(response);
        doReturn(electronicRegistrationResponse).when(imClient).changeElectronicRegistration(any());
        var orderId = createSuccessfulOrder();
        var request = TChangeTrainRegistrationStatusReq.newBuilder()
                .setOrderId(orderId)
                .setEnabled(false)
                .addBlankIds(20000001).build();

        waitForPredicateOrTimeout(client, orderId,
                rsp -> getTrainReservation(rsp).getPassengers().get(0).getTicket().getImBlankStatus()
                        == ImBlankStatus.REMOTE_CHECK_IN,
                TIMEOUT, "OrderItem must be in REMOTE_CHECK_IN state");

        client.changeTrainRegistrationStatus(request);

        waitForPredicateOrTimeout(client, orderId,
                rsp -> getTrainReservation(rsp).getPassengers().get(0).getTicket().getImBlankStatus()
                        == ImBlankStatus.NO_REMOTE_CHECK_IN && !rsp.getResult().getUserActionScheduled(),
                TIMEOUT, "OrderItem must be in NO_REMOTE_CHECK_IN state and not scheduled");

        verify(imClient).changeElectronicRegistration(any());
    }

    @Test
    public void testChangeRegistrationStatusPending() {
        var electronicRegistrationResponse = new ElectronicRegistrationResponse();
        var response = new ImOrderInfoResponseFactory(ImOperationStatus.OK, null, 1,
                null, false, LocalDateTime.now().plusHours(2 * 24)).create();
        when(imClient.orderInfo(anyInt())).thenReturn(response);
        when(imClient.changeElectronicRegistration(any())).thenReturn(electronicRegistrationResponse);
        var orderId = createSuccessfulOrder();
        var request = TChangeTrainRegistrationStatusReq.newBuilder()
                .setOrderId(orderId)
                .setEnabled(false)
                .addBlankIds(20000001).build();

        waitForPredicateOrTimeout(client, orderId,
                rsp -> getTrainReservation(rsp).getPassengers().get(0).getTicket().getImBlankStatus()
                        == ImBlankStatus.REMOTE_CHECK_IN,
                TIMEOUT, "OrderItem must be in REMOTE_CHECK_IN state");

        client.changeTrainRegistrationStatus(request);

        waitForPredicateOrTimeout(client, orderId,
                rsp -> getTrainReservation(rsp).getPassengers().get(0).getTicket().isPendingElectronicRegistration()
                        && !rsp.getResult().getUserActionScheduled(),
                TIMEOUT, "OrderItem must be pending electronic registration and not scheduled");

        verify(imClient).changeElectronicRegistration(any());
    }

    @Test(expected = StatusRuntimeException.class)
    public void testChangeRegistrationStatusBlankNotValid() {
        var electronicRegistrationResponse = new ElectronicRegistrationResponse();
        electronicRegistrationResponse.setExpirationElectronicRegistrationDateTime(LocalDateTime.now().plusMinutes(60));

        var response = new ImOrderInfoResponseFactory(ImOperationStatus.OK, null, 1,
                null, false, LocalDateTime.now().plusHours(2 * 24)).create();
        when(imClient.orderInfo(anyInt())).thenReturn(response);
        when(imClient.changeElectronicRegistration(any())).thenReturn(electronicRegistrationResponse);
        var orderId = createSuccessfulOrder();

        waitForPredicateOrTimeout(client, orderId,
                rsp -> getTrainReservation(rsp).getPassengers().get(0).getTicket().getImBlankStatus()
                        == ImBlankStatus.REMOTE_CHECK_IN,
                TIMEOUT, "OrderItem must be in REMOTE_CHECK_IN state");

        var request = TChangeTrainRegistrationStatusReq.newBuilder()
                .setOrderId(orderId)
                .setEnabled(true)
                .addBlankIds(20000001).build();

        client.changeTrainRegistrationStatus(request);
    }

    private String createSuccessfulOrder() {
        return createSuccessfulOrder(createOrderRequest(1).build());
    }

    private String createSuccessfulOrder(TCreateOrderReq createOrderReq) {
        return createSuccessfulOrder(imClient, client, urlShortenerService, transactionTemplate, orderRepository,
                trustClient, createOrderReq);
    }

    private TrainReservation getPayload(TOrderInfo orderInfo) {
        return getPayload(orderInfo.getService(0).getServiceInfo().getPayload());
    }

    private TrainReservation getPayload(TJson payloadJson) {
        return ProtoUtils.fromTJson(payloadJson, TrainReservation.class);
    }

    @Override
    protected EOrderType orderTypeForOrderReq() {
        return EOrderType.OT_TRAIN;
    }

    @Override
    protected TCreateServiceReq.Builder getServiceForPayload(String payloadJson) {
        return TCreateServiceReq.newBuilder()
                .setServiceType(EServiceType.PT_TRAIN)
                .setSourcePayload(TJson.newBuilder().setValue(payloadJson).build());
    }

    private UpdateBlanksResponse updateBlanksResponse() {
        var response = new UpdateBlanksResponse();

        var blanks = new ArrayList<RailwayBlankInfo>();
        {
            var blank = new RailwayBlankInfo();
            blank.setOrderItemBlankId(20000001);
            blank.setNumber("123456");
            blank.setBlankStatus(ImBlankStatus.REFUNDED);
            blank.setPendingElectronicRegistration(PendingElectronicRegistration.NO_VALUE);
            blanks.add(blank);
        }
        response.setBlanks(blanks);
        response.setModified(false);

        return response;
    }

    @TestConfiguration
    static class IntegrationTestConfiguration {
        @Bean
        @Primary
        public TrustClient trainTrustClient() {
            return new MockTrustClient();
        }

        @Bean
        @Primary
        public TrustClientProvider testTrustClientProvider(@Autowired TrustClient trainTrustClient) {
            return paymentProfile -> trainTrustClient;
        }

        @Bean
        @Primary
        public TrainTariffInfoDataProvider trainTariffInfoDataProvider() {
            var tariffInfos = Arrays.asList(
                    TTrainTariffInfo.newBuilder().setId(1)
                            .setCode("full")
                            .setTitleRu("Полный")
                            .setImRequestCode("Full")
                            .setImResponseCodes("Full")
                            .setWithoutPlace(false)
                            .setMinAge(0)
                            .setMinAgeIncludesBirthday(false)
                            .setMaxAge(150)
                            .setMaxAgeIncludesBirthday(false)
                            .setNeedDocument(false).build(),

                    TTrainTariffInfo.newBuilder().setId(2)
                            .setCode("senior")
                            .setTitleRu("Сеньор")
                            .setImRequestCode("Senior")
                            .setImResponseCodes("Senior")
                            .setWithoutPlace(false)
                            .setMinAge(0)
                            .setMinAgeIncludesBirthday(false)
                            .setMaxAge(150)
                            .setMaxAgeIncludesBirthday(false)
                            .setNeedDocument(false).build()
            );

            TrainTariffInfoDataProviderProperties config = new TrainTariffInfoDataProviderProperties();
            config.setTablePath("tablePath");
            config.setIndexPath("./train-tariff-index");
            config.setProxy(new ArrayList<>());

            TestLuceneIndexBuilder<TTrainTariffInfo> luceneIndexBuilder = new TestLuceneIndexBuilder<TTrainTariffInfo>()
                    .setLuceneData(tariffInfos);

            return new TrainTariffInfoService(config, luceneIndexBuilder);
        }
    }
}
