package ru.yandex.travel.orders.integration.train;

import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;

import javax.persistence.EntityManager;

import io.grpc.Context;
import io.grpc.testing.GrpcCleanupRule;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import org.junit.After;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Rule;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.stubbing.Answer;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.jdbc.AutoConfigureTestDatabase;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.context.TestConfiguration;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.context.annotation.Primary;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.credentials.UserCredentialsBuilder;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.grpc.OrdersService;
import ru.yandex.travel.orders.integration.IntegrationUtils;
import ru.yandex.travel.orders.integration.train.factories.ImOrderInfoResponseFactory;
import ru.yandex.travel.orders.proto.ERefundPartState;
import ru.yandex.travel.orders.proto.ERefundPartType;
import ru.yandex.travel.orders.proto.OrderInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.TCalculateRefundReqV2;
import ru.yandex.travel.orders.proto.TDownloadBlankToken;
import ru.yandex.travel.orders.proto.TGetOrderInfoReq;
import ru.yandex.travel.orders.proto.TRefundPartContext;
import ru.yandex.travel.orders.proto.TRefundPartInfo;
import ru.yandex.travel.orders.proto.TStartRefundReq;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.repository.migrations.TrainOrderMigrationRepository;
import ru.yandex.travel.orders.repository.migrations.TrainToGenericMigrationRepository;
import ru.yandex.travel.orders.services.UrlShortenerService;
import ru.yandex.travel.orders.services.migrations.generic.TrainToGenericMigrationProcessor;
import ru.yandex.travel.orders.services.orders.RefundPartsService;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.promo.UserOrderCounterService;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.train.model.TrainTicketRefundStatus;
import ru.yandex.travel.train.partners.im.ImClient;
import ru.yandex.travel.train.partners.im.model.AutoReturnRequest;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationStatus;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOperationType;
import ru.yandex.travel.train.partners.im.model.orderinfo.ImOrderItemType;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderInfoResponse;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderItemResponse;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyInt;
import static org.mockito.Mockito.when;
import static ru.yandex.travel.orders.integration.IntegrationUtils.waitForPredicateOrTimeout;

@RunWith(SpringRunner.class)
@SpringBootTest(
        webEnvironment = SpringBootTest.WebEnvironment.NONE,
        properties = {
                "workflow-processing.pending-workflow-polling-interval=100ms",
                "trust-hotels.clearing-refresh-timeout=1s",
                "trust-hotels.payment-refresh-timeout=1s",
                "trust-hotels.trains-new-processing-enabled=true",
                "single-node.auto-start=true",
                "train-workflow.check-ticket-refund-delay=1s",
                "train-workflow.check-ticket-refund-task-period=200ms",
                "train-workflow.check-ticket-refund-max-tries=50",
                "train-workflow.check-passenger-discounts-enabled=true",
                "train-workflow.check-expiration-task-period=100ms",
                "migration.train-to-generic-task.enabled=true",
                "migration.train-to-generic-task.initial-start-delay=10ms",
        }
)
@AutoConfigureTestDatabase(replace = AutoConfigureTestDatabase.Replace.ANY)
@DirtiesContext
@ActiveProfiles("test")
@Import(TrainOrderFlowTests.IntegrationTestConfiguration.class)
@Slf4j
@Ignore
public class TrainOrderMigrationFlowTests {
    private static final String SESSION_KEY = "qwerty";
    private static final String YANDEX_UID = "1234567890";

    @MockBean
    protected UserOrderCounterService counterService;
    private static final AtomicBoolean MIGRATION_ENABLED = new AtomicBoolean(false);

    @Rule
    public GrpcCleanupRule cleanupRule = new GrpcCleanupRule();

    @Autowired
    private OrdersService ordersService;

    private final UserCredentialsBuilder userCredentialsBuilder = new UserCredentialsBuilder();

    @MockBean
    private ImClient imClient;

    @MockBean
    private UrlShortenerService urlShortenerService;

    private Context context;

    private OrderInterfaceV1Grpc.OrderInterfaceV1BlockingStub client;

    @Autowired
    private TransactionTemplate transactionTemplate;

    @Autowired
    private OrderRepository orderRepository;

    @Autowired
    private TrustClient trustClient;

    @Autowired
    private EntityManager em;

    @Before
    public void setUpCredentialsContext() {
        UserCredentials credentials = userCredentialsBuilder.build(SESSION_KEY, YANDEX_UID, "passport-id-1", "user1",
                null, "127.0.0.1", false, false);
        context = Context.current().withValue(UserCredentials.KEY, credentials).attach();
        client = IntegrationUtils.createServerAndBlockingStub(cleanupRule, ordersService);
    }

    @After
    public void tearDownCredentialsContext() {
        Context.current().detach(context);
    }

    // the first part of the test is from TrainOrderFlowTests and the second one is from GenericTrainOrderFlowTests
    @Test
    public void testMigrateConfirmedOrderAndRefund() {
        AtomicReference<AutoReturnRequest> autoReturnRequestRef = new AtomicReference<>();
        AtomicReference<OrderInfoResponse> orderInfoResponseRef = new AtomicReference<>();

        var nonRefundedOrderInfoResponse = new ImOrderInfoResponseFactory(ImOperationStatus.OK, null, 1,
                null, false, LocalDateTime.now().plusDays(1)).create();
        orderInfoResponseRef.set(nonRefundedOrderInfoResponse);
        when(imClient.orderInfo(anyInt())).thenAnswer((Answer<OrderInfoResponse>) invocation -> orderInfoResponseRef.get());
        when(imClient.orderInfo(anyInt(), any())).thenAnswer((Answer<OrderInfoResponse>) invocation -> orderInfoResponseRef.get());

        var orderId = TrainOrderFlowTests.createSuccessfulOrder(imClient, client, urlShortenerService,
                transactionTemplate, orderRepository, trustClient,
                new TrainOrderFlowTests().createOrderRequest(1).build());
        var autoReturnRsp = TrainOrderFlowTests.createAutoReturnResponse();
        when(imClient.autoReturn(any())).thenAnswer(invocation -> {
            autoReturnRequestRef.set(invocation.getArgument(0));
            return autoReturnRsp;
        });

        IndexingRecords changesBeforeMigration = countIndexRecords(UUID.fromString(orderId));
        // for manual local tests only, db queries with jsonb won't work with h2, the refund test part should also fail
        /*UUID refundId = transactionTemplate.execute(status -> {
            Order order = em.getReference(Order.class, UUID.fromString(orderId));
            TrainOrderUserRefund refund = TrainOrderUserRefund.createForOrder(order);
            refund.setTicketRefunds(List.of(new TrainTicketRefund()));
            em.persist(refund);
            TrainTicketRefund ticketRefund = TrainTicketRefund.createRefund(
                    OrderCompatibilityUtils.getOnlyTrainOrderItem(order), List.of(), refund);
            em.persist(ticketRefund);
            return refund.getId();
        });*/

        MIGRATION_ENABLED.set(true);
        IntegrationUtils.waitForPredicateOrTimeout(() -> transactionTemplate.execute(status ->
                        orderRepository.getOne(UUID.fromString(orderId)).getPublicType() == EOrderType.OT_GENERIC
                ) == Boolean.TRUE,
                Duration.ofSeconds(10), "The legacy train order should be MIGRATED to the new flow");

        IndexingRecords changesAfterMigration = countIndexRecords(UUID.fromString(orderId));
        // the task processor updates the order and its workflow, both of which cause new OrderChange entities creation
        // unlike aggregate state refresher, order changes task processor isn't enabled here, so the numbers are stable
        assertThat(changesAfterMigration.orderChange).isEqualTo(changesBeforeMigration.orderChange + 2);
        /*transactionTemplate.execute(status -> {
            GenericOrderUserRefund refund = em.getReference(GenericOrderUserRefund.class, refundId);
            assertThat(refund).isNotNull();
            assertThat(refund.getState()).isEqualTo(EOrderRefundState.RS_NEW);
            assertThat(refund.getTicketRefunds()).hasSize(1);
            //assertThat(refund.getPayload()).isNotNull();
            return null;
        });*/

        var orderInfoRsp = client.getOrderInfo(TGetOrderInfoReq.newBuilder().setOrderId(orderId)
                .setUpdateOrderOnTheFly(true).build());
        assertThat(orderInfoRsp.getResult().getOrderType()).isEqualTo(EOrderType.OT_GENERIC);

        var calculateRefundRsp = client.calculateRefundV2(TCalculateRefundReqV2.newBuilder().setOrderId(orderId)
                .addContext(RefundPartsService.partContextToString(TRefundPartContext.newBuilder()
                        .setServiceId(orderInfoRsp.getResult().getService(0).getServiceId())
                        .setType(ERefundPartType.RPT_SERVICE)
                        .build()))
                .build());
        assertThat(calculateRefundRsp.getRefundAmount().getAmount()).isPositive();
        assertThat(calculateRefundRsp.getPenaltyAmount().getAmount()).isPositive();
        assertThat(calculateRefundRsp.getExpiresAt().getSeconds()).isGreaterThan(Instant.now().getEpochSecond());

        //noinspection ResultOfMethodCallIgnored
        client.startRefund(TStartRefundReq.newBuilder().setOrderId(orderId)
                .setRefundToken(calculateRefundRsp.getRefundToken()).build());
        waitForPredicateOrTimeout(client, orderId,
                rsp1 -> TrainOrderFlowTests.getTrainReservation(rsp1).getPassengers().get(0)
                        .getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDING &&
                        autoReturnRequestRef.get() != null,
                Duration.ofSeconds(10), "Ticket must be REFUNDING");

        orderInfoResponseRef.set(createImOrderInfoResponseWithRefund(
                autoReturnRequestRef.get().getServiceAutoReturnRequest().getAgentReferenceId(), 1, false, false));

        orderInfoRsp = waitForPredicateOrTimeout(client, orderId,
                rsp1 -> rsp1.getResult().getGenericOrderState() == EOrderState.OS_REFUNDED &&
                        TrainOrderFlowTests.getTrainReservation(rsp1).getPassengers().get(0)
                                .getTicket().getRefundStatus() == TrainTicketRefundStatus.REFUNDED,
                Duration.ofSeconds(10), "Ticket must be REFUNDED");
        TRefundPartInfo refundedTicketPartInfo = orderInfoRsp.getResult().getRefundPartsList().stream()
                .filter(x -> x.getState() == ERefundPartState.RPS_REFUNDED && x.getType() == ERefundPartType.RPT_SERVICE_PART)
                .findFirst().orElseThrow();
        assertThat(refundedTicketPartInfo.hasRefund()).isTrue();
        assertThat(refundedTicketPartInfo.getRefund().getRefundBlankToken().getOneOfDownloadBlankParamsCase())
                .isEqualByComparingTo(TDownloadBlankToken.OneOfDownloadBlankParamsCase.TRAINDOWNLOADBLANKPARAMS);
        assertThat(refundedTicketPartInfo.getRefund().getRefundAmount().getAmount()).isPositive();
    }

    static OrderInfoResponse createImOrderInfoResponseWithRefund(String refundReferenceId,
                                                                 int passengers,
                                                                 boolean refundIsExternallyLoaded,
                                                                 boolean withInsurance) {
        var factory = new ImOrderInfoResponseFactory();
        factory.setPassengers(passengers);
        factory.setRefundReferenceId(refundReferenceId);
        factory.setBuyInsuranceStatus(withInsurance ? ImOperationStatus.OK : null);
        factory.setRefundIsExternallyLoaded(refundIsExternallyLoaded);
        var result = factory.create();
        if (withInsurance) {
            var buyInsuranceItem = result.findBuyInsuranceItems().get(0);
            OrderItemResponse refundInsuranceItem = new OrderItemResponse();
            refundInsuranceItem.setPreviousOrderItemId(buyInsuranceItem.getOrderItemId());
            refundInsuranceItem.setOrderItemId(buyInsuranceItem.getOrderItemId() + 222);
            refundInsuranceItem.setOperationType(ImOperationType.REFUND);
            refundInsuranceItem.setType(ImOrderItemType.INSURANCE);
            refundInsuranceItem.setSimpleOperationStatus(ImOperationStatus.OK);
            refundInsuranceItem.setIsExternallyLoaded(refundIsExternallyLoaded);
            result.getOrderItems().add(refundInsuranceItem);
        }
        return result;
    }

    private IndexingRecords countIndexRecords(UUID orderId) {
        return transactionTemplate.execute(status -> {
            Number orderChangeRecords = em.createQuery(
                            "select count(*) from OrderChange ch where orderId = :id", Number.class)
                    .setParameter("id", orderId)
                    .getSingleResult();
            Number aggregateStateChangeRecords = em.createQuery(
                            "select count(*) from OrderAggregateStateChange where orderId = :id", Number.class)
                    .setParameter("id", orderId)
                    .getSingleResult();
            return new IndexingRecords(orderChangeRecords.intValue(), aggregateStateChangeRecords.intValue());
        });
    }

    @Value
    private static class IndexingRecords {
        private final int orderChange;
        private final int aggregateStateChange;
    }

    @TestConfiguration
    static class TestBeans {
        @Bean
        @Primary
        public TrainToGenericMigrationProcessor mockedMigrationProcessor(
                EntityManager em,
                TrainToGenericMigrationRepository migrationRepository,
                TrainOrderMigrationRepository logRepository
        ) {
            return new TrainToGenericMigrationProcessor(em, migrationRepository, logRepository) {
                @Override
                public void migrate(UUID orderId) {
                    if (MIGRATION_ENABLED.get()) {
                        log.info("migrationProcessor is ENABLED");
                        super.migrate(orderId);
                    } else {
                        log.info("migrationProcessor is disabled");
                    }
                }
            };
        }
    }
}
