package ru.yandex.travel.orders.integration.voucher;

import java.time.Instant;
import java.time.LocalDate;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicReference;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.bus.model.BusPointInfo;
import ru.yandex.travel.bus.model.BusReservation;
import ru.yandex.travel.bus.model.BusRide;
import ru.yandex.travel.buses.backend.proto.EPointKeyType;
import ru.yandex.travel.hotels.common.orders.ExpediaHotelItinerary;
import ru.yandex.travel.hotels.common.orders.OrderDetails;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.entities.BusOrderItem;
import ru.yandex.travel.orders.entities.ExpediaOrderItem;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.Voucher;
import ru.yandex.travel.orders.entities.WellKnownWorkflow;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.repository.VoucherRepository;
import ru.yandex.travel.orders.services.pdfgenerator.PdfGeneratorService;
import ru.yandex.travel.orders.services.pdfgenerator.PdfNotFoundException;
import ru.yandex.travel.orders.services.pdfgenerator.model.PdfStateResponse;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.voucher.proto.EVoucherState;
import ru.yandex.travel.orders.workflow.voucher.proto.EVoucherType;
import ru.yandex.travel.orders.workflow.voucher.proto.TGenerateVoucher;
import ru.yandex.travel.testing.TestUtils;
import ru.yandex.travel.workflow.WorkflowMessageSender;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.argThat;
import static org.mockito.Mockito.atLeastOnce;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@RunWith(SpringRunner.class)
@SpringBootTest(
        webEnvironment = SpringBootTest.WebEnvironment.NONE,
        properties = {
                "workflow-processing.pending-workflow-polling-interval=100ms",
                "single-node.auto-start=true",
                "voucher-workflow.check-created-period=100ms",
                "voucher-workflow.check-created-timeout=10s",
                "voucher-workflow.check-state-task.initial-start-delay=100ms",
                "voucher-workflow.check-state-task.schedule-rate=100ms",
        }
)
@ActiveProfiles("test")
public class VoucherFlowTest {
    @Autowired
    private OrderRepository orderRepository;

    @Autowired
    private WorkflowRepository workflowRepository;

    @Autowired
    private TransactionTemplate transactionTemplate;

    @Autowired
    private WorkflowMessageSender messageSender;

    @Autowired
    private VoucherRepository voucherRepository;

    @MockBean
    private PdfGeneratorService pdfGeneratorService;

    @Test
    public void testBusTicket() {
        AtomicReference<String> voucherFileName = new AtomicReference<>();
        AtomicBoolean voucherStateCreated = new AtomicBoolean(false);
        when(pdfGeneratorService.getState(any())).thenAnswer(rq -> {
            if (voucherStateCreated.get() && rq.getArgument(0, String.class).equals(voucherFileName.get())) {
                return new PdfStateResponse("pdfgenerator/bus/ticket", Instant.now());
            }
            throw new PdfNotFoundException("not found");
        });
        UUID voucherId = transactionTemplate.execute(i -> {
            var order = createBusOrder();
            Voucher voucher = Voucher.createForOrder(order);
            voucher = voucherRepository.saveAndFlush(voucher);
            Workflow voucherWorkflow = Workflow.createWorkflowForEntity(voucher,
                    WellKnownWorkflow.GENERIC_ERROR_SUPERVISOR.getUuid());
            voucherWorkflow = workflowRepository.saveAndFlush(voucherWorkflow);
            messageSender.scheduleEvent(voucherWorkflow.getId(), TGenerateVoucher.newBuilder().build());
            voucherFileName.set(voucher.getVoucherName());
            return voucher.getId();
        });

        TestUtils.waitForState("Voucher CREATING", () -> transactionTemplate.execute(
                s -> voucherRepository.getOne(voucherId).getState() == EVoucherState.VS_CREATING));
        verify(pdfGeneratorService).generateBusesTickets(argThat(rq -> rq.getFileName().equals(voucherFileName.get())));
        voucherStateCreated.set(true);

        TestUtils.waitForState("Voucher CREATED", () -> transactionTemplate.execute(
                s -> voucherRepository.getOne(voucherId).getState() == EVoucherState.VS_CREATED));

        transactionTemplate.execute(i -> {
            Voucher voucher = voucherRepository.getOne(voucherId);
            assertThat(voucher.getVoucherUrl()).isEqualTo("pdfgenerator/bus/ticket");
            assertThat(voucher.getNextCheckCreatedAt()).isNull();
            return null;
        });
    }

    @Test
    public void testHotelVoucher() {
        AtomicReference<String> voucherFileName = new AtomicReference<>();
        AtomicBoolean voucherStateCreated = new AtomicBoolean(false);
        when(pdfGeneratorService.getState(any())).thenAnswer(rq -> {
            if (voucherStateCreated.get() && rq.getArgument(0, String.class).equals(voucherFileName.get())) {
                return new PdfStateResponse("pdfgenerator/hotels/voucher", Instant.now());
            }
            throw new PdfNotFoundException("not found");
        });
        UUID voucherId = transactionTemplate.execute(i -> {
            var order = createHotelOrder();
            Voucher voucher = Voucher.createForOrder(order);
            voucher = voucherRepository.saveAndFlush(voucher);
            Workflow voucherWorkflow = Workflow.createWorkflowForEntity(voucher,
                    WellKnownWorkflow.GENERIC_ERROR_SUPERVISOR.getUuid());
            voucherWorkflow = workflowRepository.saveAndFlush(voucherWorkflow);
            messageSender.scheduleEvent(voucherWorkflow.getId(), TGenerateVoucher.newBuilder().build());
            voucherFileName.set(voucher.getVoucherName());
            return voucher.getId();
        });

        TestUtils.waitForState("Voucher CREATING", () -> transactionTemplate.execute(
                s -> voucherRepository.getOne(voucherId).getState() == EVoucherState.VS_CREATING));
        verify(pdfGeneratorService, atLeastOnce()).generateHotelsVoucher(argThat(rq -> rq.getFileName().equals(voucherFileName.get())));
        voucherStateCreated.set(true);

        TestUtils.waitForState("Voucher CREATED", () -> transactionTemplate.execute(
                s -> voucherRepository.getOne(voucherId).getState() == EVoucherState.VS_CREATED));

        transactionTemplate.execute(i -> {
            Voucher voucher = voucherRepository.getOne(voucherId);
            assertThat(voucher.getVoucherUrl()).isEqualTo("pdfgenerator/hotels/voucher");
            assertThat(voucher.getNextCheckCreatedAt()).isNull();
            return null;
        });
    }

    @Test
    public void testHotelBusinessTripDoc() {
        AtomicReference<String> voucherFileName = new AtomicReference<>();
        AtomicBoolean voucherStateCreated = new AtomicBoolean(false);
        when(pdfGeneratorService.getState(any())).thenAnswer(rq -> {
            if (voucherStateCreated.get() && rq.getArgument(0, String.class).equals(voucherFileName.get())) {
                return new PdfStateResponse("pdfgenerator/hotels/businesstrip", Instant.now());
            }
            throw new PdfNotFoundException("not found");
        });
        UUID voucherId = transactionTemplate.execute(i -> {
            var order = createHotelOrder();
            String fileName = "hotels/business_trip.pdf";
            Voucher tripDoc = Voucher.createDocument(EVoucherType.VT_HOTELS_BUSINESS_TRIP, fileName, order);
            tripDoc = voucherRepository.saveAndFlush(tripDoc);
            Workflow voucherWorkflow = Workflow.createWorkflowForEntity(tripDoc,
                    WellKnownWorkflow.GENERIC_ERROR_SUPERVISOR.getUuid());
            voucherWorkflow = workflowRepository.saveAndFlush(voucherWorkflow);
            messageSender.scheduleEvent(voucherWorkflow.getId(), TGenerateVoucher.newBuilder().build());
            voucherFileName.set(tripDoc.getVoucherName());
            return tripDoc.getId();
        });

        TestUtils.waitForState("Voucher CREATING", () -> transactionTemplate.execute(
                s -> voucherRepository.getOne(voucherId).getState() == EVoucherState.VS_CREATING));
        verify(pdfGeneratorService).generateHotelsBusinessTripDoc(argThat(rq -> rq.getFileName().equals(voucherFileName.get())));
        voucherStateCreated.set(true);

        TestUtils.waitForState("Voucher CREATED", () -> transactionTemplate.execute(
                s -> voucherRepository.getOne(voucherId).getState() == EVoucherState.VS_CREATED));

        transactionTemplate.execute(i -> {
            Voucher voucher = voucherRepository.getOne(voucherId);
            assertThat(voucher.getVoucherUrl()).isEqualTo("pdfgenerator/hotels/businesstrip");
            assertThat(voucher.getNextCheckCreatedAt()).isNull();
            return null;
        });
    }

    private GenericOrder createBusOrder() {
        var order = new GenericOrder();
        order.setState(EOrderState.OS_CONFIRMED);
        order.setDisplayType(EDisplayOrderType.DT_BUS);
        order.setId(UUID.randomUUID());
        var item = new BusOrderItem();
        item.setReservation(new BusReservation());
        var ride = new BusRide();
        item.getReservation().setRide(ride);
        ride.setPointFrom(new BusPointInfo());
        ride.getPointFrom().setId(213);
        ride.getPointFrom().setType(EPointKeyType.POINT_KEY_TYPE_SETTLEMENT);
        ride.getPointFrom().setTimezone("Europe/Moscow");
        ride.setPointTo(new BusPointInfo());
        ride.getPointTo().setId(2);
        ride.getPointTo().setType(EPointKeyType.POINT_KEY_TYPE_SETTLEMENT);
        ride.getPointTo().setTimezone("Europe/Moscow");
        order.addOrderItem(item);
        Workflow orderWorkflow = Workflow.createWorkflowForEntity(order);
        orderWorkflow.setSupervisorId(WellKnownWorkflow.ORDER_SUPERVISOR.getUuid());
        workflowRepository.saveAndFlush(orderWorkflow);
        order = orderRepository.saveAndFlush(order);
        return order;
    }

    private HotelOrder createHotelOrder() {
        var order = new HotelOrder();
        order.setState(EHotelOrderState.OS_CONFIRMED);
        order.setDisplayType(EDisplayOrderType.DT_HOTEL);
        order.setId(UUID.randomUUID());
        var orderItem = new ExpediaOrderItem();
        orderItem.setItinerary(new ExpediaHotelItinerary());
        orderItem.getItinerary().setOrderDetails(OrderDetails.builder()
                .checkinDate(LocalDate.now())
                .build());
        order.addOrderItem(orderItem);
        Workflow orderWorkflow = Workflow.createWorkflowForEntity(order);
        orderWorkflow.setSupervisorId(WellKnownWorkflow.ORDER_SUPERVISOR.getUuid());
        workflowRepository.saveAndFlush(orderWorkflow);
        order = orderRepository.saveAndFlush(order);
        return order;
    }
}
