package ru.yandex.travel.orders.repository;

import java.time.Instant;
import java.util.List;
import java.util.Set;

import com.google.common.base.Strings;
import com.google.common.collect.Sets;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.orm.jpa.DataJpaTest;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;

import ru.yandex.travel.orders.entities.finances.ProcessingTasksInfo;
import ru.yandex.travel.orders.entities.partners.BillingPartnerConfig;

import static org.assertj.core.api.Assertions.assertThat;
import static ru.yandex.travel.orders.repository.BillingPartnerConfigRepository.NO_EXCLUDE_IDS;

@RunWith(SpringRunner.class)
@DataJpaTest
@ActiveProfiles("test")
public class BillingPartnerConfigRepositoryTest {
    @Autowired
    private BillingPartnerConfigRepository repository;

    @Before
    public void init() {
        repository.deleteAll();
    }

    @Test
    public void agreementForSynchronization_filters() {
        BillingPartnerConfig c1 = repository.save(config(1, "2020-01-24T15:34:00Z"));
        BillingPartnerConfig c2 = repository.save(config(2, "2020-01-24T15:24:00Z"));
        BillingPartnerConfig c3 = repository.save(config(3, "2020-01-24T15:14:00Z"));

        Instant noDtLimit = Instant.parse("3000-01-01T00:00:00Z");
        Pageable noPaging = PageRequest.of(0, 10, Sort.Direction.ASC, "billingClientId");
        assertThat(repository.countIdsForAgreementSynchronization(noDtLimit, NO_EXCLUDE_IDS)).isEqualTo(3);
        assertThat(Sets.newHashSet(repository.findIdsForAgreementSynchronization(noDtLimit, NO_EXCLUDE_IDS, noPaging)))
                .isEqualTo(Set.of(c1.getBillingClientId(), c2.getBillingClientId(), c3.getBillingClientId()));
        ProcessingTasksInfo stats1 = repository.findOldestTimestampForAgreementSynchronization(noDtLimit);
        assertThat(stats1.getOldestProcessAt()).isEqualTo("2020-01-24T15:14:00Z");
        assertThat(stats1.getCount()).isEqualTo(3);

        Instant dtLimit = Instant.parse("2020-01-24T15:30:00Z");
        assertThat(repository.countIdsForAgreementSynchronization(dtLimit, NO_EXCLUDE_IDS)).isEqualTo(2);
        assertThat(Sets.newHashSet(repository.findIdsForAgreementSynchronization(dtLimit, NO_EXCLUDE_IDS, noPaging)))
                .isEqualTo(Set.of(c2.getBillingClientId(), c3.getBillingClientId()));
        ProcessingTasksInfo stats2 = repository.findOldestTimestampForAgreementSynchronization(dtLimit);
        assertThat(stats2.getOldestProcessAt()).isEqualTo("2020-01-24T15:14:00Z");
        assertThat(stats2.getCount()).isEqualTo(2);

        Set<Long> excludeIds = Set.of(c3.getBillingClientId());
        assertThat(repository.countIdsForAgreementSynchronization(noDtLimit, excludeIds)).isEqualTo(2);
        assertThat(Sets.newHashSet(repository.findIdsForAgreementSynchronization(noDtLimit, excludeIds, noPaging)))
                .isEqualTo(Set.of(c1.getBillingClientId(), c2.getBillingClientId()));

        ProcessingTasksInfo stats3 = repository.findOldestTimestampForAgreementSynchronization(
                Instant.parse("2000-01-01T00:00:00Z"));
        assertThat(stats3.getOldestProcessAt()).isNull();
        assertThat(stats3.getCount()).isEqualTo(0);

        c3.setSynchronizeAgreement(false);
        repository.save(c3);

        assertThat(repository.countIdsForAgreementSynchronization(noDtLimit, NO_EXCLUDE_IDS)).isEqualTo(2);
        assertThat(Sets.newHashSet(repository.findIdsForAgreementSynchronization(noDtLimit, NO_EXCLUDE_IDS, noPaging)))
                .isEqualTo(Set.of(c1.getBillingClientId(), c2.getBillingClientId()));
        assertThat(repository.findOldestTimestampForAgreementSynchronization(noDtLimit).getOldestProcessAt())
                .isEqualTo("2020-01-24T15:24:00Z");
    }

    @Test
    public void agreementForSynchronization_nullSynchronizedAt() {
        BillingPartnerConfig c1 = repository.save(config(1, "2020-01-24T15:34:00Z"));
        BillingPartnerConfig c2 = repository.save(config(2, null));
        BillingPartnerConfig c3 = repository.save(config(3, "2020-05-28T12:34:00Z"));

        Instant noDtLimit = Instant.parse("3000-01-01T00:00:00Z");
        Pageable paging = PageRequest.of(0, 10);
        assertThat(repository.countIdsForAgreementSynchronization(noDtLimit, NO_EXCLUDE_IDS)).isEqualTo(3);
        assertThat(repository.findIdsForAgreementSynchronization(noDtLimit, NO_EXCLUDE_IDS, paging))
                .isEqualTo(List.of(c2.getBillingClientId(), c1.getBillingClientId(), c3.getBillingClientId()));
        assertThat(repository.findOldestTimestampForAgreementSynchronization(noDtLimit).getOldestProcessAt())
                // the missing synchronized_at is replaced with the created_at value
                .isEqualTo("2000-01-01T00:00:00Z");
    }

    private BillingPartnerConfig config(long id, String synchronizedAt) {
        return BillingPartnerConfig.builder()
                .billingClientId(id)
                .synchronizeAgreement(true)
                .createdAt(Instant.parse("2000-01-01T00:00:00Z"))
                .synchronizedAt(!Strings.isNullOrEmpty(synchronizedAt) ? Instant.parse(synchronizedAt) : null)
                .build();
    }
}
