package ru.yandex.travel.orders.repository;

import java.time.Duration;
import java.time.Instant;
import java.util.List;

import javax.persistence.EntityManager;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.orm.jpa.DataJpaTest;
import org.springframework.dao.IncorrectResultSizeDataAccessException;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;

import ru.yandex.travel.orders.entities.finances.BillingTransaction;
import ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentSystemType;
import ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentType;
import ru.yandex.travel.orders.entities.finances.BillingTransactionType;
import ru.yandex.travel.orders.entities.finances.FinancialEvent;
import ru.yandex.travel.orders.entities.finances.ProcessingTasksInfo;
import ru.yandex.travel.orders.entities.partners.BillingPartnerConfig;

import static java.util.stream.Collectors.toList;
import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.Assertions.assertThatThrownBy;
import static ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentSystemType.PROMO_CODE;
import static ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentSystemType.YANDEX_MONEY;
import static ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentType.COST;
import static ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentType.REWARD;
import static ru.yandex.travel.orders.entities.finances.BillingTransactionType.PAYMENT;
import static ru.yandex.travel.orders.entities.finances.BillingTransactionType.REFUND;
import static ru.yandex.travel.orders.repository.BillingTransactionRepository.NO_EXCLUDE_IDS;

@RunWith(SpringRunner.class)
@DataJpaTest
@ActiveProfiles("test")
public class BillingTransactionRepositoryTest {
    private static final long TEST_PARTNER_ID = 1253138269424223452L;

    @Autowired
    private EntityManager em;
    @Autowired
    private BillingTransactionRepository txRepository;
    @Autowired
    private FinancialEventRepository eventsRepository;

    @Test
    public void findBySourceFinancialEventAndPaymentType() {
        FinancialEvent e1 = eventsRepository.save(new FinancialEvent());
        FinancialEvent e2 = eventsRepository.save(new FinancialEvent());
        BillingTransaction tx1 = txRepository.save(txWithSourceEvent(e1, COST, YANDEX_MONEY));
        BillingTransaction tx2 = txRepository.save(txWithSourceEvent(e2, COST, YANDEX_MONEY));
        BillingTransaction tx3 = txRepository.save(txWithSourceEvent(e2, REWARD, YANDEX_MONEY));
        BillingTransaction tx4 = txRepository.save(txWithSourceEvent(e2, COST, PROMO_CODE));

        assertThat(findSourceTx(e1, COST, YANDEX_MONEY)).isEqualTo(tx1);
        assertThat(findSourceTx(e1, REWARD, YANDEX_MONEY)).isNull();
        assertThat(findSourceTx(e2, COST, YANDEX_MONEY)).isEqualTo(tx2);
        assertThat(findSourceTx(e2, REWARD, YANDEX_MONEY)).isEqualTo(tx3);
        assertThat(findSourceTx(e2, COST, PROMO_CODE)).isEqualTo(tx4);
        assertThat(findSourceTx(e2, REWARD, PROMO_CODE)).isNull();

        // just to check that something should break
        txRepository.save(txWithSourceEvent(e2, REWARD, YANDEX_MONEY));
        assertThatThrownBy(() -> findSourceTx(e2, REWARD, YANDEX_MONEY))
                .isExactlyInstanceOf(IncorrectResultSizeDataAccessException.class);
    }

    private BillingTransaction findSourceTx(FinancialEvent event, BillingTransactionPaymentType cost,
                                            BillingTransactionPaymentSystemType yandexMoney) {
        return txRepository.findBySourceFinancialEventAndPaymentTypeAndPaymentSystemType(event, cost, yandexMoney);
    }

    @Test
    public void countReadyTransactionsWithoutYtId_ytIdFilter() {
        em.persist(partnerConfig());
        txRepository.save(txForYtIdGenerator("2019-12-11T12:34:56.78Z", PAYMENT));
        txRepository.save(txForYtIdGenerator("2019-12-11T12:34:56.79Z", PAYMENT));

        Instant noDtLimit = Instant.parse("2022-01-01T01:01:01Z");

        assertThat(txRepository.countReadyTransactionsWithoutYtId(noDtLimit)).isEqualTo(2);

        BillingTransaction tx1 = txRepository.findAll().get(0);
        tx1.setYtId(1L);
        txRepository.save(tx1);

        assertThat(txRepository.countReadyTransactionsWithoutYtId(noDtLimit)).isEqualTo(1);
    }

    @Test
    public void countReadyTransactionsWithoutYtId_payoutAtFiltering() {
        em.persist(partnerConfig());
        txRepository.save(txForYtIdGenerator("2019-12-11T12:34:56.78Z", PAYMENT));
        txRepository.save(txForYtIdGenerator("2019-12-11T12:34:56.79Z", PAYMENT));
        txRepository.save(txForYtIdGenerator("2020-01-11T12:34:56.87Z", PAYMENT));

        Instant dt1 = Instant.parse("2022-01-01T01:01:01Z");
        assertThat(txRepository.countReadyTransactionsWithoutYtId(dt1)).isEqualTo(3);

        Instant dt2 = Instant.parse("2019-12-11T12:34:56.79Z");
        assertThat(txRepository.countReadyTransactionsWithoutYtId(dt2)).isEqualTo(1);

        Instant dt3 = Instant.parse("2010-12-11T12:34:56.99Z");
        assertThat(txRepository.countReadyTransactionsWithoutYtId(dt3)).isEqualTo(0);
    }

    @Test
    public void generateNewYtId_payoutAtFilter() {
        em.persist(partnerConfig());
        BillingTransaction tx1 = txRepository.save(txForYtIdGenerator("2019-12-11T12:34:56.78Z", PAYMENT));
        BillingTransaction tx2 = txRepository.save(txForYtIdGenerator("2019-12-11T12:34:56.79Z", PAYMENT));
        BillingTransaction tx3 = txRepository.save(txForYtIdGenerator("2020-01-11T12:34:56.87Z", PAYMENT));

        Instant dt1 = Instant.parse("2019-12-11T12:34:56.79Z");
        assertThat(txRepository.generateNewYtIdsForReadyTransactions(dt1)).isEqualTo(1);
        em.clear(); // dropping internal ORM caches after executing a native DML query
        assertThat(txRepository.getOne(tx1.getId()).getYtId()).isNotNull();
        assertThat(txRepository.getOne(tx2.getId()).getYtId()).isNull();
        assertThat(txRepository.getOne(tx3.getId()).getYtId()).isNull();

        Instant dt2 = Instant.parse("2029-12-11T12:34:56.79Z");
        assertThat(txRepository.generateNewYtIdsForReadyTransactions(dt2)).isEqualTo(2);
        em.clear(); // dropping internal ORM caches after executing a native DML query
        assertThat(txRepository.findAll()).allMatch(tx -> tx.getYtId() != null);
    }

    @Test
    public void generateNewYtId_ordering() {
        em.persist(partnerConfig());
        BillingTransaction tx1 = txRepository.save(txForYtIdGenerator("2019-12-24T21:00:00Z", PAYMENT));
        BillingTransaction tx2 = txRepository.save(txForYtIdGenerator("2019-12-24T21:00:00Z", REFUND));
        BillingTransaction tx3 = txRepository.save(txForYtIdGenerator("2018-06-21T21:00:00Z", PAYMENT));

        Instant dt1 = Instant.parse("2019-12-24T21:00:01Z");
        assertThat(txRepository.generateNewYtIdsForReadyTransactions(dt1)).isEqualTo(3);
        em.clear(); // dropping internal ORM caches after executing a native DML query

        List<BillingTransaction> updatedTxs = List.of(
                txRepository.getOne(tx1.getId()),
                txRepository.getOne(tx2.getId()),
                txRepository.getOne(tx3.getId())
        );
        assertThat(updatedTxs).allSatisfy(tx ->
                assertThat(tx.getYtId()).isNotNull());

        // the old record should be assigned with a new yt id before the other records: tx3 < tx1 < tx2
        // (this test doesn't work in h2, the generated ids order differs; common table expressions aren't materialized)
        //assertThat(updatedTxs.get(2).getYtId()).isLessThan(updatedTxs.get(0).getYtId());
        //assertThat(updatedTxs.get(0).getYtId()).isLessThan(updatedTxs.get(1).getYtId());
    }

    @Test
    public void findOldestTimestampReadyForYtIdGeneration() {
        em.persist(partnerConfig());
        List<BillingTransaction> txs = txRepository.saveAll(List.of(
                txForYtIdGenerator("2019-12-24T17:00:00Z", PAYMENT),
                txForYtIdGenerator("2019-12-24T18:00:00Z", PAYMENT),
                txForYtIdGenerator("2019-12-24T19:00:00Z", PAYMENT)
        ));

        Instant ts = Instant.parse("2019-12-24T18:35:00Z");
        ProcessingTasksInfo stats1 = txRepository.findOldestTimestampReadyForYtIdGeneration(ts);
        assertThat(stats1.getOldestProcessAt()).isEqualTo("2019-12-24T17:00:00Z");
        assertThat(stats1.getCount()).isEqualTo(2);

        txs.get(0).setYtId(-1L);
        txRepository.save(txs.get(0));
        ProcessingTasksInfo stats2 = txRepository.findOldestTimestampReadyForYtIdGeneration(ts);
        assertThat(stats2.getOldestProcessAt()).isEqualTo("2019-12-24T18:00:00Z");
        assertThat(stats2.getCount()).isEqualTo(1);

        txs.get(1).setYtId(-2L);
        txRepository.save(txs.get(1));
        ProcessingTasksInfo stats3 = txRepository.findOldestTimestampReadyForYtIdGeneration(ts);
        assertThat(stats3.getOldestProcessAt()).isEqualTo((Instant) null);
        assertThat(stats3.getCount()).isEqualTo(0);
    }

    @Test
    public void testGenerateYtIdMethods_disabledPartner() {
        BillingPartnerConfig config1 = partnerConfig(75152414124762164L);
        BillingPartnerConfig config2 = partnerConfig(98462317412742344L);
        em.persist(config1);
        em.persist(config2);
        txRepository.save(txForYtIdGenerator("2019-12-11T00:00:00Z", PAYMENT, config1.getBillingClientId()));
        txRepository.save(txForYtIdGenerator("2019-12-12T00:00:00Z", PAYMENT, config1.getBillingClientId()));
        txRepository.save(txForYtIdGenerator("2019-12-13T00:00:00Z", PAYMENT, config2.getBillingClientId()));
        List<Long> ids = txRepository.findAll().stream().map(BillingTransaction::getId).collect(toList());

        Instant dt1 = Instant.parse("2022-01-01T01:01:01Z");
        assertThat(txRepository.countReadyTransactionsWithoutYtId(dt1)).isEqualTo(3);
        assertThat(txRepository.findOldestTimestampReadyForYtIdGeneration(dt1).getOldestProcessAt())
                .isEqualTo(Instant.parse("2019-12-11T00:00:00Z"));

        config1.setExportToYt(false);
        em.persist(config1);

        assertThat(txRepository.countReadyTransactionsWithoutYtId(dt1)).isEqualTo(1);
        assertThat(txRepository.findOldestTimestampReadyForYtIdGeneration(dt1).getOldestProcessAt())
                .isEqualTo(Instant.parse("2019-12-13T00:00:00Z"));
        assertThat(txRepository.generateNewYtIdsForReadyTransactions(dt1)).isEqualTo(1);
        em.clear(); // dropping internal ORM caches after executing a native DML query
        assertThat(txRepository.getOne(ids.get(2)).getYtId()).isNotNull();
    }

    @Test
    public void testGenerateYtIdMethods_paused() {
        em.persist(partnerConfig());
        txRepository.save(txForYtIdGenerator("2019-12-11T00:00:00Z", PAYMENT));
        txRepository.save(txForYtIdGenerator("2019-12-12T00:00:00Z", PAYMENT));
        txRepository.save(txForYtIdGenerator("2019-12-13T00:00:00Z", PAYMENT));
        List<Long> ids = txRepository.findAll().stream().map(BillingTransaction::getId).collect(toList());

        Instant noDtLimit = Instant.parse("2100-01-01T01:01:01Z");
        assertThat(txRepository.countReadyTransactionsWithoutYtId(noDtLimit)).isEqualTo(3);
        assertThat(txRepository.findOldestTimestampReadyForYtIdGeneration(noDtLimit).getOldestProcessAt())
                .isEqualTo(Instant.parse("2019-12-11T00:00:00Z"));

        BillingTransaction tx1 = txRepository.getOne(ids.get(0));
        tx1.setPaused(true);
        txRepository.save(tx1);

        assertThat(txRepository.countReadyTransactionsWithoutYtId(noDtLimit)).isEqualTo(2);
        assertThat(txRepository.findOldestTimestampReadyForYtIdGeneration(noDtLimit).getOldestProcessAt())
                .isEqualTo(Instant.parse("2019-12-12T00:00:00Z"));
        assertThat(txRepository.generateNewYtIdsForReadyTransactions(noDtLimit)).isEqualTo(2);
        em.clear(); // dropping internal ORM caches after executing a native DML query
        assertThat(txRepository.findAllById(ids.subList(1, 3))).hasSize(2)
                .allMatch(tx -> tx.getYtId() != null);
        assertThat(txRepository.getOne(ids.get(0)).getYtId()).isNull();
    }

    @Test
    public void findByYtIdNotNullAndExportedToYtFalse_filters() {
        List<BillingTransaction> txs = txRepository.saveAll(List.of(
                txForYtExport(1L, true),
                txForYtExport(2L, false),
                txForYtExport(null, true), // shouldn't happen
                txForYtExport(null, false)
        ));

        Pageable defaultSorting = PageRequest.of(0, 10, Sort.by("ytId").ascending());
        assertThat(txRepository.countReadyForExport()).isEqualTo(1);
        assertThat(txRepository.findReadyForExport(defaultSorting)).hasSize(1)
                .first().isEqualTo(txs.get(1));

        txs.get(1).setExportedToYt(true);
        txs.get(3).setYtId(5L);
        txRepository.saveAll(txs);

        assertThat(txRepository.countReadyForExport()).isEqualTo(1);
        assertThat(txRepository.findReadyForExport(defaultSorting)).hasSize(1)
                .first().isEqualTo(txs.get(3));
    }

    @Test
    public void findByYtIdNotNullAndExportedToYtFalse_pagingAndOrdering() {
        List<BillingTransaction> txs = txRepository.saveAll(List.of(
                txForYtExport(1L, true),
                txForYtExport(2L, false),
                txForYtExport(3L, false),
                txForYtExport(4L, false)
        ));

        Pageable p1 = PageRequest.of(0, 10, Sort.by("ytId").ascending());
        assertThat(txRepository.findReadyForExport(p1)).hasSize(3)
                .isEqualTo(txs.subList(1, 4));

        Pageable p2 = PageRequest.of(0, 2, Sort.by("ytId").descending());
        assertThat(txRepository.findReadyForExport(p2)).hasSize(2)
                .isEqualTo(List.of(txs.get(3), txs.get(2)));
    }

    @Test
    public void findOldestTimestampReadyForYtExport() {
        List<BillingTransaction> txs = txRepository.saveAll(List.of(
                txForYtExport(1L, true, Instant.parse("2019-12-27T12:26:00Z")),
                txForYtExport(2L, false, Instant.parse("2019-12-27T13:26:00Z")),
                txForYtExport(3L, false, Instant.parse("2019-12-27T14:26:00Z"))
        ));

        ProcessingTasksInfo stats1 = txRepository.findOldestTimestampReadyForYtExport();
        assertThat(stats1.getOldestProcessAt()).isEqualTo("2019-12-27T13:26:00Z");
        assertThat(stats1.getCount()).isEqualTo(2);

        txs.get(1).setExportedToYt(true);
        txRepository.save(txs.get(1));
        ProcessingTasksInfo stats2 = txRepository.findOldestTimestampReadyForYtExport();
        assertThat(stats2.getOldestProcessAt()).isEqualTo("2019-12-27T14:26:00Z");
        assertThat(stats2.getCount()).isEqualTo(1);

        txs.get(2).setExportedToYt(true);
        txRepository.save(txs.get(2));
        ProcessingTasksInfo stats3 = txRepository.findOldestTimestampReadyForYtExport();
        assertThat(stats3.getOldestProcessAt()).isEqualTo((Instant) null);
        assertThat(stats3.getCount()).isEqualTo(0);
    }

    @Test
    public void findAndCountIdsReadyForActCommit_flagFilters() {
        BillingTransaction tx1 = txRepository.save(txForAct("2019-12-11T12:34:56.78Z", false, false));
        BillingTransaction tx2 = txRepository.save(txForAct("2019-12-11T12:34:56.79Z", false, false));
        txRepository.save(txForAct("2019-12-12T12:34:56.78Z", false, false));

        Instant dt1 = Instant.parse("2022-12-12T12:34:56.78Z");
        Pageable p1 = PageRequest.of(0, 10, Sort.by("id").ascending());
        assertThat(txRepository.countReadyForActCommit(dt1, dt1, NO_EXCLUDE_IDS)).isEqualTo(0);
        assertThat(txRepository.findIdsReadyForActCommit(dt1, dt1, NO_EXCLUDE_IDS, p1)).hasSize(0);

        for (BillingTransaction tx : List.of(tx1, tx2)) {
            tx.setExportedToYt(true);
            txRepository.save(tx);
        }
        assertThat(txRepository.countReadyForActCommit(dt1, dt1, NO_EXCLUDE_IDS)).isEqualTo(2);
        assertThat(txRepository.findIdsReadyForActCommit(dt1, dt1, NO_EXCLUDE_IDS, p1))
                .isEqualTo(List.of(tx1.getId(), tx2.getId()));

        tx1.setActCommitted(true);
        txRepository.save(tx1);
        assertThat(txRepository.countReadyForActCommit(dt1, dt1, NO_EXCLUDE_IDS)).isEqualTo(1);
        assertThat(txRepository.findIdsReadyForActCommit(dt1, dt1, NO_EXCLUDE_IDS, p1))
                .isEqualTo(List.of(tx2.getId()));
    }

    @Test
    public void findAndCountIdsReadyForActCommit_payoutAtActAtFilters() {
        txRepository.save(txForAct("2019-12-09T12:34:56.78Z", "2019-12-11T12:34:56.78Z", true, false));
        txRepository.save(txForAct("2019-12-09T12:34:56.79Z", "2019-12-11T12:34:56.79Z", true, false));
        txRepository.save(txForAct("2019-12-10T12:34:56.78Z", "2019-12-12T12:34:56.78Z", true, false));
        Pageable p1 = PageRequest.of(0, 10, Sort.by("id").ascending());
        List<Long> ids = txRepository.findAll(p1).stream().map(BillingTransaction::getId).collect(toList());

        Instant pdt1 = Instant.parse("2019-12-08T12:34:56.79Z");
        Instant adt1 = Instant.parse("2019-12-11T12:34:56.79Z");
        assertThat(txRepository.countReadyForActCommit(pdt1, adt1, NO_EXCLUDE_IDS)).isEqualTo(0);
        assertThat(txRepository.findIdsReadyForActCommit(pdt1, adt1, NO_EXCLUDE_IDS, p1)).hasSize(0);

        Instant pdt2 = Instant.parse("2019-12-09T12:34:56.79Z");
        assertThat(txRepository.countReadyForActCommit(pdt2, adt1, NO_EXCLUDE_IDS)).isEqualTo(1);
        assertThat(txRepository.findIdsReadyForActCommit(pdt2, adt1, NO_EXCLUDE_IDS, p1))
                .isEqualTo(List.of(ids.get(0)));

        Instant pdt3 = Instant.parse("2019-12-09T12:34:56.80Z");
        assertThat(txRepository.countReadyForActCommit(pdt3, adt1, NO_EXCLUDE_IDS)).isEqualTo(1);
        assertThat(txRepository.findIdsReadyForActCommit(pdt3, adt1, NO_EXCLUDE_IDS, p1))
                .isEqualTo(List.of(ids.get(0)));

        Instant adt2 = Instant.parse("2019-12-11T12:34:56.80Z");
        assertThat(txRepository.countReadyForActCommit(pdt3, adt2, NO_EXCLUDE_IDS)).isEqualTo(2);
        assertThat(txRepository.findIdsReadyForActCommit(pdt3, adt2, NO_EXCLUDE_IDS, p1))
                .isEqualTo(List.of(ids.get(0), ids.get(1)));
    }

    @Test
    public void findAndCountIdsReadyForActCommit_paging() {
        txRepository.save(txForAct("2019-12-11T12:34:56.78Z", true, true));
        BillingTransaction tx2 = txRepository.save(txForAct("2019-12-11T12:34:56.79Z", true, false));
        txRepository.save(txForAct("2019-12-12T12:34:56.78Z", true, false));

        Instant dt1 = Instant.parse("2222-12-11T12:34:56.78Z");
        Pageable p1 = PageRequest.of(0, 1, Sort.by("id").ascending());
        assertThat(txRepository.countReadyForActCommit(dt1, dt1, NO_EXCLUDE_IDS)).isEqualTo(2);
        assertThat(txRepository.findIdsReadyForActCommit(dt1, dt1, NO_EXCLUDE_IDS, p1)).hasSize(1)
                .isEqualTo(List.of(tx2.getId()));
    }

    @Test
    public void findMaxDelayOfTxReadyForActCommit() {
        List<BillingTransaction> txs = txRepository.saveAll(List.of(
                txForAct("2019-12-27T10:26:00Z", "2019-12-27T10:26:00Z", true, true),
                txForAct("2019-12-27T16:10:00Z", "2019-12-27T16:20:00Z", true, false),
                txForAct("2019-12-27T16:08:00Z", "2019-12-27T16:23:00Z", true, false),
                txForAct("2019-12-27T16:06:00Z", "2019-12-27T16:26:00Z", true, false),
                txForAct("2019-12-27T16:04:00Z", "2019-12-27T17:26:00Z", true, false)
        ));

        Instant maxActAt = Instant.parse("2019-12-27T16:30:00Z");
        ProcessingTasksInfo stats1 = txRepository.findMaxDelaySecondsOfTxReadyForActCommit(
                Instant.parse("2019-12-27T16:30:00Z"), maxActAt);
        // the delays for tx 2, 3 and 4: 20+(10) min, 22+(7) min, 24+(4) min
        assertThat(stats1.getOldestProcessAt().getEpochSecond()).isEqualTo(Duration.ofMinutes(10).toSeconds());
        assertThat(stats1.getCount()).isEqualTo(3);

        ProcessingTasksInfo stats2 = txRepository.findMaxDelaySecondsOfTxReadyForActCommit(
                Instant.parse("2019-12-27T16:13:00Z"), maxActAt);
        // the delays for tx 2, 3 and 4: (3)+10 min, (5)+7 min, 7+(4) min
        assertThat(stats2.getOldestProcessAt().getEpochSecond()).isEqualTo(Duration.ofMinutes(5).toSeconds());
        assertThat(stats2.getCount()).isEqualTo(3);

        txs.get(2).setActCommitted(true);
        txRepository.save(txs.get(2));
        ProcessingTasksInfo stats3 = txRepository.findMaxDelaySecondsOfTxReadyForActCommit(
                Instant.parse("2019-12-27T16:13:00Z"), maxActAt);
        // the delays for tx 2 and 4: (3)+10 min, 7+(4) min
        assertThat(stats3.getOldestProcessAt().getEpochSecond()).isEqualTo(Duration.ofMinutes(4).toSeconds());
        assertThat(stats3.getCount()).isEqualTo(2);

        // no delayed transactions
        ProcessingTasksInfo stats4 = txRepository.findMaxDelaySecondsOfTxReadyForActCommit(
                Instant.parse("2019-12-27T16:04:00Z"), maxActAt);
        assertThat(stats4.getOldestProcessAt()).isNull();
        assertThat(stats4.getCount()).isEqualTo(0);
    }

    @Test
    public void testThatProjectionSelectionWorks() {
        em.persist(partnerConfig());
        var now = Instant.now();
        var fe = new FinancialEvent();
        eventsRepository.saveAndFlush(fe);

        txRepository.saveAndFlush(BillingTransaction.builder()
                .serviceOrderId("TEST_ORDER_ID")
                .partnerId(TEST_PARTNER_ID)
                .exportedToYt(true)
                .exportedToYtAt(now)
                .sourceFinancialEvent(fe)
                .build());

        var transactions = txRepository.billingTransactionsForPayoutReport( TEST_PARTNER_ID,
                now.minusSeconds(10), now.plusSeconds(10));

        assertThat(transactions.size()).isEqualTo(1);
        var tx = transactions.get(0);
        assertThat(tx.getId()).isPositive();
        assertThat(tx.getServiceOrderId()).isEqualTo("TEST_ORDER_ID");
        assertThat(tx.getSourceFinancialEvent()).isNotNull();
        assertThat(tx.getSourceFinancialEvent().getId()).isPositive();
    }

    private BillingTransaction txForAct(String actAt, boolean exported, boolean committed) {
        return txForAct(Instant.now().toString(), actAt, exported, committed);
    }

    private BillingTransaction txForAct(String payoutAt, String actAt, boolean exported, boolean committed) {
        return BillingTransaction.builder()
                .payoutAt(Instant.parse(payoutAt))
                .accountingActAt(Instant.parse(actAt))
                .exportedToYt(exported).actCommitted(committed).build();
    }

    private BillingTransaction txWithSourceEvent(FinancialEvent event,
                                                 BillingTransactionPaymentType paymentType,
                                                 BillingTransactionPaymentSystemType paymentSystemType) {
        return BillingTransaction.builder()
                .sourceFinancialEvent(event)
                .paymentType(paymentType)
                .paymentSystemType(paymentSystemType)
                .build();
    }

    private BillingTransaction txForYtIdGenerator(String payoutAt, BillingTransactionType type) {
        return txForYtIdGenerator(payoutAt, type, TEST_PARTNER_ID);
    }

    private BillingTransaction txForYtIdGenerator(String payoutAt, BillingTransactionType type, long partnerId) {
        return BillingTransaction.builder()
                .payoutAt(Instant.parse(payoutAt))
                .accountingActAt(Instant.now())
                .transactionType(type)
                .partnerId(partnerId)
                .build();
    }

    private BillingTransaction txForYtExport(Long ytId, Boolean exportedToYt) {
        return txForYtExport(ytId, exportedToYt, null);
    }

    private BillingTransaction txForYtExport(Long ytId, Boolean exportedToYt, Instant payoutAt) {
        return BillingTransaction.builder().ytId(ytId).exportedToYt(exportedToYt).payoutAt(payoutAt).build();
    }

    private BillingPartnerConfig partnerConfig() {
        return partnerConfig(TEST_PARTNER_ID);
    }

    private BillingPartnerConfig partnerConfig(long billingClientId) {
        return BillingPartnerConfig.builder()
                .billingClientId(billingClientId)
                .exportToYt(true)
                .build();
    }
}
