package ru.yandex.travel.orders.repository;

import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneOffset;
import java.util.List;
import java.util.UUID;

import javax.persistence.EntityManager;

import org.javamoney.moneta.Money;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.orm.jpa.DataJpaTest;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;

import ru.yandex.travel.orders.entities.DolphinOrderItem;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.finances.FinancialEvent;
import ru.yandex.travel.orders.entities.finances.ProcessingTasksInfo;
import ru.yandex.travel.orders.entities.partners.BillingPartnerConfig;

import static java.util.stream.Collectors.toList;
import static org.assertj.core.api.Assertions.assertThat;
import static ru.yandex.travel.orders.entities.finances.FinancialEventPaymentScheme.HOTELS;
import static ru.yandex.travel.orders.entities.finances.FinancialEventType.PAYMENT;
import static ru.yandex.travel.orders.entities.finances.FinancialEventType.REFUND;
import static ru.yandex.travel.orders.repository.FinancialEventRepository.NO_EXCLUDE_IDS;

@RunWith(SpringRunner.class)
@DataJpaTest
@ActiveProfiles("test")
public class FinancialEventRepositoryTest {
    private static final Long TEST_BILLING_CLIENT_ID = 8152981894121414L;

    @Autowired
    private FinancialEventRepository repository;
    @Autowired
    private EntityManager em;

    @Before
    public void init() {
        em.persist(BillingPartnerConfig.builder()
                .billingClientId(TEST_BILLING_CLIENT_ID)
                .generateTransactions(true)
                .build());
    }

    @Test
    public void testInsertion() {
        // simply validating the structure of db tables with not so real data
        HotelOrder order = new HotelOrder();
        order.setId(UUID.randomUUID());
        order.setPrettyId("YA-PID");
        DolphinOrderItem orderItem = new DolphinOrderItem();
        orderItem.setId(UUID.randomUUID());
        orderItem.setOrder(order);
        Instant now = Instant.now();
        Instant payoutDt = Instant.now().plus(Duration.ofDays(5));
        Instant accountingActDt = payoutDt.plus(Duration.ofDays(1).plusMinutes(11));

        FinancialEvent event = FinancialEvent.create(orderItem, PAYMENT, HOTELS, now);
        event.setBillingClientId(8765312L);
        event.setBillingContractId(2114124L);
        event.setPayoutAt(payoutDt);
        event.setAccountingActAt(accountingActDt);
        event.setPartnerAmount(Money.of(8700, "RUB"));
        event.setPartnerRefundAmount(Money.of(870, "USD"));
        event.setFeeAmount(Money.of(1300, "RUB"));
        event.setFeeRefundAmount(Money.of(130, "EUR"));

        em.persist(order);
        em.persist(orderItem);
        em.persist(BillingPartnerConfig.builder().billingClientId(8765312L).build());
        repository.save(event);

        List<FinancialEvent> allEvents = repository.findAll();
        assertThat(allEvents).hasSize(1).first().satisfies(fe -> {
            assertThat(fe.getId()).isNotNull();
            assertThat(fe.getOrder().getId()).isEqualTo(order.getId());
            assertThat(fe.getOrderPrettyId()).isEqualTo("YA-PID");
            assertThat(fe.getOrderItem().getId()).isEqualTo(orderItem.getId());
            assertThat(fe.getType()).isEqualTo(PAYMENT);
            assertThat(fe.getAccrualAt()).isEqualTo(now);
            assertThat(fe.getBillingClientId()).isEqualTo(8765312);
            assertThat(fe.getBillingContractId()).isEqualTo(2114124L);
            assertThat(fe.getPayoutAt()).isEqualTo(payoutDt);
            assertThat(fe.getAccountingActAt()).isEqualTo(accountingActDt);
            assertThat(fe.getPartnerAmount()).isEqualTo(Money.of(8700, "RUB"));
            assertThat(fe.getPartnerRefundAmount()).isEqualTo(Money.of(870, "USD"));
            assertThat(fe.getFeeAmount()).isEqualTo(Money.of(1300, "RUB"));
            assertThat(fe.getFeeRefundAmount()).isEqualTo(Money.of(130, "EUR"));
        });
    }

    @Test
    public void testFindAndCountUnprocessedIds_processingFlag() {
        Pageable noPaging = Pageable.unpaged();
        OrderItem orderItem1 = createStoredOrderItem();
        OrderItem orderItem2 = createStoredOrderItem();
        OrderItem orderItem3 = createStoredOrderItem();

        assertThat(repository.countUnprocessedIds(NO_EXCLUDE_IDS)).isEqualTo(0);
        List<Long> r1 = repository.findUnprocessedIds(NO_EXCLUDE_IDS, noPaging);
        assertThat(r1).hasSize(0);

        FinancialEvent e1 = repository.save(eventBuilder(orderItem1, ts("2019-12-02")).build());
        assertThat(repository.countUnprocessedIds(NO_EXCLUDE_IDS)).isEqualTo(1);
        List<Long> r2 = repository.findUnprocessedIds(NO_EXCLUDE_IDS, noPaging);
        assertThat(r2).hasSize(1);

        repository.save(eventBuilder(orderItem2, ts("2019-12-01")).build());
        repository.save(eventBuilder(orderItem3, ts("2019-11-30")).build());
        assertThat(repository.countUnprocessedIds(NO_EXCLUDE_IDS)).isEqualTo(3);
        List<Long> r3 = repository.findUnprocessedIds(NO_EXCLUDE_IDS, noPaging);
        assertThat(r3).hasSize(3);

        e1.setProcessed(true);
        repository.save(e1);

        assertThat(repository.countUnprocessedIds(NO_EXCLUDE_IDS)).isEqualTo(2);
        List<Long> r4 = repository.findUnprocessedIds(NO_EXCLUDE_IDS, noPaging);
        assertThat(r4).hasSize(2)
                .doesNotContain(e1.getId());
    }

    @Test
    public void testFindAndCountUnprocessedIds_excludeList() {
        repository.save(eventBuilder(createStoredOrderItem(), ts("2019-11-17")).build());
        repository.save(eventBuilder(createStoredOrderItem(), ts("2019-11-18")).build());
        repository.save(eventBuilder(createStoredOrderItem(), ts("2019-11-19")).build());
        List<Long> ids = repository.findAll().stream().map(FinancialEvent::getId).collect(toList());
        Pageable noPaging = Pageable.unpaged();

        // it seems like empty values list works only in h2, use NO_EXCLUDE_IDS for production code
        assertThat(repository.countUnprocessedIds(List.of())).isEqualTo(3);
        assertThat(repository.findUnprocessedIds(List.of(), noPaging)).hasSize(3)
                .containsAll(ids);

        assertThat(repository.countUnprocessedIds(NO_EXCLUDE_IDS)).isEqualTo(3);
        assertThat(repository.findUnprocessedIds(NO_EXCLUDE_IDS, noPaging)).hasSize(3)
                .containsAll(ids);

        assertThat(repository.countUnprocessedIds(List.of(ids.get(0)))).isEqualTo(2);
        assertThat(repository.findUnprocessedIds(List.of(ids.get(0)), noPaging)).hasSize(2)
                .containsAll(ids.subList(1, 3));

        assertThat(repository.countUnprocessedIds(ids)).isEqualTo(0);
        assertThat(repository.findUnprocessedIds(ids, noPaging)).hasSize(0);
    }

    @Test
    public void testFindUnprocessedIds_pagerAndSorting() {
        repository.save(eventBuilder(createStoredOrderItem(), ts("2019-11-17")).build());
        repository.save(eventBuilder(createStoredOrderItem(), ts("2019-11-18")).build());
        repository.save(eventBuilder(createStoredOrderItem(), ts("2019-11-19")).build());
        List<Long> ids = repository.findAll(Sort.by("id").ascending()).stream().map(FinancialEvent::getId).collect(toList());

        Pageable p1 = PageRequest.of(0, 1);
        assertThat(repository.findUnprocessedIds(List.of(), p1)).hasSize(1).contains(ids.get(0));

        Pageable p2 = PageRequest.of(0, 1, Sort.by("id").descending()); // the parameter is ignored
        assertThat(repository.findUnprocessedIds(List.of(), p2)).hasSize(1).contains(ids.get(2));

        Pageable p3 = PageRequest.of(1, 1, Sort.by("id").ascending());
        assertThat(repository.findUnprocessedIds(List.of(), p3)).hasSize(1).contains(ids.get(1));

        Pageable p4 = PageRequest.of(0, 2, Sort.by("id").ascending());
        assertThat(repository.findUnprocessedIds(List.of(), p4)).hasSize(2).containsAll(ids.subList(0, 2));

        Pageable p5 = PageRequest.of(1, 2, Sort.by("id").ascending());
        assertThat(repository.findUnprocessedIds(List.of(), p5)).hasSize(1).contains(ids.get(2));
    }

    @Test
    public void testGenerateTransactionMethods_disabledPartner() {
        long billingClientId = 8725348629846294324L;
        BillingPartnerConfig config = BillingPartnerConfig.builder()
                .billingClientId(billingClientId)
                .generateTransactions(true)
                .build();
        em.persist(config);
        repository.save(eventBuilder(createStoredOrderItem(), ts("2019-11-17")).billingClientId(billingClientId)
                .accrualAt(ts("2019-10-07")).build());
        repository.save(eventBuilder(createStoredOrderItem(), ts("2019-11-18")).billingClientId(billingClientId)
                .accrualAt(ts("2019-10-08")).build());
        repository.save(eventBuilder(createStoredOrderItem(), ts("2019-11-19"))
                .accrualAt(ts("2019-10-09")).build());
        List<Long> ids = repository.findAll(Sort.by("id").ascending()).stream().map(FinancialEvent::getId).collect(toList());

        Pageable noPaging = Pageable.unpaged();
        assertThat(repository.countUnprocessedIds(NO_EXCLUDE_IDS)).isEqualTo(3);
        assertThat(repository.findUnprocessedIds(NO_EXCLUDE_IDS, noPaging))
                .hasSize(3).containsAll(ids);
        ProcessingTasksInfo stats1 = repository.findOldestUnprocessedTimestamp();
        assertThat(stats1.getOldestProcessAt()).isEqualTo(ts("2019-10-07"));
        assertThat(stats1.getCount()).isEqualTo(3);

        config.setGenerateTransactions(false);
        em.persist(config);

        assertThat(repository.countUnprocessedIds(NO_EXCLUDE_IDS)).isEqualTo(1);
        assertThat(repository.findUnprocessedIds(NO_EXCLUDE_IDS, noPaging))
                .hasSize(1).containsAll(ids.subList(2, 3)); // only the last ids with enabled client
        ProcessingTasksInfo stats2 = repository.findOldestUnprocessedTimestamp();
        assertThat(stats2.getOldestProcessAt()).isEqualTo(ts("2019-10-09"));
        assertThat(stats2.getCount()).isEqualTo(1);
    }

    @Test
    public void findFirstByOrderItemAndType() {
        OrderItem orderItem1 = createStoredOrderItem();
        OrderItem orderItem2 = createStoredOrderItem();

        repository.save(eventBuilder(null).type(PAYMENT).build());
        assertThat(repository.findFirstByOrderItemAndTypeOrderByIdAsc(orderItem1, PAYMENT)).isNull();

        FinancialEvent e1 = repository.save(eventBuilder(orderItem1).type(PAYMENT).build());
        assertThat(repository.findFirstByOrderItemAndTypeOrderByIdAsc(orderItem1, PAYMENT)).isEqualTo(e1);
        assertThat(repository.findFirstByOrderItemAndTypeOrderByIdAsc(orderItem1, REFUND)).isNull();
        assertThat(repository.findFirstByOrderItemAndTypeOrderByIdAsc(orderItem2, PAYMENT)).isNull();

        repository.save(eventBuilder(orderItem1).type(PAYMENT).build());
        assertThat(repository.findFirstByOrderItemAndTypeOrderByIdAsc(orderItem1, PAYMENT)).isEqualTo(e1);
    }

    @Test
    public void findOldestUnprocessedTimestamp() {
        FinancialEvent e1 = repository.save(eventBuilder(null).accrualAt(ts("2019-12-27T12:17:00Z")).build());
        FinancialEvent e2 = repository.save(eventBuilder(null).accrualAt(ts("2019-12-27T12:28:00Z")).build());

        assertThat(repository.findOldestUnprocessedTimestamp().getOldestProcessAt()).isEqualTo("2019-12-27T12:17:00Z");

        e1.setProcessed(true);
        repository.save(e1);
        assertThat(repository.findOldestUnprocessedTimestamp().getOldestProcessAt()).isEqualTo("2019-12-27T12:28:00Z");

        e2.setProcessed(true);
        repository.save(e2);
        assertThat(repository.findOldestUnprocessedTimestamp().getOldestProcessAt()).isEqualTo((Instant) null);
    }

    private OrderItem createStoredOrderItem() {
        OrderItem orderItem = new DolphinOrderItem();
        orderItem.setId(UUID.randomUUID());
        em.persist(orderItem);
        return orderItem;
    }

    private FinancialEvent.FinancialEventBuilder eventBuilder(OrderItem orderItem) {
        return FinancialEvent.builder().billingClientId(TEST_BILLING_CLIENT_ID).orderItem(orderItem);
    }

    private FinancialEvent.FinancialEventBuilder eventBuilder(OrderItem orderItem, Instant payoutAt) {
        return eventBuilder(orderItem).payoutAt(payoutAt);
    }

    private Instant ts(String ts) {
        if (ts.matches("\\d{4}-\\d{2}-\\d{2}")) {
            return LocalDate.parse(ts).atStartOfDay().toInstant(ZoneOffset.UTC);
        }
        return Instant.parse(ts);
    }
}
