package ru.yandex.travel.orders.repository;

import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneOffset;
import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;

import javax.persistence.EntityManager;

import com.google.common.base.Preconditions;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.orm.jpa.DataJpaTest;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;

import ru.yandex.travel.hotels.common.orders.ExpediaHotelItinerary;
import ru.yandex.travel.hotels.common.orders.OrderDetails;
import ru.yandex.travel.orders.entities.AeroflotInvoice;
import ru.yandex.travel.orders.entities.AuthorizedUser;
import ru.yandex.travel.orders.entities.ExpediaOrderItem;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.entities.WellKnownInvoiceDiscriminator;
import ru.yandex.travel.orders.services.payments.InvoicePaymentFlags;

import static org.assertj.core.api.Assertions.assertThat;

@RunWith(SpringRunner.class)
@DataJpaTest
@ActiveProfiles("test")
public class InvoiceRepositoryTest {
    @Autowired
    private InvoiceRepository repository;
    @Autowired
    private EntityManager em;

    @Test
    public void findByUpdatedAtAndType() {
        createInvoice("2020-01-16", false);
        createInvoice("2020-01-15", true);
        Invoice invoice3 = createInvoice("2020-01-16", true);
        createInvoice("2020-01-17", true);

        List<String> trustType = List.of(WellKnownInvoiceDiscriminator.INVOICE_TRUST);
        Instant ts1 = Instant.parse("2020-01-16T00:00:00Z");
        Instant ts2 = Instant.parse("2020-01-16T00:00:01Z");
        assertThat(repository.findByUpdatedAtAndType(ts1, ts2, trustType)).hasSize(1).first()
                .matches(o -> invoice3.getId().equals(o.getId()));
    }

    private Invoice createInvoice(String updatedAt, boolean ofTrustType) {
        ExpediaHotelItinerary itinerary = new ExpediaHotelItinerary();
        itinerary.setOrderDetails(OrderDetails.builder()
                .checkinDate(LocalDate.now())
                .build());
        ExpediaOrderItem orderItem = new ExpediaOrderItem();
        orderItem.setId(UUID.randomUUID());
        orderItem.setItinerary(itinerary);

        Order order = new HotelOrder();
        order.setId(UUID.randomUUID());
        order.addOrderItem(orderItem);
        em.persist(order);

        AuthorizedUser user = AuthorizedUser.createGuest(null, "sKey", null, null);

        Invoice invoice = ofTrustType ?
                TrustInvoice.createInvoice(order, user, InvoicePaymentFlags.builder()
                        .force3ds(true)
                        .useMirPromo(false)
                        .processThroughYt(false)
                        .build()) :
                AeroflotInvoice.createInvoice(
                        order,
                        order.getOrderItems()
                                .stream()
                                .flatMap(oi -> oi.getFiscalItems().stream()).collect(Collectors.toList()),
                        user, false
                );
        invoice.setId(UUID.randomUUID());
        em.persist(invoice);

        // forcing a concrete update_at timestamp
        int updated = em.createQuery("update Invoice i set i.updatedAt = ?1 where id = ?2")
                .setParameter(1, LocalDate.parse(updatedAt).atStartOfDay(ZoneOffset.UTC).toInstant())
                .setParameter(2, invoice.getId())
                .executeUpdate();
        Preconditions.checkState(updated == 1, "Invoice update failed; updated rows - %s", updated);

        em.clear();
        return em.find(Invoice.class, invoice.getId());
    }
}
