package ru.yandex.travel.orders.repository;

import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneOffset;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import javax.persistence.EntityManager;

import com.google.common.base.Preconditions;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.orm.jpa.DataJpaTest;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;

import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.DolphinOrderItem;
import ru.yandex.travel.orders.entities.ExpediaOrderItem;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.WellKnownOrderItemDiscriminator;

import static java.util.stream.Collectors.toSet;
import static org.assertj.core.api.Assertions.assertThat;

@RunWith(SpringRunner.class)
@DataJpaTest
@ActiveProfiles("test")
public class OrderItemRepositoryTest {
    @Autowired
    private OrderItemRepository repository;
    @Autowired
    private EntityManager em;

    @Test
    public void findByUpdatedAtAndType() {
        createOrderWithUpdatedAt("2020-01-16", false);
        OrderItem o1 = createOrderWithUpdatedAt("2020-01-15", true);
        OrderItem o2 = createOrderWithUpdatedAt("2020-01-16", true);
        OrderItem o3 = createOrderWithUpdatedAt("2020-01-17", true);

        List<String> dolphinType = List.of(WellKnownOrderItemDiscriminator.ORDER_ITEM_DOLPHIN);
        Instant ts1 = Instant.parse("2020-01-16T00:00:00Z");
        Instant ts2 = Instant.parse("2020-01-16T00:00:01Z");
        Instant ts3 = Instant.parse("2020-01-17T00:00:00Z");
        assertThat(repository.findByUpdatedAtAndType(ts1, ts1, dolphinType)).isEmpty();
        assertThat(repository.findByUpdatedAtAndType(ts1, ts2, dolphinType)).hasSize(1).first()
                .matches(o -> o2.getId().equals(o.getId()));
        assertThat(repository.findByUpdatedAtAndType(ts1, ts3, dolphinType)).hasSize(1).first()
                .matches(o -> o2.getId().equals(o.getId()));

        Instant ts4 = Instant.parse("2020-01-15T00:00:00Z");
        Instant ts5 = Instant.parse("2020-01-17T00:00:00.001Z");
        List<OrderItem> allDolphinOrders = repository.findByUpdatedAtAndType(ts4, ts5, dolphinType);
        assertThat(allDolphinOrders).hasSize(3)
                .allMatch(o -> o.getPublicType() == EServiceType.PT_DOLPHIN_HOTEL);
        assertThat(allDolphinOrders.stream().map(OrderItem::getId).collect(toSet()))
                .isEqualTo(Set.of(o1.getId(), o2.getId(), o3.getId()));

        List<String> allTypes = List.of(
                WellKnownOrderItemDiscriminator.ORDER_ITEM_DOLPHIN,
                WellKnownOrderItemDiscriminator.ORDER_ITEM_EXPEDIA
        );
        List<OrderItem> allOrders = repository.findByUpdatedAtAndType(ts4, ts5, allTypes);
        assertThat(allOrders).hasSize(4);
        assertThat(allOrders.stream().map(OrderItem::getPublicType).collect(toSet()))
                .isEqualTo(Set.of(EServiceType.PT_DOLPHIN_HOTEL, EServiceType.PT_EXPEDIA_HOTEL));
    }

    private OrderItem createOrderWithUpdatedAt(String date, boolean ofDolphinType) {
        OrderItem orderItem = ofDolphinType ? new DolphinOrderItem() : new ExpediaOrderItem();
        orderItem.setId(UUID.randomUUID());
        repository.save(orderItem);
        // forcing a concrete update_at timestamp
        int updated = em.createQuery("update OrderItem oi set oi.updatedAt = ?1 where id = ?2")
                .setParameter(1, LocalDate.parse(date).atStartOfDay(ZoneOffset.UTC).toInstant())
                .setParameter(2, orderItem.getId())
                .executeUpdate();
        Preconditions.checkState(updated == 1, "Item update failed; updated rows - %s", updated);
        em.clear();
        return repository.getOne(orderItem.getId());
    }
}
