package ru.yandex.travel.orders.repository.cpa;

import java.time.Instant;
import java.time.LocalDate;
import java.time.temporal.ChronoUnit;
import java.util.List;
import java.util.UUID;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.orm.jpa.DataJpaTest;
import org.springframework.data.domain.PageRequest;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;

import ru.yandex.travel.hotels.common.orders.DolphinHotelItinerary;
import ru.yandex.travel.hotels.common.orders.ExpediaHotelItinerary;
import ru.yandex.travel.hotels.common.orders.OrderDetails;
import ru.yandex.travel.orders.entities.DolphinOrderItem;
import ru.yandex.travel.orders.entities.ExpediaOrderItem;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.WellKnownWorkflowEntityType;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

import static org.assertj.core.api.Assertions.assertThat;

@RunWith(SpringRunner.class)
@DataJpaTest
@ActiveProfiles("test")
public class CpaHotelOrderRepositoryTest {
    @Autowired
    private CpaHotelOrderRepository cpaHotelOrderRepository;
    @Autowired
    private WorkflowRepository workflowRepository;

    @Test
    public void testCustomSelectIsWorking() {
        ExpediaOrderItem expediaOrderItem = new ExpediaOrderItem();
        var itinerary = new ExpediaHotelItinerary();
        itinerary.setOrderDetails(OrderDetails.builder().checkinDate(LocalDate.now()).build());
        expediaOrderItem.setItinerary(itinerary);
        expediaOrderItem.setId(UUID.randomUUID());
        Workflow workflow = Workflow.createWorkflowForEntity(expediaOrderItem);
        workflowRepository.save(workflow);

        DolphinOrderItem dolphinOrderItem = new DolphinOrderItem();
        var dolphinItinerary = new DolphinHotelItinerary();
        dolphinItinerary.setCheckinDate(LocalDate.now());
        dolphinOrderItem.setItinerary(dolphinItinerary);
        dolphinOrderItem.setId(UUID.randomUUID());
        Workflow dolphinWorkflow = Workflow.createWorkflowForEntity(dolphinOrderItem);
        workflowRepository.save(dolphinWorkflow);

        cpaHotelOrderRepository.saveAndFlush(createOrder(expediaOrderItem));
        cpaHotelOrderRepository.saveAndFlush(createOrder(dolphinOrderItem));

        List<HotelOrder> orderList = cpaHotelOrderRepository.findOrdersByItemTypeAndUpdatedAtAndStateIn(
                Instant.now().minus(10, ChronoUnit.MINUTES),
                Instant.now(),
                List.of(EHotelOrderState.OS_CONFIRMED),
                WellKnownWorkflowEntityType.EXPEDIA_ORDER_ITEM.getDiscriminatorValue(),
                PageRequest.of(0, 10));

        assertThat(orderList.size()).isEqualTo(1);
    }

    private HotelOrder createOrder(OrderItem item) {
        HotelOrder order = new HotelOrder();
        order.setId(UUID.randomUUID());
        order.setPrettyId(order.getId().toString());
        order.setState(EHotelOrderState.OS_CONFIRMED);
        order.setUpdatedAt(Instant.now());
        order.setCreatedAt(Instant.now());
        order.addOrderItem(item);

        return order;
    }
}
