package ru.yandex.travel.orders.repository.cpa;

import java.time.Instant;
import java.util.Set;
import java.util.UUID;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.orm.jpa.DataJpaTest;
import org.springframework.data.domain.PageRequest;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;

import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.SuburbanOrderItem;
import ru.yandex.travel.orders.repository.SuburbanOrderItemRepository;

import static org.assertj.core.api.Assertions.assertThat;


@RunWith(SpringRunner.class)
@DataJpaTest
@ActiveProfiles("test")
public class CpaOrderRepositoryTest {
    @Autowired
    private SuburbanOrderItemRepository orderItemRepository;

    @Autowired
    private CpaOrderRepository cpaOrderRepository;

    @Test
    public void testFindOrders() {
        createOrder(EDisplayOrderType.DT_AVIA);
        createOrder(EDisplayOrderType.DT_SUBURBAN);
        createOrder(EDisplayOrderType.DT_TRAIN);
        createOrder(EDisplayOrderType.DT_BUS);
        createOrder(EDisplayOrderType.DT_SUBURBAN);
        createOrder(EDisplayOrderType.DT_BUS);

        Instant now = Instant.now();
        var orders = cpaOrderRepository.findOrdersByUpdatedAt(
                now.minusSeconds(20),
                now.minusSeconds(0),
                Set.of(EDisplayOrderType.DT_TRAIN, EDisplayOrderType.DT_SUBURBAN),
                PageRequest.of(0, 2)
        );
        assertThat(orders.size()).isEqualTo(2);
        assertThat(orders.get(0).getDisplayType()).isEqualTo(EDisplayOrderType.DT_SUBURBAN);
        assertThat(orders.get(0).getOrderItems().size()).isEqualTo(1);
        assertThat(orders.get(1).getDisplayType()).isEqualTo(EDisplayOrderType.DT_TRAIN);

        orders = cpaOrderRepository.findOrdersByUpdatedAt(
                now.minusSeconds(20),
                now.minusSeconds(15),
                Set.of(EDisplayOrderType.DT_TRAIN, EDisplayOrderType.DT_SUBURBAN),
                PageRequest.of(0, 2)
        );
        assertThat(orders.size()).isEqualTo(0);
    }

    private GenericOrder createOrder(EDisplayOrderType displayOrderType) {
        var orderItem = new SuburbanOrderItem();
        orderItem.setId(UUID.randomUUID());
        orderItemRepository.saveAndFlush(orderItem);

        var order = new GenericOrder();
        order.setId(UUID.randomUUID());
        order.setDisplayType(displayOrderType);
        order.addOrderItem(orderItem);
        cpaOrderRepository.saveAndFlush(order);

        return order;
    }
}
