package ru.yandex.travel.orders.repository.migrations;

import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ThreadLocalRandom;

import javax.persistence.EntityManager;

import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.orm.jpa.DataJpaTest;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;

import ru.yandex.travel.orders.entities.AeroflotOrder;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.TrainOrder;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.train.proto.ETrainOrderState;

import static org.assertj.core.api.Assertions.assertThat;

@RunWith(SpringRunner.class)
@DataJpaTest
@ActiveProfiles("test")
public class TrainToGenericMigrationRepositoryTest {
    private static final Set<UUID> NO_LOCKED_IDS = Set.of(UUID.randomUUID());
    private static final Pageable NO_PAGE_LIMIT = PageRequest.of(0, 10);

    @Autowired
    private TrainToGenericMigrationRepository repository;
    @Autowired
    private EntityManager em;

    @Test
    public void testChangeOldTrainOrderType() {
        TrainOrder trainOrder = createOldTrainOrder(ETrainOrderState.OS_CONFIRMED);
        assertThat(trainOrder.getState()).isEqualTo(ETrainOrderState.OS_CONFIRMED);

        repository.changeOldTrainOrderTypeAndState(trainOrder.getId(), EOrderState.OS_CONFIRMED);

        // evicting it from the em's cache
        em.detach(trainOrder);

        GenericOrder genericOrder = repository.getOne(trainOrder.getId());
        assertThat(genericOrder.getEmail()).isEqualTo(trainOrder.getEmail());
        assertThat(genericOrder.getState()).isEqualTo(EOrderState.OS_CONFIRMED);
    }

    @Test
    public void testFindAndCountTaskIds() {
        // just in case
        createAeroflotOrder();

        UUID orderId1 = createOldTrainOrder(ETrainOrderState.OS_CANCELLED).getId();
        assertThat(repository.findOldTrainOrderIds(NO_LOCKED_IDS, NO_PAGE_LIMIT)).isSubsetOf(orderId1);
        assertThat(repository.countOldTrainOrderIds(NO_LOCKED_IDS)).isEqualTo(1);

        UUID orderId2 = createOldTrainOrder(ETrainOrderState.OS_WAITING_PAYMENT).getId();
        assertThat(repository.findOldTrainOrderIds(NO_LOCKED_IDS, NO_PAGE_LIMIT)).isSubsetOf(orderId1, orderId2);
        assertThat(repository.countOldTrainOrderIds(NO_LOCKED_IDS)).isEqualTo(2);

        assertThat(repository.findOldTrainOrderIds(Set.of(orderId1), NO_PAGE_LIMIT)).isSubsetOf(orderId2);
        assertThat(repository.countOldTrainOrderIds(Set.of(orderId1))).isEqualTo(1);

        assertThat(repository.findOldTrainOrderIds(NO_LOCKED_IDS, PageRequest.of(0, 1))).hasSize(1);
    }

    private TrainOrder createOldTrainOrder(ETrainOrderState state) {
        TrainOrder order = new TrainOrder();
        order.setId(UUID.randomUUID());
        order.setState(state);
        order.setEmail("old-order-owner-" + ThreadLocalRandom.current().nextInt() + "@example.com");
        em.persist(order);
        return order;
    }

    private void createAeroflotOrder() {
        AeroflotOrder order = new AeroflotOrder();
        order.setId(UUID.randomUUID());
        em.persist(order);
    }
}
