package ru.yandex.travel.orders.repository.promo.aeroflotplus;

import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneId;
import java.util.List;

import javax.persistence.EntityManager;

import org.hibernate.exception.ConstraintViolationException;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.orm.jpa.DataJpaTest;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;

import ru.yandex.travel.orders.entities.promo.aeroflotplus.AeroflotPlusPromoCode;

import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.Assertions.assertThatThrownBy;

@RunWith(SpringRunner.class)
@DataJpaTest
@ActiveProfiles("test")
public class AeroflotPlusPromoCodeRepositoryTest {
    @Autowired
    private AeroflotPlusPromoCodeRepository repository;
    @Autowired
    private EntityManager em;

    @Test
    public void testFindFreeCode() {
        assertNoFreeCodesCanBeFound();
        assertThat(repository.countByUsedAtIsNull()).isEqualTo(0);

        AeroflotPlusPromoCode code1 = repository.saveAndFlush(code("code1"));
        Integer code1Version = code1.getVersion();
        assertThat(repository.findFreeCodes(1).get(0).getCode()).isEqualTo("code1");
        assertThat(repository.countByUsedAtIsNull()).isEqualTo(1);

        code1.setUsedAt(Instant.now());
        repository.saveAndFlush(code1);
        assertNoFreeCodesCanBeFound();
        assertThat(repository.countByUsedAtIsNull()).isEqualTo(0);
        assertThat(code1.getVersion()).isNotEqualTo(code1Version);

        repository.saveAndFlush(code("code2"));
        repository.saveAndFlush(code("code3"));
        repository.saveAndFlush(code("code4"));
        List<AeroflotPlusPromoCode> codes = repository.findFreeCodes(2);
        assertThat(codes).hasSize(2).allSatisfy(code -> {
            assertThat(code.getUsedAt()).isNull();
            assertThat(code.getCode()).isIn(List.of("code2", "code3", "code4"));
            assertThat(code.getPlusPoints()).isEqualTo(1_000);
        });
        assertThat(repository.countByUsedAtIsNull()).isEqualTo(3);
    }

    @Test
    public void testFindFreeCodeOrdering() {
        AeroflotPlusPromoCode code1 = repository.saveAndFlush(code("code1_upload1", 1, "2021-06-15"));
        AeroflotPlusPromoCode code2 = repository.saveAndFlush(code("code2_upload2", 1, "2021-06-16"));

        assertThat(repository.findFreeCodes(1).get(0).getCode()).isEqualTo("code1_upload1");
    }

    @Test
    public void testDupCodes() {
        em.persist(code("codeDup"));
        em.flush();
        em.clear();
        assertThatThrownBy(() -> {
            em.persist(code("codeDup"));
            em.flush();
        }).hasCauseExactlyInstanceOf(ConstraintViolationException.class);
    }

    private void assertNoFreeCodesCanBeFound() {
        assertThatThrownBy(() -> repository.findFreeCodes(1))
                .isExactlyInstanceOf(RuntimeException.class)
                .hasMessageContaining("Failed to find")
                .hasMessageContaining("free codes");
    }

    private AeroflotPlusPromoCode code(String code) {
        return code(code, 1_000, LocalDate.now().toString());
    }

    private AeroflotPlusPromoCode code(String code, Integer points, String addedAtDt) {
        Instant addedAt = LocalDate.parse(addedAtDt).atStartOfDay(ZoneId.of("UTC")).toInstant();
        return AeroflotPlusPromoCode.builder()
                .code(code)
                .plusPoints(points)
                .addedAt(addedAt)
                .expiresAt(addedAt.plus(Duration.ofDays(4_000_000)))
                .build();
    }
}
