package ru.yandex.travel.orders.repository.promo.taxi2020;

import java.time.Instant;
import java.util.List;

import javax.persistence.EntityManager;

import org.hibernate.exception.ConstraintViolationException;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.orm.jpa.DataJpaTest;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;

import ru.yandex.travel.orders.entities.promo.taxi2020.Taxi2020PromoCode;

import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.Assertions.assertThatThrownBy;

@RunWith(SpringRunner.class)
@DataJpaTest
@ActiveProfiles("test")
public class Taxi2020PromoCodeRepositoryTest {
    @Autowired
    private Taxi2020PromoCodeRepository repository;
    @Autowired
    private EntityManager em;

    @Test
    public void testFindFreeCode() {
        assertThat(repository.findAnyByUsedAtIsNull()).isNull();
        assertThat(repository.countByUsedAtIsNull()).isEqualTo(0);

        Taxi2020PromoCode code1 = repository.saveAndFlush(code("code1"));
        assertThat(repository.findAnyByUsedAtIsNull().getCode()).isEqualTo("code1");
        assertThat(repository.countByUsedAtIsNull()).isEqualTo(1);

        code1.setUsedAt(Instant.now());
        repository.saveAndFlush(code1);
        assertThat(repository.findAnyByUsedAtIsNull()).isNull();
        assertThat(repository.countByUsedAtIsNull()).isEqualTo(0);

        repository.saveAndFlush(code("code2"));
        repository.saveAndFlush(code("code3"));
        Taxi2020PromoCode oneOfFreeCodes = repository.findAnyByUsedAtIsNull();
        assertThat(oneOfFreeCodes).isNotNull();
        assertThat(oneOfFreeCodes.getUsedAt()).isNull();
        assertThat(oneOfFreeCodes.getCode()).isIn(List.of("code2", "code3"));
        assertThat(repository.countByUsedAtIsNull()).isEqualTo(2);
    }

    @Test
    public void testFindFreeCodeOrdering() {
        Taxi2020PromoCode code1 = repository.saveAndFlush(code("code1_upload1", Instant.parse("2021-01-11T14:20:00Z")));
        Taxi2020PromoCode code2 = repository.saveAndFlush(code("code2_upload2", Instant.parse("2021-03-11T14:20:00Z")));

        assertThat(repository.findAnyByUsedAtIsNull().getCode()).isEqualTo("code1_upload1");
    }

    @Test
    public void testDupCodes() {
        em.persist(code("codeDup"));
        em.flush();
        em.clear();
        assertThatThrownBy(() -> {
            em.persist(code("codeDup"));
            em.flush();
        }).hasCauseExactlyInstanceOf(ConstraintViolationException.class);
    }

    private Taxi2020PromoCode code(String code) {
        return code(code, Instant.now());
    }

    private Taxi2020PromoCode code(String code, Instant addedAt) {
        return Taxi2020PromoCode.builder()
                .code(code)
                .addedAt(addedAt)
                .expiresAt(Instant.parse("4020-08-11T14:20:00Z"))
                .build();
    }
}
