package ru.yandex.travel.orders.repository.promo.taxi2020;

import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import javax.persistence.EntityManager;

import com.google.common.base.Preconditions;
import org.hibernate.exception.ConstraintViolationException;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.orm.jpa.DataJpaTest;
import org.springframework.data.domain.PageRequest;
import org.springframework.data.domain.Pageable;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;

import ru.yandex.travel.hotels.common.orders.OrderDetails;
import ru.yandex.travel.hotels.common.orders.TravellineHotelItinerary;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.TravellineOrderItem;
import ru.yandex.travel.orders.entities.promo.taxi2020.Taxi2020PromoCode;
import ru.yandex.travel.orders.entities.promo.taxi2020.Taxi2020PromoOrder;
import ru.yandex.travel.orders.entities.promo.taxi2020.Taxi2020PromoOrderStatus;

import static org.assertj.core.api.Assertions.assertThat;
import static org.assertj.core.api.Assertions.assertThatThrownBy;
import static ru.yandex.travel.orders.entities.promo.taxi2020.Taxi2020PromoOrderStatus.ELIGIBLE;
import static ru.yandex.travel.orders.entities.promo.taxi2020.Taxi2020PromoOrderStatus.EMAIL_SCHEDULED;
import static ru.yandex.travel.orders.entities.promo.taxi2020.Taxi2020PromoOrderStatus.EMAIL_SENT;
import static ru.yandex.travel.orders.entities.promo.taxi2020.Taxi2020PromoOrderStatus.NOT_ELIGIBLE;
import static ru.yandex.travel.orders.entities.promo.taxi2020.Taxi2020PromoOrderStatus.PROMO_CODE_CAN_BE_ASSIGNED;

@RunWith(SpringRunner.class)
@DataJpaTest
@ActiveProfiles("test")
public class Taxi2020PromoOrderRepositoryTest {
    public static final Pageable DEFAULT_LIMIT = PageRequest.of(0, 10);

    @Autowired
    private Taxi2020PromoOrderRepository repository;
    @Autowired
    private EntityManager em;

    @Before
    public void init() {
        // unrelated orders
        createOrder(uuid(101), ts("2020-08-01"), ts("2020-08-02"));
        createOrder(uuid(102), ts("2020-12-01"));

        // pending order without status changes
        createOrder(uuid(103), ts("2020-09-10"));
        createPromoOrderRef(uuid(103), ts("2020-09-10"), NOT_ELIGIBLE);
        createOrder(uuid(104), ts("2020-09-10"));
        createPromoOrderRef(uuid(104), ts("2020-09-10"), ELIGIBLE);

        // already processed orders
        createOrder(uuid(105), ts("2020-08-21"));
        createPromoOrderRef(uuid(105), ts("2020-08-21"), EMAIL_SCHEDULED);
        createOrder(uuid(106), ts("2020-08-21"));
        createPromoOrderRef(uuid(106), ts("2020-08-21"), EMAIL_SENT);
    }

    @Test
    public void findAndCountOrdersToCheck_initialDbState() {
        // updatedAt works as expected
        HotelOrder order = em.find(HotelOrder.class, uuid(101));
        assertThat(order.getUpdatedAt()).isEqualTo(ts("2020-08-01"));
        assertThat(order.getOrderItems()).hasSize(1)
                .first().satisfies(item -> assertThat(item.getUpdatedAt()).isEqualTo(ts("2020-08-02")));

        assertThat(repository.count()).isEqualTo(4);
        assertThat(repository.findOrdersToCheck(ts("2020-08-02"), DEFAULT_LIMIT)).isEqualTo(List.of());
        assertThat(repository.countOrderToCheck(ts("2020-08-02"))).isEqualTo(0);
    }

    @Test
    public void findAndCountOrdersToCheck_readyEmail() {
        createOrder(uuid(1), ts("2020-09-15"));
        createPromoOrderRef(uuid(1), ts("2020-09-15"), PROMO_CODE_CAN_BE_ASSIGNED);

        assertThat(repository.findOrdersToCheck(ts("2020-09-15"), DEFAULT_LIMIT)).hasSize(1)
                .first().satisfies(promoOrder -> assertThat(promoOrder.getOrderId()).isEqualTo(uuid(1)));
        assertThat(repository.countOrderToCheck(ts("2020-09-15"))).isEqualTo(1);
    }

    @Test
    public void findAndCountOrdersToCheck_noStatusUpdate() {
        createOrder(uuid(1), ts("2020-09-14"));
        createPromoOrderRef(uuid(1), ts("2020-09-15"), ELIGIBLE);

        assertThat(repository.findOrdersToCheck(ts("2020-09-15"), DEFAULT_LIMIT)).hasSize(0);
        assertThat(repository.countOrderToCheck(ts("2020-09-15"))).isEqualTo(0);
    }

    @Test
    public void findAndCountOrdersToCheck_statusUpdateOrder() {
        createOrder(uuid(1), ts("2020-09-16"), ts("2020-09-14"));
        createPromoOrderRef(uuid(1), ts("2020-09-15"), ELIGIBLE);

        assertThat(repository.findOrdersToCheck(ts("2020-09-16"), DEFAULT_LIMIT)).hasSize(1)
                .first().satisfies(promoOrder -> assertThat(promoOrder.getOrderId()).isEqualTo(uuid(1)));
        assertThat(repository.countOrderToCheck(ts("2020-09-16"))).isEqualTo(1);
    }

    @Test
    public void findAndCountOrdersToCheck_statusUpdateOrderItem() {
        createOrder(uuid(1), ts("2020-09-14"), ts("2020-09-16"));
        createPromoOrderRef(uuid(1), ts("2020-09-15"), NOT_ELIGIBLE);

        assertThat(repository.findOrdersToCheck(ts("2020-09-16"), DEFAULT_LIMIT)).hasSize(1)
                .first().satisfies(promoOrder -> assertThat(promoOrder.getOrderId()).isEqualTo(uuid(1)));
        assertThat(repository.countOrderToCheck(ts("2020-09-16"))).isEqualTo(1);
    }

    @Test
    public void findAndCountOrdersToCheck_eligibleBecomesReady() {
        createOrder(uuid(1), ts("2020-09-15"), ts("2020-09-15"));
        createPromoOrderRef(uuid(1), ts("2020-09-15"), ELIGIBLE, ts("2021-01-25"), null);

        // not ready yet
        assertThat(repository.findOrdersToCheck(Instant.parse("2021-01-24T23:59:59Z"), DEFAULT_LIMIT)).hasSize(0);
        assertThat(repository.countOrderToCheck(Instant.parse("2021-01-24T23:59:59Z"))).isEqualTo(0);

        // check-in day comes, can start sending emails
        assertThat(repository.findOrdersToCheck(Instant.parse("2021-01-25T00:00:00Z"), DEFAULT_LIMIT)).hasSize(1)
                .first().satisfies(promoOrder -> assertThat(promoOrder.getOrderId()).isEqualTo(uuid(1)));
        assertThat(repository.countOrderToCheck(Instant.parse("2021-01-25T00:00:00Z"))).isEqualTo(1);
    }

    @Test
    public void findAndCountOrdersToCheck_updatedButTrackingStopped() {
        createOrder(uuid(1), ts("2020-09-16"));
        createPromoOrderRef(uuid(1), ts("2020-09-15"), EMAIL_SCHEDULED);

        assertThat(repository.findOrdersToCheck(ts("2020-09-16"), DEFAULT_LIMIT)).hasSize(0);
        assertThat(repository.countOrderToCheck(ts("2020-09-16"))).isEqualTo(0);
    }

    @Test
    public void findAndCountOrdersToCheck_multipleUpdates() {
        createOrder(uuid(1), ts("2020-09-14"));
        createPromoOrderRef(uuid(1), ts("2020-09-15"), PROMO_CODE_CAN_BE_ASSIGNED);
        createOrder(uuid(2), ts("2020-09-16"));
        createPromoOrderRef(uuid(2), ts("2020-09-15"), ELIGIBLE);

        var promoOrders = repository.findOrdersToCheck(ts("2020-09-16"), DEFAULT_LIMIT);
        assertThat(promoOrders).hasSize(2);
        assertThat(promoOrders.stream().map(Taxi2020PromoOrder::getOrderId).collect(Collectors.toSet()))
                .isEqualTo(Set.of(uuid(1), uuid(2)));
        assertThat(repository.countOrderToCheck(ts("2020-09-16"))).isEqualTo(2);
    }

    @Test
    public void findAndCountOrdersToCheck_multipleUpdatesLimit() {
        createOrder(uuid(1), ts("2020-09-14"));
        createPromoOrderRef(uuid(1), ts("2020-09-15"), PROMO_CODE_CAN_BE_ASSIGNED);
        createOrder(uuid(2), ts("2020-09-16"));
        createPromoOrderRef(uuid(2), ts("2020-09-15"), ELIGIBLE);

        var promoOrders = repository.findOrdersToCheck(ts("2020-09-16"), PageRequest.of(0, 1));
        assertThat(promoOrders).hasSize(1);
        assertThat(promoOrders.stream().map(Taxi2020PromoOrder::getOrderId).collect(Collectors.toSet()))
                .containsAnyOf(uuid(1), uuid(2));
    }

    @Test
    public void orderRefUniqueness() {
        createOrder(uuid(1), null);

        createPromoOrderRef(uuid(1), ts("2020-09-01"), NOT_ELIGIBLE);
        em.flush();
        em.clear();

        // we rely on references uniqueness in business logic
        createPromoOrderRef(uuid(1), ts("2020-09-01"), ELIGIBLE);
        assertThatThrownBy(() -> em.flush())
                .hasCauseExactlyInstanceOf(ConstraintViolationException.class);
    }

    @Test
    public void promoCodeRefUniqueness() {
        createOrder(uuid(1), null);
        createOrder(uuid(2), null);
        createPromoCode("code1");
        createPromoCode("code2");

        createPromoOrderRef(uuid(1), ts("2020-09-01"), ELIGIBLE, "code1");
        em.flush();

        // we rely on references uniqueness in business logic
        createPromoOrderRef(uuid(2), ts("2020-09-01"), ELIGIBLE, "code1");
        assertThatThrownBy(() -> em.flush())
                .hasCauseExactlyInstanceOf(ConstraintViolationException.class);
    }

    private void createOrder(UUID id, Instant updatedAt) {
        createOrder(id, updatedAt, updatedAt);
    }

    private void createOrder(UUID id, Instant orderUpdatedAt, Instant orderItemUpdatedAt) {
        HotelOrder order = new HotelOrder();
        order.setId(id);
        order.setPrettyId("YA-" + id);
        order.setUpdatedAt(orderUpdatedAt);
        TravellineOrderItem orderItem = new TravellineOrderItem();
        orderItem.setOrder(order);
        orderItem.setUpdatedAt(orderItemUpdatedAt);
        TravellineHotelItinerary itinerary = new TravellineHotelItinerary();
        itinerary.setOrderDetails(OrderDetails.builder()
                .checkoutDate(LocalDate.of(1970, 1, 1).plusDays(100))
                .build());
        orderItem.setItinerary(itinerary);
        order.addOrderItem(orderItem);
        em.persist(order);

        // forcefully setting the requested updated_at values
        int updatedOrders = em.createNativeQuery("update orders set updated_at = :ts where id = :id")
                .setParameter("ts", orderUpdatedAt)
                .setParameter("id", order.getId())
                .executeUpdate();
        Preconditions.checkArgument(updatedOrders == 1, "Expected exactly 1 updated order but got %s", updatedOrders);
        int updatedItems = em.createNativeQuery("update order_items set updated_at = :ts where order_id = :id")
                .setParameter("ts", orderItemUpdatedAt)
                .setParameter("id", orderItem.getOrder().getId())
                .executeUpdate();
        Preconditions.checkArgument(updatedItems == 1, "Expected exactly 1 updated item but got %s", updatedItems);

        // forcing objects reload (including the orderItems field of the order)
        em.clear();
    }

    private void createPromoOrderRef(UUID id, Instant statusUpdatedAt, Taxi2020PromoOrderStatus status) {
        createPromoOrderRef(id, statusUpdatedAt, status, null);
    }

    private void createPromoOrderRef(UUID id, Instant statusUpdatedAt, Taxi2020PromoOrderStatus status,
                                     String promoCode) {
        createPromoOrderRef(id, statusUpdatedAt, status, null, promoCode);
    }

    private void createPromoOrderRef(UUID id, Instant statusUpdatedAt, Taxi2020PromoOrderStatus status,
                                     Instant emailScheduledAt, String promoCode) {
        Taxi2020PromoOrder orderRef = new Taxi2020PromoOrder();
        orderRef.setOrderId(id);
        orderRef.setStatus(status);
        orderRef.setStatusUpdatedAt(statusUpdatedAt);
        orderRef.setEmailScheduledAt(emailScheduledAt);
        orderRef.setPromoCode(promoCode);
        em.persist(orderRef);
    }

    private void createPromoCode(String code) {
        em.persist(Taxi2020PromoCode.builder()
                .code(code)
                .addedAt(Instant.now())
                .expiresAt(Instant.now().plus(Duration.ofDays(180)))
                .build());
    }

    private static UUID uuid(int id) {
        return UUID.fromString("0-0-0-0-" + id);
    }

    private static Instant ts(String date) {
        return Instant.parse(date + "T00:00:00Z");
    }
}
