package ru.yandex.travel.orders.services;

import java.util.UUID;

import org.assertj.core.api.Assertions;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.jdbc.AutoConfigureTestDatabase;
import org.springframework.boot.test.autoconfigure.orm.jpa.DataJpaTest;
import org.springframework.boot.test.context.TestConfiguration;
import org.springframework.context.annotation.Bean;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.entities.AuthorizedUser;
import ru.yandex.travel.orders.repository.AuthorizedUserRepository;
import ru.yandex.travel.orders.repository.UserOrderCounterRepository;
import ru.yandex.travel.orders.services.promo.UserOrderCounterService;


@RunWith(SpringRunner.class)
@DataJpaTest
@AutoConfigureTestDatabase(replace = AutoConfigureTestDatabase.Replace.ANY)
@ActiveProfiles("test")
@Transactional
public class UserOrderCounterServiceTest {

    @Autowired
    private UserOrderCounterService userOrderCounterService;

    @Autowired
    private AuthorizedUserRepository authorizedUserRepository;

    private long passportId;

    private UserCredentials userCredentials;

    private UUID orderId;

    @Before
    public void setUp() {
        passportId = System.currentTimeMillis();
        userCredentials = new UserCredentials("", "", Long.toString(passportId), "", "", "",
                false, false);

        orderId = UUID.randomUUID();

        authorizedUserRepository.save(AuthorizedUser.create(orderId, userCredentials,
                AuthorizedUser.OrderUserRole.OWNER));

    }

    @Test
    public void testOrderConfirmedUserExists() {
        int ordersCount = userOrderCounterService.getNumberOfHotelOrdersConfirmedByUser(passportId);

        Assertions.assertThat(ordersCount).isEqualTo(0);

        userOrderCounterService.onOrderConfirmed(orderId, EDisplayOrderType.DT_HOTEL);

        ordersCount = userOrderCounterService.getNumberOfHotelOrdersConfirmedByUser(passportId);
        Assertions.assertThat(ordersCount).isEqualTo(1);

        userOrderCounterService.onOrderRefunded(orderId, EDisplayOrderType.DT_HOTEL);
        ordersCount = userOrderCounterService.getNumberOfOrdersConfirmedByUser(passportId, EDisplayOrderType.DT_HOTEL);
        Assertions.assertThat(ordersCount).isEqualTo(0);
    }

    @Test
    public void testOrderConfirmedUserExistsTrain() {
        userOrderCounterService.onOrderConfirmed(orderId, EDisplayOrderType.DT_TRAIN);

        Assertions.assertThat(
                userOrderCounterService.getNumberOfOrdersConfirmedByUser(passportId, EDisplayOrderType.DT_TRAIN))
                .isEqualTo(1);
    }

    @TestConfiguration
    static class IntegrationTestConfiguration {
        @Bean
        public UserOrderCounterService service(UserOrderCounterRepository counterRepo,
                                               AuthorizedUserRepository userRepo) {
            return new UserOrderCounterService(counterRepo, userRepo);
        }
    }

}
