package ru.yandex.travel.orders.services.cloud.s3;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.concurrent.TimeUnit;

import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.model.Bucket;
import com.amazonaws.services.s3.model.ListObjectsRequest;
import com.amazonaws.services.s3.model.ObjectListing;
import com.amazonaws.services.s3.model.S3ObjectSummary;
import com.google.common.base.Stopwatch;
import lombok.extern.slf4j.Slf4j;
import org.junit.After;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;
import org.springframework.http.MediaType;

import ru.yandex.travel.orders.configurations.S3ConfigurationLocalTest;
import ru.yandex.travel.orders.configurations.TvmConfigurationLocalTest;
import ru.yandex.travel.testing.TestUtils;
import ru.yandex.travel.tvm.TvmWrapper;

import static org.assertj.core.api.Assertions.assertThat;

@Slf4j
@Ignore
public class S3ServiceImplLocalTest {
    public static final String DEV_BUCKET_NAME = "order-service-dev";

    private TvmWrapper tvm;
    private AmazonS3 s3;
    private S3ServiceImpl service;

    @Before
    public void init() {
        tvm = TvmConfigurationLocalTest.createDefaultManualTestsTvmClient();
        s3 = S3ConfigurationLocalTest.createManualTestsClientWithTvm(tvm, "s3_prod");
        service = new S3ServiceImpl(S3ServiceProperties.builder()
                .bucket(DEV_BUCKET_NAME)
                .maxInMemoryFileSize(512 * 1024 * 1024/*512Mb*/)
                .build(), s3);
        // the test results can be checked here:
        // https://yc.yandex-team.ru/folders/fook6nom9g5ph8gulcfu/storage/bucket/order-service-dev
    }

    @After
    public void destroy() {
        s3.shutdown();
        tvm.close();
    }

    @Test
    public void readWrite() {
        String testFileKey = "manual-tests/readWrite_test.txt";

        log.info("Test file already exists: {}", service.checkObjectExists(testFileKey));
        log.info("Fake test file already exists: {}", service.checkObjectExists(testFileKey + ".fake"));

        String fileData = "Some text data: " + LocalDateTime.now();
        service.uploadObject(testTextFile(testFileKey, "File1.txt", fileData));

        InMemoryS3Object object = service.readObject(testFileKey);
        String restoredText = new String(object.getData());
        log.info("Restored content: {}", restoredText);
        assertThat(restoredText).isEqualTo(fileData);
        assertThat(object.getId()).isEqualTo(testFileKey);
        assertThat(object.getMimeType()).isEqualTo(MediaType.TEXT_PLAIN_VALUE);
        assertThat(object.getFileName()).isEqualTo("File1.txt");

        log.info("Test file should exist now: {}", service.checkObjectExists(testFileKey));
    }

    @Test
    public void readWrite_cyrillic() {
        String key = "manual-tests/Контрольный купон.pdf.647f4550-3253-44a4-aed6-1ab46d1d9b8e";
        service.uploadObject(mockPdf(key, "Контрольный купон.pdf", "some data"));
        assertThat(service.readObject(key).getFileName()).isEqualTo("Контрольный купон.pdf");
    }

    @Test
    public void sampleAttachmentsStructure() {
        service.uploadObject(mockPdf("attachments/fiscal_receipts/0-0-0-0-1", "FR1.pdf", "pdf data 1"));
        service.uploadObject(mockPdf("attachments/fiscal_receipts/0-0-0-0-2", "FR2.pdf", "pdf data 2"));
        service.uploadObject(mockPdf("attachments/hotel_vouchers/0-0-0-0-3", "Voucher 1.pdf", "voucher info 1"));
        service.uploadObject(mockPdf("attachments/hotel_vouchers/0-0-0-0-4", "Voucher 2.pdf", "voucher info 2"));
        service.uploadObject(mockPdf("attachments/im_blanks/0-0-0-0-5", "Blank 1.pdf", "im blank data"));
        service.uploadObject(mockPdf("attachments/movista_coupons/0-0-0-0-6", "Blank 2.pdf", "movista coupon data"));
    }

    @Test
    public void exceptionsTest() {
        for (int i = 0; i < 1000; i++) {
            try {
                TestUtils.sleep(Duration.ofSeconds(5));
                log.info("Call start");
                service.checkObjectExists("some.file");
                log.info("done");
            } catch (Exception e) {
                log.info("got {}: {}", e.getClass(), e.getMessage(), e);
            }
        }
    }

    @Test
    public void bigFileUpload() {
        int oneMbBytes = 1024 * 1024;
        byte[] lotsOfZeroes = new byte[256 * oneMbBytes];
        Stopwatch sw = Stopwatch.createStarted();
        log.info("Starting upload");
        service.uploadObject(new InMemoryS3Object("manual-tests/big-file.bin",
                MediaType.APPLICATION_OCTET_STREAM_VALUE, "Some Big File.bin", lotsOfZeroes));
        log.info("Finished upload in {} seconds", sw.elapsed(TimeUnit.SECONDS));
    }

    @Test
    public void bigFileRead() {
        Stopwatch sw = Stopwatch.createStarted();
        log.info("Starting read");
        InMemoryS3Object object = service.readObject("manual-tests/big-file.bin");
        log.info("Finished read in {} seconds, read {} bytes", sw.elapsed(TimeUnit.SECONDS), object.getData().length);
    }

    // ========== MISC CLIENT TESTS ==========

    @Test
    public void listAccessibleBuckets() {
        log.info("Buckets:");
        for (Bucket bucket : s3.listBuckets()) {
            log.info("{}", bucket);
        }
    }

    @Test
    public void listObjectsFlat() {
        ObjectListing result = s3.listObjects(DEV_BUCKET_NAME);
        log.info("Objects:");
        for (S3ObjectSummary summary : result.getObjectSummaries()) {
            log.info("{}", summary.getKey());
        }
    }

    @Test
    public void listObjectsWithGrouping() {
        ObjectListing result = s3.listObjects(new ListObjectsRequest()
                .withBucketName(DEV_BUCKET_NAME)
                .withDelimiter("/"));

        log.info("Common prefixes:");
        for (String prefix : result.getCommonPrefixes()) {
            log.info("    {}", prefix);
        }
        log.info("Objects:");
        for (S3ObjectSummary summary : result.getObjectSummaries()) {
            log.info("    {}", summary.getKey());
        }
    }

    private static InMemoryS3Object mockPdf(String id, String name, String data) {
        return new InMemoryS3Object(id, MediaType.APPLICATION_PDF_VALUE, name, data.getBytes());
    }

    private static InMemoryS3Object testTextFile(String id, String name, String data) {
        return new InMemoryS3Object(id, MediaType.TEXT_PLAIN_VALUE, name, data.getBytes());
    }
}
