package ru.yandex.travel.orders.services.email;

import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.atomic.AtomicInteger;

import com.fasterxml.jackson.databind.JsonNode;
import lombok.AllArgsConstructor;
import lombok.Getter;
import lombok.extern.slf4j.Slf4j;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.jdbc.AutoConfigureTestDatabase;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.context.annotation.Import;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;

import ru.yandex.travel.orders.entities.notifications.MailTarget;
import ru.yandex.travel.orders.services.MailSenderService;
import ru.yandex.travel.orders.test.CommonIntegrationTestConfiguration;
import ru.yandex.travel.orders.test.TestsTxHelper;
import ru.yandex.travel.testing.TestUtils;
import ru.yandex.travel.workflow.single_operation.SingleOperationService;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.doAnswer;
import static org.mockito.Mockito.verify;

@RunWith(SpringRunner.class)
@SpringBootTest(
        webEnvironment = SpringBootTest.WebEnvironment.NONE,
        properties = {
                // enabling WorkflowProcessService
                "quartz.enabled=true",
                "workflow-processing.pending-workflow-polling-interval=100ms",
                "single-node.auto-start=true",
        }
)
@ActiveProfiles("test")
@DirtiesContext
@AutoConfigureTestDatabase(replace = AutoConfigureTestDatabase.Replace.ANY)
@Import({CommonIntegrationTestConfiguration.class})
@Slf4j
public class AsyncEmailOperationIntegrationTest {
    @MockBean
    private MailSenderService sender;
    @Autowired
    private SingleOperationService operations;
    @Autowired
    private TestsTxHelper txHelper;

    @Test
    public void testConfiguredAndWorks() {
        AtomicInteger senderCalls = new AtomicInteger(0);
        doAnswer(inv -> senderCalls.incrementAndGet())
                .when(sender).sendEmailSync(any(), any(), any(), any(), any(), any(), any());

        TestData args = new TestData("123", 432);
        SendEmailParams emailData = SendEmailParams.builder()
                .campaignId("campaignId")
                .targets(List.of(new MailTarget("test@example.com")))
                .arguments(SendEmailParams.wrapArgs(args))
                .contextEntityId(UUID.randomUUID())
                .build();
        txHelper.runInTx(() -> {
            operations.runOperation(
                    "AeroflotPlusPromoSendEmail_9836462942",
                    AsyncEmailOperation.TYPE.getValue(),
                    emailData
            );
        });

        TestUtils.waitForState("Email sender client called", () -> senderCalls.get() == 1);

        JsonNode deserializedArgs = SendEmailParams.wrapArgs(Map.of("some_text", "123", "some_num", 432));
        verify(sender).sendEmailSync(eq(emailData.getCampaignId()), eq(null), eq(emailData.getTargets()),
                eq(null), eq(null), eq(deserializedArgs), eq(null));
    }

    @AllArgsConstructor
    @Getter
    static class TestData {
        private final String someText;
        private final Integer someNum;
    }
}
