package ru.yandex.travel.orders.services.finances;

import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.Callable;
import java.util.concurrent.CopyOnWriteArrayList;

import javax.persistence.EntityManager;

import com.fasterxml.jackson.databind.JsonNode;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.Answers;
import org.mockito.Mockito;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.test.autoconfigure.jdbc.AutoConfigureTestDatabase;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.context.TestConfiguration;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.boot.test.mock.mockito.SpyBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Import;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;

import ru.yandex.bolts.collection.IteratorF;
import ru.yandex.inside.yt.kosher.Yt;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.tables.YtTables;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.hotels.administrator.export.proto.HotelAgreement;
import ru.yandex.travel.hotels.common.orders.OrderDetails;
import ru.yandex.travel.hotels.common.orders.TravellineHotelItinerary;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.integration.balance.BillingApiClient;
import ru.yandex.travel.integration.balance.BillingClientContract;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.commons.proto.EVat;
import ru.yandex.travel.orders.entities.FiscalItem;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.TravellineOrderItem;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.entities.YandexPlusTopup;
import ru.yandex.travel.orders.entities.finances.BillingTransaction;
import ru.yandex.travel.orders.entities.finances.FinancialEvent;
import ru.yandex.travel.orders.entities.finances.FinancialEventType;
import ru.yandex.travel.orders.entities.partners.BillingPartnerConfig;
import ru.yandex.travel.orders.repository.BillingPartnerConfigRepository;
import ru.yandex.travel.orders.repository.BillingTransactionRepository;
import ru.yandex.travel.orders.repository.FinancialEventRepository;
import ru.yandex.travel.orders.services.finances.billing.BillingPartnerAgreementSynchronizer;
import ru.yandex.travel.orders.services.finances.billing.BillingTransactionActCommitter;
import ru.yandex.travel.orders.services.finances.billing.BillingTransactionYtIdGenerator;
import ru.yandex.travel.orders.services.finances.billing.BillingTransactionYtTableClientProperties;
import ru.yandex.travel.orders.services.payments.PaymentProfile;
import ru.yandex.travel.orders.test.CommonIntegrationTestConfiguration;
import ru.yandex.travel.orders.test.TestsTxHelper;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.orders.workflow.hotels.travelline.proto.ETravellineItemState;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.spring.tx.ForcedRollbackTxManagerWrapper;
import ru.yandex.travel.task_processor.TaskProcessor;
import ru.yandex.travel.testing.time.SettableClock;
import ru.yandex.travel.utils.ClockService;

import static java.util.stream.Collectors.toList;
import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyBoolean;
import static org.mockito.ArgumentMatchers.anyInt;
import static org.mockito.ArgumentMatchers.anyLong;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.doAnswer;
import static org.mockito.Mockito.when;
import static ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentSystemType.YANDEX_MONEY;
import static ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentType.COST;
import static ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentType.REWARD;
import static ru.yandex.travel.orders.integration.IntegrationUtils.waitForPredicateOrTimeout;
import static ru.yandex.travel.testing.TestUtils.waitForState;
import static ru.yandex.travel.testing.misc.MockitoUtils.getMockInvocations;
import static ru.yandex.travel.testing.misc.MockitoUtils.waitForMockCalls;
import static ru.yandex.travel.testing.misc.TestBaseObjects.rub;
import static ru.yandex.travel.testing.spring.SpringUtils.unwrapAopProxy;

// TODO (tlg-13,mbobrov): rework this test to remove provider specific logic to local mock

@RunWith(SpringRunner.class)
@SpringBootTest(
        webEnvironment = SpringBootTest.WebEnvironment.NONE,
        properties = {
                "financial-events.enabled=true",
                "billing-transactions.generator-task.enabled=true",
                "billing-transactions.generator-task.schedule-rate=100ms",
                "billing-transactions.yt-id-generator-task.enabled=true",
                "billing-transactions.yt-id-generator-task.schedule-rate=100ms",
                "billing-transactions.export-task.enabled=true",
                "billing-transactions.export-task.schedule-rate=100ms",
                "billing-transactions.act-commit-task.enabled=true",
                "billing-transactions.act-commit-task.schedule-rate=100ms",
                "billing-partners.agreement-sync-task.enabled=true",
                "billing-partners.agreement-sync-task.schedule-rate=100ms",
        }
)
//@TestExecutionListeners(listeners = TruncateDatabaseTestExecutionListener.class, mergeMode =
//        TestExecutionListeners.MergeMode.MERGE_WITH_DEFAULTS)
@ActiveProfiles("test")
@Import({CommonIntegrationTestConfiguration.class})
@AutoConfigureTestDatabase(replace = AutoConfigureTestDatabase.Replace.ANY)
@DirtiesContext(classMode = DirtiesContext.ClassMode.BEFORE_CLASS)
@Slf4j
public class FinancialEventsIntegrationTest {
    private static final long TEST_BILLING_CLIENT_ID = 98629423L;

    @Autowired
    private TestsTxHelper txHelper;
    @Autowired
    private EntityManager em;
    @Autowired
    private ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper;
    @Autowired
    private SettableClock settableClock;

    @Autowired
    @Qualifier("billingTransactionGeneratorTaskProcessor")
    private TaskProcessor<Long> billingTransactionGeneratorTaskProcessor;
    @Autowired
    @Qualifier("billingTransactionYtIdGeneratorTaskProcessor")
    private TaskProcessor<String> billingTransactionYtIdGeneratorTaskProcessor;
    @Autowired
    @Qualifier("billingTransactionExportTaskProcessor")
    private TaskProcessor<String> billingTransactionExportTaskProcessor;
    @Autowired
    @Qualifier("billingTransactionActCommitTaskProcessor")
    private TaskProcessor<Long> billingTransactionActCommitTaskProcessor;
    @Autowired
    @Qualifier("billingPartnerAgreementSynchronizerTaskProcessor")
    private TaskProcessor<Long> billingPartnerAgreementSynchronizerTaskProcessor;

    @Autowired
    private FinancialEventRepository financialEventRepository;
    @Autowired
    private BillingTransactionRepository billingTransactionRepository;
    @Autowired
    private BillingPartnerConfigRepository billingPartnerConfigRepository;
    @Autowired
    private FinancialEventService financialEventService;
    @SpyBean
    private BillingTransactionYtIdGenerator billingTransactionYtIdGenerator;
    @SpyBean
    private BillingTransactionActCommitter billingTransactionActCommitter;
    @SpyBean
    private BillingPartnerAgreementSynchronizer billingPartnerAgreementSynchronizer;

    @MockBean(answer = Answers.RETURNS_DEEP_STUBS)
    private Yt billingTransactionExporterYtClient;
    @MockBean
    private BillingApiClient billingApiClient;

    @Before
    public void init() {
        // Spring AOP + Mockito = <3, https://github.com/spring-projects/spring-boot/issues/5837
        billingTransactionYtIdGenerator = unwrapAopProxy(billingTransactionYtIdGenerator);
        billingTransactionActCommitter = unwrapAopProxy(billingTransactionActCommitter);
        billingPartnerAgreementSynchronizer = unwrapAopProxy(billingPartnerAgreementSynchronizer);

        // preventing standard partners synchronization (just in case, not to interfere with other integration tests)
        runInTx(() -> billingPartnerConfigRepository.findAll()
                .forEach(partnerConfig -> {
                    partnerConfig.setSynchronizeAgreement(false);
                    billingPartnerConfigRepository.save(partnerConfig);
                }));

        // There is no need to start a full Master node + Quartz
        forcedRollbackTxManagerWrapper.resumeCommits();
        billingTransactionGeneratorTaskProcessor.resume();
        billingTransactionYtIdGeneratorTaskProcessor.resume();
        billingTransactionExportTaskProcessor.resume();
        billingTransactionActCommitTaskProcessor.resume();
        billingPartnerAgreementSynchronizerTaskProcessor.resume();
    }

    @After
    public void tearDown() {
        billingPartnerAgreementSynchronizerTaskProcessor.pause();
        billingTransactionActCommitTaskProcessor.pause();
        billingTransactionExportTaskProcessor.pause();
        billingTransactionYtIdGeneratorTaskProcessor.pause();
        billingTransactionGeneratorTaskProcessor.pause();
    }

    @Test
    public void testFinancialEventProcessingFlow() {
        settableClock.setCurrentTime(Instant.parse("2019-12-18T10:03:07Z"));
        Duration timeout = Duration.ofSeconds(15);
        Duration retryDelay = Duration.ofMillis(50);

        // Stage 1: Financial event generation (from the main order workflow)
        runInTx(this::createTestPartnerConfig);
        OrderItem orderItem = callInTx(this::createTestOrderItem);
        runInTx(() -> assertThat(financialEventRepository.findFirstByOrderItemAndTypeOrderByIdAsc(
                orderItem, FinancialEventType.PAYMENT)).isNull());

        runInTx(() -> financialEventService.registerConfirmedService(orderItem));
        FinancialEvent fe = callInTx(() -> financialEventRepository.findFirstByOrderItemAndTypeOrderByIdAsc(
                orderItem, FinancialEventType.PAYMENT));
        assertThat(fe).isNotNull();

        // Stage 2: Billing transactions generation
        waitForPredicateOrTimeout(() -> callInTx(() -> financialEventRepository.getOne(fe.getId()).isProcessed()),
                timeout, retryDelay, "Billing Transactions generation");
        BillingTransaction costTx = callInTx(() -> billingTransactionRepository
                .findBySourceFinancialEventAndPaymentTypeAndPaymentSystemType(fe, COST, YANDEX_MONEY));
        BillingTransaction rewardTx = callInTx(() -> billingTransactionRepository
                .findBySourceFinancialEventAndPaymentTypeAndPaymentSystemType(fe, REWARD, YANDEX_MONEY));
        assertThat(costTx).isNotNull();
        assertThat(rewardTx).isNotNull();

        // Stage 3.1: YT export - YT IDs generation
        waitForMockCalls(billingTransactionYtIdGenerator, g -> g.getYtIdBulkGenerationTaskIds(any()), 2,
                timeout, retryDelay, "Getting transaction ready for YT ID generation");
        assertThat(callInTx(() -> billingTransactionRepository.getOne(costTx.getId()).getYtId())).isNull();

        List<JsonNode> exportedObjects = initYtWriteTransactionsMocks(orderItem.getOrder().getPrettyId());
        assertThat(costTx.getPayoutAt()).isAfter(Instant.parse("2019-12-20T20:00:00Z"));
        settableClock.setCurrentTime(Instant.parse("2019-12-21T21:00:00.01Z"));

        waitForPredicateOrTimeout(() -> callInTx(() -> billingTransactionRepository
                        .findAllById(List.of(costTx.getId(), rewardTx.getId()))
                        .stream().map(BillingTransaction::getYtId).allMatch(Objects::nonNull)),
                timeout, retryDelay, "Billing Transaction YT IDs generation");
        long costTxYtId = callInTx(() -> billingTransactionRepository.getOne(costTx.getId()).getYtId());
        long rewardTxYtId = callInTx(() -> billingTransactionRepository.getOne(rewardTx.getId()).getYtId());
        assertThat(costTxYtId).isNotEqualTo(rewardTxYtId);

        // Stage 3.2: YT export - writing transactions to YT table
        waitForPredicateOrTimeout(() -> callInTx(() -> billingTransactionRepository.getOne(costTx.getId()).isExportedToYt()),
                timeout, retryDelay, "Billing Transactions export");
        assertThat(exportedObjects).anyMatch(txJson -> txJson.get("transaction_id").asLong(-1) == costTxYtId);
        assertThat(exportedObjects).anyMatch(txJson -> txJson.get("transaction_id").asLong(-1) == rewardTxYtId);
        assertThat(callInTx(() -> billingTransactionRepository.findAllById(List.of(costTx.getId(), rewardTx.getId()))))
                .hasSize(2)
                .allMatch(BillingTransaction::isExportedToYt)
                .allMatch(tx -> tx.getExportedToYtAt() != null);

        // Stage 4: Act generation
        waitForMockCalls(billingTransactionActCommitter, c -> c.getTransactionIdsWaitingForCommit(any(), any()), 2,
                timeout, retryDelay, "Act Committer idle runs");
        assertThat(callInTx(() -> billingTransactionRepository.findAllById(List.of(costTx.getId(), rewardTx.getId()))))
                .hasSize(2)
                .noneMatch(BillingTransaction::isActCommitted)
                .noneMatch(tx -> tx.getActCommittedAt() != null);

        List<Long> committedIds = initBillingApiMocks();
        assertThat(costTx.getAccountingActAt()).isAfter(Instant.parse("2020-03-06T20:00:00.01Z"));
        settableClock.setCurrentTime(Instant.parse("2020-03-06T21:00:00.01Z"));

        waitForPredicateOrTimeout(() -> committedIds.containsAll(callInTx(() ->
                        List.of(
                                billingTransactionRepository.getOne(costTx.getId()).getYtId(),
                                billingTransactionRepository.getOne(rewardTx.getId()).getYtId()
                        ))),
                timeout, retryDelay, "Billing Transaction Act generated");
        assertThat(callInTx(() -> billingTransactionRepository.findAllById(List.of(costTx.getId(), rewardTx.getId()))))
                .hasSize(2)
                .allMatch(BillingTransaction::isActCommitted)
                .allMatch(tx -> tx.getActCommittedAt() != null);
    }

    @Test
    public void testPlusTopupFinancialEventsFlow() {
        settableClock.setCurrentTime(Instant.parse("2021-08-31T23:03:07Z"));

        // Stage 0: Mock data
        runInTx(() -> createTestPartnerConfig(7521341, false));
        OrderItem orderItem = callInTx(this::createTestOrderItem);
        String prettyId = orderItem.getOrder().getPrettyId();

        YandexPlusTopup topup = YandexPlusTopup.builder()
                .orderItem(orderItem)
                // tmp migration test, will be changed to 'now' soon
                .authorizedAt(settableClock.instant().minus(Duration.ofDays(7)))
                .amount(300)
                .currency(ProtoCurrencyUnit.RUB)
                .trustPaymentId("trust-payment-id-7826342")
                .build();

        // Stage 1: Financial event generation (from the async topup workflow)
        List<JsonNode> exportedObjects = initYtWriteTransactionsMocks(prettyId);
        runInTx(() -> financialEventService.registerPlusPointsTopup(topup));

        List<FinancialEvent> events = callInTx(() -> financialEventRepository.findAllByOrderItem(orderItem)).stream()
                .filter(e -> e.getType() == FinancialEventType.YANDEX_ACCOUNT_TOPUP_PAYMENT)
                .collect(toList());
        assertThat(events).hasSize(1);
        FinancialEvent fe = events.get(0);
        assertThat(fe.getPayoutAt()).isEqualTo("2021-08-24T23:03:07Z");
        assertThat(fe.getAccountingActAt()).isEqualTo("2021-08-24T23:03:07Z");

        // Stage 2: Billing transactions generation
        waitForState("Billing Transactions generation",
                () -> callInTx(() -> financialEventRepository.getOne(fe.getId()).isProcessed()));
        List<BillingTransaction> transactions = callInTx(() -> billingTransactionRepository
                .findAllByServiceOrderId(prettyId));
        assertThat(transactions).hasSize(1);
        BillingTransaction tx = transactions.get(0);

        // Stage 3: YT export
        settableClock.setCurrentTime(Instant.parse("2021-08-31T23:03:08Z"));
        waitForState("Billing Transactions export",
                () -> callInTx(() -> billingTransactionRepository.getOne(tx.getId()).isExportedToYt()));

        assertThat(exportedObjects).hasSize(1);
        JsonNode txJson = exportedObjects.get(0);
        assertThat(txJson.get("service_id").longValue()).isEqualTo(641);
        assertThat(txJson.get("transaction_type").textValue()).isEqualTo("payment");
        assertThat(txJson.get("payment_type").textValue()).isEqualTo("yandex_account_topup");
        assertThat(txJson.get("paysys_type_cc").textValue()).isEqualTo("promocode");
        assertThat(txJson.get("partner_id").longValue()).isEqualTo(95227453);
        assertThat(txJson.get("dt").textValue()).isEqualTo("2021-09-01T02:03:07");
        assertThat(txJson.get("price").textValue()).isEqualTo("300.00");
        assertThat(txJson.get("currency").textValue()).isEqualTo("RUB");
        assertThat(txJson.get("trust_payment_id").textValue()).isEqualTo("trust-payment-id-7826342");

        // no need to check other details checked by the main events flow test above (yt ids, act commit, etc)
    }

    @Test
    public void testExternalPlusTopupFinancialEventsFlow() {
        settableClock.setCurrentTime(Instant.parse("2021-08-31T23:03:07Z"));

        // Stage 0: Mock data
        runInTx(() -> createTestPartnerConfig(7521342, false));
        String externalOrderId = "some-order-id-" + UUID.randomUUID().toString();
        String trustPaymentId = UUID.randomUUID().toString();

        YandexPlusTopup topup = YandexPlusTopup.builder()
                .externalOrderId(externalOrderId)
                .paymentProfile(PaymentProfile.HOTEL)
                .authorizedAt(settableClock.instant())
                .amount(300)
                .currency(ProtoCurrencyUnit.RUB)
                .trustPaymentId(trustPaymentId)
                .build();

        // Stage 1: Financial event generation (from the async topup workflow)
        List<JsonNode> exportedObjects = initYtWriteTransactionsMocks(externalOrderId);
        runInTx(() -> financialEventService.registerPlusPointsTopup(topup));

        List<FinancialEvent> events = callInTx(() -> financialEventRepository.findAll().stream()
                .filter(x -> x.getOrderPrettyId().equals(externalOrderId))
                .filter(e -> e.getType() == FinancialEventType.YANDEX_ACCOUNT_TOPUP_PAYMENT)
                .collect(toList()));
        assertThat(events).hasSize(1);
        FinancialEvent fe = events.get(0);
        assertThat(fe.getPayoutAt()).isEqualTo("2021-08-31T23:03:07Z");
        assertThat(fe.getAccountingActAt()).isEqualTo("2021-08-31T23:03:07Z");

        // Stage 2: Billing transactions generation
        waitForState("Billing Transactions generation",
                () -> callInTx(() -> financialEventRepository.getOne(fe.getId()).isProcessed()));
        List<BillingTransaction> transactions = callInTx(() -> billingTransactionRepository
                .findAllByServiceOrderId(externalOrderId));
        assertThat(transactions).hasSize(1);
        BillingTransaction tx = transactions.get(0);

        // Stage 3: YT export
        settableClock.setCurrentTime(Instant.parse("2021-08-31T23:03:08Z"));
        waitForState("Billing Transactions export",
                () -> callInTx(() -> billingTransactionRepository.getOne(tx.getId()).isExportedToYt()));

        assertThat(exportedObjects).hasSize(1);
        JsonNode txJson = exportedObjects.get(0);
        assertThat(txJson.get("service_id").longValue()).isEqualTo(641);
        assertThat(txJson.get("transaction_type").textValue()).isEqualTo("payment");
        assertThat(txJson.get("payment_type").textValue()).isEqualTo("yandex_account_topup");
        assertThat(txJson.get("paysys_type_cc").textValue()).isEqualTo("promocode");
        assertThat(txJson.get("partner_id").longValue()).isEqualTo(95227453);
        assertThat(txJson.get("dt").textValue()).isEqualTo("2021-09-01T02:03:07");
        assertThat(txJson.get("price").textValue()).isEqualTo("300.00");
        assertThat(txJson.get("currency").textValue()).isEqualTo("RUB");
        assertThat(txJson.get("trust_payment_id").textValue()).isEqualTo(trustPaymentId);

        // no need to check other details checked by the main events flow test above (yt ids, act commit, etc)
    }

    @Test
    public void testBillingPartnerAgreementSyncFlow() {
        long testClientId = 842581634921414L;
        Duration timeout = Duration.ofSeconds(3);
        Duration retryDelay = Duration.ofMillis(50);

        settableClock.setCurrentTime(Instant.parse("2020-01-27T13:00:00Z"));
        int taskCalls = getMockInvocations(billingPartnerAgreementSynchronizer, c -> c.processTask(eq(testClientId)));
        BillingPartnerConfig partnerConfig = callInTx(() -> createTestPartnerConfig(testClientId, true));
        assertThat(partnerConfig.isAgreementActive()).isTrue();

        // the new active partner will be disabled by the synchronizer job as there are no active (stub) agreements
        taskCalls = waitForMockCalls(billingPartnerAgreementSynchronizer, c -> c.processTask(eq(testClientId)),
                taskCalls, 1, timeout, retryDelay, "Sync call #1");
        waitForMockCalls(billingPartnerAgreementSynchronizer, c -> c.getReadyTasks(any(), anyInt()),
                2, timeout, retryDelay, "Synchronizer idle runs");
        assertThat(callInTx(() -> billingPartnerConfigRepository.getOne(testClientId).isAgreementActive())).isFalse();

        // re-activating the partner with a new active agreement
        when(billingApiClient.getClientContracts(eq(testClientId))).thenReturn(
                List.of(BillingClientContract.builder().active(true).build()));

        settableClock.setCurrentTime(Instant.parse("2020-01-27T15:00:00Z"));
        waitForMockCalls(billingPartnerAgreementSynchronizer, c -> c.processTask(eq(testClientId)),
                taskCalls, 1, timeout, retryDelay, "Sync call #2");
        waitForMockCalls(billingPartnerAgreementSynchronizer, c -> c.getReadyTasks(any(), anyInt()),
                2, timeout, retryDelay, "Synchronizer idle runs");
        assertThat(callInTx(() -> billingPartnerConfigRepository.getOne(testClientId).isAgreementActive())).isTrue();
    }

    private List<JsonNode> initYtWriteTransactionsMocks(String orderPrettyId) {
        List<JsonNode> exportedTransactions = new CopyOnWriteArrayList<>();
        YtTables mockTables = Mockito.mock(YtTables.class);
        when(billingTransactionExporterYtClient.tables()).thenReturn(mockTables);
        doAnswer(invocation -> {
            IteratorF<JsonNode> records = invocation.getArgument(4);
            exportedTransactions.addAll(records.toList().stream()
                    .filter(tx -> tx.get("service_order_id").textValue().equals(orderPrettyId))
                    .collect(toList()));
            return null;
        }).when(mockTables).write(anyGuid(), anyBoolean(), any(), any(), any());
        return exportedTransactions;
    }

    private List<Long> initBillingApiMocks() {
        List<Long> committedTransactionIds = new CopyOnWriteArrayList<>();
        doAnswer(invocation -> {
            committedTransactionIds.add(invocation.getArgument(1));
            return null;
        }).when(billingApiClient).updatePayment(anyLong(), anyLong(), any());
        return committedTransactionIds;
    }

    private Optional<GUID> anyGuid() {
        return any();
    }

    private void runInTx(Runnable r) {
        txHelper.runInTx(r);
    }

    private <T> T callInTx(Callable<T> r) {
        return txHelper.callInTx(r);
    }

    private void createTestPartnerConfig() {
        createTestPartnerConfig(TEST_BILLING_CLIENT_ID, false);
    }

    private BillingPartnerConfig createTestPartnerConfig(long clientId, boolean syncAgreement) {
        BillingPartnerConfig config = BillingPartnerConfig.builder()
                .billingClientId(clientId)
                .agreementActive(true)
                .generateTransactions(true)
                .exportToYt(true)
                .synchronizeAgreement(syncAgreement)
                .build();
        em.persist(config);
        return config;
    }

    private OrderItem createTestOrderItem() {
        TravellineOrderItem orderItem = new TravellineOrderItem();
        orderItem.setId(UUID.randomUUID());
        orderItem.setState(ETravellineItemState.IS_CONFIRMED);
        orderItem.setConfirmedAt(Instant.now(settableClock));
        orderItem.setAgreement(HotelAgreement.newBuilder()
                .setId(1L)
                .setHotelId("1")
                .setPartnerId(EPartnerId.PI_TRAVELLINE)
                .setInn("")
                .setFinancialClientId(TEST_BILLING_CLIENT_ID)
                .setFinancialContractId(745238462L)
                .setOrderConfirmedRate("0.1")
                .setOrderRefundedRate("0.1")
                .setAgreementStartDate(0)
                .setEnabled(true)
                .setVatType(EVat.VAT_NONE)
                .setSendEmptyOrdersReport(true)
                .build());
        orderItem.setItinerary(testItinerary(rub(10_000)));
        orderItem.addFiscalItem(FiscalItem.builder()
                .moneyAmount(rub(10_000))
                .build());

        TrustInvoice invoice = new TrustInvoice();
        invoice.setId(UUID.randomUUID());
        invoice.setState(ETrustInvoiceState.IS_NEW);
        invoice.setTrustPaymentId("9127547861532612846912");
        // the order doesn't cascade operations to this property
        em.persist(invoice);

        HotelOrder order = new HotelOrder();
        order.setId(UUID.randomUUID());
        order.setPrettyId("YA-" + UUID.randomUUID());
        order.setState(EHotelOrderState.OS_NEW);
        order.addOrderItem(orderItem);
        order.addInvoice(invoice);
        order.setDisplayType(EDisplayOrderType.DT_HOTEL);
        order.setCurrency(ProtoCurrencyUnit.RUB);
        em.persist(order);

        return orderItem;
    }

    private TravellineHotelItinerary testItinerary(Money price) {
        TravellineHotelItinerary itinerary = new TravellineHotelItinerary();
        itinerary.setFiscalPrice(price);
        itinerary.setOrderDetails(OrderDetails.builder()
                .checkoutDate(LocalDate.parse("2020-03-07"))
                .build());
        return itinerary;
    }

    @TestConfiguration
    public static class TestBeans {
        @Bean
        public SettableClock settableClock() {
            return new SettableClock();
        }

        @Bean
        public ClockService clockService() {
            return ClockService.create(settableClock());
        }

        @Bean
        public BillingTransactionYtTableClientProperties billingTransactionYtTableClientProperties() {
            // no need to fill these in as we use a mock YT Client
            return BillingTransactionYtTableClientProperties.builder()
                    .tablesDirectory("//test/dir")
                    .incomeTablesDirectory("//test/income/dir")
                    .transactionDuration(Duration.ofSeconds(1))
                    .batchSize(10)
                    .build();
        }
    }
}
