package ru.yandex.travel.orders.services.finances.billing;

import java.math.BigInteger;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.junit.After;
import org.junit.Before;
import org.junit.Ignore;
import org.junit.Test;

import ru.yandex.bolts.collection.Cf;
import ru.yandex.bolts.collection.Option;
import ru.yandex.inside.yt.kosher.CloseableYt;
import ru.yandex.inside.yt.kosher.common.GUID;
import ru.yandex.inside.yt.kosher.cypress.YPath;
import ru.yandex.inside.yt.kosher.impl.YtUtils;
import ru.yandex.inside.yt.kosher.ytree.YTreeStringNode;
import ru.yandex.travel.orders.entities.finances.BillingTransaction;
import ru.yandex.travel.orders.entities.finances.FinancialEventPaymentScheme;
import ru.yandex.travel.testing.local.LocalTestUtils;

import static java.util.stream.Collectors.toList;
import static ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentSystemType.YANDEX_MONEY;

@Ignore
@Slf4j
public class BillingTransactionYtTableClientLocalTest {
    private String login;
    private BillingTransactionYtTableClientProperties properties;
    private CloseableYt yt;

    @Before
    public void init() throws Exception {
        login = LocalTestUtils.getUserName();
        String ytToken = LocalTestUtils.readLocalToken(".yt/token");
        // WARN: be extra careful with prod table updates
        //String ytToken = LocalTestUtils.readYavSecret("ver-01d1z62f61168w4ysxqg7cnfn2", "value");
        properties = BillingTransactionYtTableClientProperties.builder()
                .cluster("hahn.yt.yandex.net")
                .token(ytToken)
                .tablesDirectory("//home/travel/LOGIN/billing_tx_test".replace("LOGIN", login))
                .transactionDuration(Duration.ofMinutes(5))
                .build();
        yt = (CloseableYt) YtUtils.http(properties.getCluster(), properties.getToken());
    }

    @After
    public void destroy() throws Exception {
        yt.close();
    }

    @Test
    public void copyTestData() {
        YPath srcPath = YPath.simple("//home/travel/LOGIN/dolphin_tx_data_sample_2019-10-08"
                .replace("LOGIN", login));
        BillingTransactionYtTableClient tableClient = new BillingTransactionYtTableClient(properties, yt);
        Map<Long, BillingTransaction> allTransactions = tableClient.readTransactions(null, srcPath);
        allTransactions.remove(107L);
        allTransactions.remove(108L);

        Instant newPayoutAt = Instant.now();
        List<BillingTransaction> newTransactions = new ArrayList<>();
        for (int i = 0; i < 3; i++) {
            for (BillingTransaction tx : allTransactions.values()) {
                tx = tx.toBuilder().build();
                tx.setPayoutAt(newPayoutAt);
                int shift = 110 + i * 10;
                tx.setYtId(tx.getYtId() + shift);
                if (tx.getOriginalTransaction() != null) {
                    tx.setOriginalTransaction(new BillingTransaction());
                    tx.getOriginalTransaction().setYtId(tx.getYtId() - 2);
                }
                tx.setPaymentSystemType(YANDEX_MONEY);
                tx.setTrustPaymentId(new BigInteger(tx.getTrustPaymentId(), 16)
                        .add(BigInteger.valueOf(shift)).toString(16));
                tx.setServiceOrderId(tx.getServiceOrderId() + "-" + shift);
                tx.setCreatedAt(tx.getCreatedAt().plus(Duration.ofSeconds(shift)));
                tx.setPayoutAt(tx.getPayoutAt().plus(Duration.ofSeconds(shift)));
                tx.setValue(tx.getValue().add(Money.of(shift, tx.getValue().getCurrency())));
                newTransactions.add(tx);
            }
        }
        LocalDate payoutDate = BillingHelper.toBillingDate(newPayoutAt).toLocalDate();
        tableClient.exportTransactions(payoutDate, newTransactions);
    }

    /**
     * <b>Migration guide:</b>
     * <ol>
     *     <li>Modify the migration code according to your needs</li>
     *     <li>Test your migrations on a single dev transactions table</li>
     *     <ul>
     *         <li>Make a copy of a single transactions table from the testing</li>
     *         <li>Dump its contents into a text file (e.g. Ctrl+A -> Ctrl+C in the YT interface)</li>
     *         <li>Configure the tablesDirectory param to point to your directory with the test table</li>
     *         <li>Run the migration code with dryRun=true</li>
     *         <li>Make sure only the expected transaction tables are listed with proper record counters</li>
     *         <li>Run the migration code with dryRun=false</li>
     *         <li>Compare the updated table contents to the saved copy and make sure there are only expected changes</li>
     *     </ul>
     *     <li>Migrate all tables from the testing</li>
     *     <ul>
     *         <li>Log in under the test robot, get its yt token, use it in the init method</li>
     *         <li>Change the tablesDirectory param to point to the 'transactions' directory from the testing</li>
     *         <li>Prepare a copy some testing table contents and use it validate the migrated version later</li>
     *         <li>Migrate the tables (a dry run first, the actual migration after it)</li>
     *     </ul>
     *     <li>Migrate all tables from the prod</li>
     *     <ul>
     *         <li>Make a backup copy of the whole 'transactions' directory by copying it in the sibling 'backups' directory</li>
     *         <li>Then repeat the same process as for the testing tables migration</li>
     *     </ul>
     * </ol>
     * <p/>
     * <b>Environments:</b>
     * <p><a href="https://yt.yandex-team.ru/hahn/navigation?path=//home/travel/testing/billing">testing</a> transactions</p>
     * <p><a href="https://yt.yandex-team.ru/hahn/navigation?path=//home/travel/prod/billing">prod</a> transactions & backups</p>
     * <p/>
     * <b>Robots and tokens:</b>
     * <p><a href="https://yav.yandex-team.ru/secret/sec-01d83kawsae4gwh115nyn1sr5x/explore/version/ver-01d83kawsfzxxr729chxy3b77n">robot-travel-testing</a></p>
     * <p><a href="https://yav.yandex-team.ru/secret/sec-01d8t7zednvzpayps7k0pf12vb/explore/version/ver-01d8t7zee5p0p0aedkp2mmfta1">robot-travel-prod</a></p>
     * <p>To access a robot's tokens log in under it via <a href="https://mail.yandex-team.ru">the mail</a></p>
     * <p>To get the YT access token open <a href="https://oauth.yt.yandex.net/new">the link</a></p>
     */
    @Test
    public void migrateOldTables() {
        properties = properties.toBuilder()
                .tablesDirectory("//home/travel/LOGIN/billing_tx_test".replace("LOGIN", login))
                //.tablesDirectory("//home/travel/testing/billing/transactions")
                .build();
        boolean dryRun = true;
        YPath transactionDir = YPath.simple(properties.getTablesDirectory());
        BillingTransactionYtTableClient tableClient = new BillingTransactionYtTableClient(properties, yt);

        // IMPORTANT: if you modify this code, make sure to run all nested yt-related operations on the same transaction
        tableClient.doInTx(txId -> {
            // a whole directory lock may be needed for more complex cases
            //yt.cypress().lock(txId, transactionDir, LockMode.EXCLUSIVE);
            for (YTreeStringNode node : yt.cypress().list(Optional.of(txId), false, transactionDir, Cf.set("type"))) {
                String type = node.getAttributeOrThrow("type").stringValue();
                if (!type.equals("table")) {
                    log.info("Skipping the object {}", node);
                    continue;
                }
                String tableName = node.getValue();
                YPath tablePath = transactionDir.child(tableName);
                log.info("Processing table {}: {}", node.getValue(), tablePath);
                boolean newColumnExists = tableClient.checkColumnExists(txId, tablePath, "service_id");
                if (newColumnExists) {
                    log.info("The column already exists, skipping this table");
                } else {
                    log.info("Old schema, re-creating the table with a new column: {}", tablePath);
                    LocalDate transactionsDate = LocalDate.parse(tablePath.name());
                    Collection<BillingTransaction> transactions =
                            tableClient.readTransactions(txId, tablePath, false).values();
                    for (BillingTransaction transaction : transactions) {
                        Preconditions.checkArgument(transaction.getServiceId() == null,
                                "The missing column value has to be null; %s", transaction);
                        transaction.setServiceId(FinancialEventPaymentScheme.HOTELS.getServiceId());
                    }
                    //noinspection ConstantConditions
                    if (!dryRun) {
                        log.info("Dropping the table with {} transactions", transactions.size());
                        yt.cypress().remove(txId, false, tablePath);
                        if (!transactions.isEmpty()) {
                            log.info("Importing the transactions again");
                            tableClient.exportTransactions(txId, transactionsDate, transactions);
                        } else {
                            log.info("The table is empty, no need to re-create it");
                        }
                    } else {
                        log.info("Dry run, {} transactions to migrate", transactions.size());
                    }
                }
            }
        });
    }

    @Test
    public void transferBillingTransaction() {
        properties = properties.toBuilder()
                .tablesDirectory("//home/travel/LOGIN/billing_tx_test".replace("LOGIN", login))
                .build();
        boolean dryRun = true;
        YPath transactionDir = YPath.simple(properties.getTablesDirectory());
        List<Long> ytIdsToMigrate = List.of(0L, 0L);
        YPath fromTablePath = transactionDir.child("1021-01-01");
        YPath toTablePath = transactionDir.child("1021-01-02");

        BillingTransactionYtTableClient tableClient = new BillingTransactionYtTableClient(properties, yt);

        // IMPORTANT: if you modify this code, make sure to run all nested yt-related operations in the same transaction
        tableClient.doInTx(txId -> {
            Option<GUID> txIdOpt = Option.of(txId);
            log.info("Migrating tx {} from table {} to {}", ytIdsToMigrate, fromTablePath, toTablePath);

            //tableClient.ensureTransactionsTableExists(txId, toTablePath);
            Collection<BillingTransaction> fromTransactions =
                    tableClient.readTransactions(txId, fromTablePath).values();
            Collection<BillingTransaction> toTransactions =
                    tableClient.readTransactions(txId, toTablePath).values();
            log.info("Read transactions: {} - {}, {} - {}",
                    fromTablePath.name(), fromTransactions.size(), toTablePath.name(), toTransactions.size());

            List<BillingTransaction> transactionsToMigrate =
                    fromTransactions.stream().filter(t -> ytIdsToMigrate.contains(t.getYtId())).collect(toList());
            Preconditions.checkState(transactionsToMigrate.size() == ytIdsToMigrate.size(),
                    "Exactly %s transactions are expected but got %s", ytIdsToMigrate.size(), transactionsToMigrate);
            List<BillingTransaction> newFromTransactions =
                    fromTransactions.stream().filter(t -> !transactionsToMigrate.contains(t)).collect(toList());
            List<BillingTransaction> newToTransactions = new ArrayList<>(toTransactions);
            newToTransactions.addAll(transactionsToMigrate);

            transactionsToMigrate.forEach(t -> {
                t.setPayoutAt(Instant.now());
                t.setAccountingActAt(Instant.now());
            });

            //noinspection ConstantConditions
            if (!dryRun) {
                log.info("Dropping the tables: {}, {}", fromTablePath, toTablePath);
                yt.cypress().remove(txIdOpt.getOrNull(), false, fromTablePath);
                yt.cypress().remove(txIdOpt.getOrNull(), false, toTablePath);
                if (!newFromTransactions.isEmpty()) {
                    log.info("Re-Importing the transactions again: {} -> {}",
                            fromTablePath.name(), newFromTransactions.size());
                    tableClient.exportTransactions(txId, LocalDate.parse(fromTablePath.name()), newFromTransactions);
                }
                if (!newToTransactions.isEmpty()) {
                    log.info("Re-Importing the transactions again: {} -> {}",
                            toTablePath.name(), newToTransactions.size());
                    tableClient.exportTransactions(txId, LocalDate.parse(toTablePath.name()), newToTransactions);
                }
            } else {
                log.info("Dry run, {}+{} transactions to migrate",
                        newFromTransactions.size(), newToTransactions.size());
            }
        });
    }

    @Test
    public void createSymLink() {
        BillingTransactionYtTableClient tableClient = new BillingTransactionYtTableClient(properties, yt);

        // IMPORTANT: if you modify this code, make sure to run all nested yt-related operations on the same transaction
        tableClient.doInTx(txId -> {
            YPath actualDir = YPath.simple("//home/travel/LOGIN/billing_tx_test".replace("LOGIN", login));
            YPath symLink = YPath.simple("//home/travel/LOGIN/billing_tx_test_link".replace("LOGIN", login));
            yt.cypress().link(Optional.of(txId), false, actualDir, symLink);
        });
    }

    @SuppressWarnings("UnnecessaryLocalVariable")
    @Test
    public void invertDirectoryAndSymLink() {
        BillingTransactionYtTableClient tableClient = new BillingTransactionYtTableClient(properties, yt);

        // IMPORTANT: if you modify this code, make sure to run all nested yt-related operations on the same transaction
        tableClient.doInTx(txId -> {
            YPath actualDir = YPath.simple("//home/travel/LOGIN/billing_tx_test".replace("LOGIN", login));
            YPath symLink = YPath.simple("//home/travel/LOGIN/billing_tx_test_link".replace("LOGIN", login));
            YPath newActualDir = symLink;
            YPath newSymLink = actualDir;

            yt.cypress().remove(txId, false, symLink);
            yt.cypress().move(txId, false, actualDir, newActualDir, true, false, true);
            yt.cypress().link(Optional.of(txId), false, newActualDir, newSymLink);
        });
    }
}
