package ru.yandex.travel.orders.services.mock;

import java.time.Duration;
import java.util.Collections;

import com.google.common.base.Strings;
import junit.framework.TestCase;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;

import ru.yandex.travel.orders.commons.proto.EAviaTokenizationOutcome;
import ru.yandex.travel.orders.commons.proto.EPaymentOutcome;
import ru.yandex.travel.orders.commons.proto.TAviaPaymentTestContext;
import ru.yandex.travel.orders.commons.proto.TPaymentTestContext;
import ru.yandex.travel.orders.entities.FiscalItemType;
import ru.yandex.travel.orders.integration.IntegrationUtils;
import ru.yandex.travel.orders.services.payments.PaymentProfile;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.payments.TrustUserInfo;
import ru.yandex.travel.orders.services.payments.model.PaymentStatusEnum;
import ru.yandex.travel.orders.services.payments.model.TrustCreateBasketRequest;

import static org.assertj.core.api.AssertionsForClassTypes.assertThat;

@SuppressWarnings("ConstantConditions")
@RunWith(SpringRunner.class)
@SpringBootTest(
        webEnvironment = SpringBootTest.WebEnvironment.NONE,
        properties = {
                "single-node.auto-start=true",
                "trust-db-mock.user-action-task-processor.schedule-rate=100ms",
                "trust-db-mock.user-action-task-processor.initial-start-delay=100ms",
//
//                user-action-task-processor:
//        name: MockTrustUserActionTaskProcessor
//        initial-start-delay: 1s
//        pool-size: 2
//        schedule-rate: 1s
        }
)
@ActiveProfiles("test")
public class MockDBTrustClientTest extends TestCase {
    @Autowired
    private TrustClientProvider trustClientProvider;

    private final TrustUserInfo trustUserInfo = null;

    @Test
    public void testNullPaymentContextAviaFlow() {
        TrustClient trustClient = getMockTrustClient();
        TrustCreateBasketRequest createBasketRequest = createTruestBasketRequest();
        createBasketRequest.setProductId(FiscalItemType.FLIGHT_AEROFLOT.getTrustId());

        var createBasketResponse = trustClient.createBasket(createBasketRequest, trustUserInfo, null);
        trustClient.startPayment(createBasketResponse.getPurchaseToken(), trustUserInfo);

        waitForBasketStatus(createBasketResponse.getPurchaseToken(), PaymentStatusEnum.CLEARED);

        var basketStatus = trustClient.getBasketStatus(createBasketResponse.getPurchaseToken(), trustUserInfo);
        assertThat(PaymentStatusEnum.CLEARED).isEqualTo(basketStatus.getPaymentStatus());
        assertThat("").isEqualTo(Strings.nullToEmpty(basketStatus.getPaymentRespCode()));
        assertThat("").isEqualTo(Strings.nullToEmpty(basketStatus.getPaymentRespDesc()));
    }

    @Test
    public void testNullPaymentContextFlow() {
        TrustClient trustClient = getMockTrustClient();
        TrustCreateBasketRequest createBasketRequest = createTruestBasketRequest();

        var createBasketResponse = trustClient.createBasket(createBasketRequest, trustUserInfo, null);
        trustClient.startPayment(createBasketResponse.getPurchaseToken(), trustUserInfo);

        waitForBasketStatus(createBasketResponse.getPurchaseToken(), PaymentStatusEnum.AUTHORIZED);

        var basketStatus = trustClient.getBasketStatus(createBasketResponse.getPurchaseToken(), trustUserInfo);

        assertThat(PaymentStatusEnum.AUTHORIZED).isEqualTo(basketStatus.getPaymentStatus());
        assertThat("").isEqualTo(Strings.nullToEmpty(basketStatus.getPaymentRespCode()));
        assertThat("").isEqualTo(Strings.nullToEmpty(basketStatus.getPaymentRespDesc()));
    }


    @Test
    public void testAviaPaymentContextFlowSuccess() {
        TrustClient trustClient = getMockTrustClient();
        TAviaPaymentTestContext aviaTestContext;
        var createBasketRequest = createTruestBasketRequest();
        createBasketRequest.setProductId(FiscalItemType.FLIGHT_AEROFLOT.getTrustId());

        aviaTestContext = TAviaPaymentTestContext.newBuilder()
                .setTokenizationOutcome(EAviaTokenizationOutcome.TO_SUCCESS)
                .build();

        var createBasketResponse = trustClient.createBasket(createBasketRequest, trustUserInfo, aviaTestContext);
        trustClient.startPayment(createBasketResponse.getPurchaseToken(), trustUserInfo);

        waitForBasketStatus(createBasketResponse.getPurchaseToken(), PaymentStatusEnum.CLEARED);

        var basketStatus = trustClient.getBasketStatus(createBasketResponse.getPurchaseToken(), trustUserInfo);

        assertThat(PaymentStatusEnum.CLEARED).isEqualTo(basketStatus.getPaymentStatus());
        assertThat("").isEqualTo(Strings.nullToEmpty(basketStatus.getPaymentRespCode()));
        assertThat("").isEqualTo(Strings.nullToEmpty(basketStatus.getPaymentRespDesc()));
    }

    @Test
    public void testAviaPaymentContextFlowFailure() {
        TrustClient trustClient = getMockTrustClient();
        TAviaPaymentTestContext aviaTestContext;
        var createBasketRequest = createTruestBasketRequest();
        createBasketRequest.setProductId(FiscalItemType.FLIGHT_AEROFLOT.getTrustId());

        aviaTestContext = TAviaPaymentTestContext.newBuilder()
                .setTokenizationOutcome(EAviaTokenizationOutcome.TO_FAILURE)
                .build();

        var createBasketResponse = trustClient.createBasket(createBasketRequest, trustUserInfo, aviaTestContext);
        trustClient.startPayment(createBasketResponse.getPurchaseToken(), trustUserInfo);

        waitForBasketStatus(createBasketResponse.getPurchaseToken(), PaymentStatusEnum.NOT_AUTHORIZED);

        var basketStatus = trustClient.getBasketStatus(createBasketResponse.getPurchaseToken(), trustUserInfo);

        assertThat(PaymentStatusEnum.NOT_AUTHORIZED).isEqualTo(basketStatus.getPaymentStatus());
        assertThat("USER_CANCELLED").isEqualTo(Strings.nullToEmpty(basketStatus.getPaymentRespCode()));
        assertThat(Strings.nullToEmpty(basketStatus.getPaymentRespDesc())).isNotEmpty();
    }

    @Test
    public void testPaymentContextFlowAviaOrderSuccess() {
        TrustClient trustClient = getMockTrustClient();
        TPaymentTestContext testContext;
        var createBasketRequest = createTruestBasketRequest();
        createBasketRequest.setProductId(FiscalItemType.FLIGHT_AEROFLOT.getTrustId());

        testContext = TPaymentTestContext.newBuilder()
                .setPaymentOutcome(EPaymentOutcome.PO_SUCCESS)
                .build();

        var createBasketResponse = trustClient.createBasket(createBasketRequest, trustUserInfo, testContext);
        trustClient.startPayment(createBasketResponse.getPurchaseToken(), trustUserInfo);
        waitForBasketStatus(createBasketResponse.getPurchaseToken(), PaymentStatusEnum.CLEARED);

        var basketStatus = trustClient.getBasketStatus(createBasketResponse.getPurchaseToken(), trustUserInfo);

        assertThat(PaymentStatusEnum.CLEARED).isEqualTo(basketStatus.getPaymentStatus());
        assertThat(Strings.nullToEmpty(basketStatus.getPaymentRespCode())).isEmpty();
        assertThat(Strings.nullToEmpty(basketStatus.getPaymentRespDesc())).isEmpty();
    }

    @Test
    public void testPaymentContextFlowAviaOrderFailure() {
        TrustClient trustClient = getMockTrustClient();
        TPaymentTestContext testContext;
        var createBasketRequest = createTruestBasketRequest();
        createBasketRequest.setProductId(FiscalItemType.FLIGHT_AEROFLOT.getTrustId());

        testContext = TPaymentTestContext.newBuilder()
                .setPaymentOutcome(EPaymentOutcome.PO_FAILURE)
                .build();

        var createBasketResponse = trustClient.createBasket(createBasketRequest, trustUserInfo, testContext);
        trustClient.startPayment(createBasketResponse.getPurchaseToken(), trustUserInfo);

        waitForBasketStatus(createBasketResponse.getPurchaseToken(), PaymentStatusEnum.NOT_AUTHORIZED);

        var basketStatus = trustClient.getBasketStatus(createBasketResponse.getPurchaseToken(), trustUserInfo);
        assertThat(PaymentStatusEnum.NOT_AUTHORIZED).isEqualTo(basketStatus.getPaymentStatus());
        assertThat("USER_CANCELLED").isEqualTo(Strings.nullToEmpty(basketStatus.getPaymentRespCode()));
        assertThat(Strings.nullToEmpty(basketStatus.getPaymentRespDesc())).isNotEmpty();
    }

    @Test
    public void testPaymentContextFlowSuccess() {
        TrustClient trustClient = getMockTrustClient();
        TPaymentTestContext testContext;
        var createBasketRequest = createTruestBasketRequest();

        testContext = TPaymentTestContext.newBuilder()
                .setPaymentOutcome(EPaymentOutcome.PO_SUCCESS)
                .setPaymentFailureResponseCode("failure_code")
                .setPaymentFailureResponseDescription("failure_description")
                .setPaymentUrl("some://payment.url")
                .build();

        var createBasketResponse = trustClient.createBasket(createBasketRequest, trustUserInfo, testContext);
        var startResponse = trustClient.startPayment(createBasketResponse.getPurchaseToken(), trustUserInfo);
        assertThat(startResponse.getPaymentUrl()).isEqualTo("some://payment.url");
        waitForBasketStatus(createBasketResponse.getPurchaseToken(), PaymentStatusEnum.AUTHORIZED);

        var basketStatus = trustClient.getBasketStatus(createBasketResponse.getPurchaseToken(), trustUserInfo);
        assertThat(PaymentStatusEnum.AUTHORIZED).isEqualTo(basketStatus.getPaymentStatus());
        assertThat(Strings.nullToEmpty(basketStatus.getPaymentRespCode())).isEmpty();
        assertThat(Strings.nullToEmpty(basketStatus.getPaymentRespDesc())).isEmpty();
    }

    @Test
    public void testPaymentContextFlowFailure() {
        TrustClient trustClient = getMockTrustClient();
        TPaymentTestContext testContext;
        var createBasketRequest = createTruestBasketRequest();

        testContext = TPaymentTestContext.newBuilder()
                .setPaymentOutcome(EPaymentOutcome.PO_FAILURE)
                .setPaymentFailureResponseCode("failure_code")
                .setPaymentFailureResponseDescription("failure_description")
                .build();

        var createBasketResponse = trustClient.createBasket(createBasketRequest, trustUserInfo, testContext);
        trustClient.startPayment(createBasketResponse.getPurchaseToken(), trustUserInfo);
        waitForBasketStatus(createBasketResponse.getPurchaseToken(), PaymentStatusEnum.NOT_AUTHORIZED);

        var basketStatus = trustClient.getBasketStatus(createBasketResponse.getPurchaseToken(), trustUserInfo);

        assertThat(PaymentStatusEnum.NOT_AUTHORIZED).isEqualTo(basketStatus.getPaymentStatus());
        assertThat("failure_code").isEqualTo(Strings.nullToEmpty(basketStatus.getPaymentRespCode()));
        assertThat("failure_description").isEqualTo(Strings.nullToEmpty(basketStatus.getPaymentRespDesc()));
    }

    private TrustClient getMockTrustClient() {
        var client = trustClientProvider.getTrustClientForPaymentProfile(PaymentProfile.MOCK_PAYMENT);
        assertThat(client).isInstanceOf(MockDBTrustClient.class);
        return client;
    }

    private TrustCreateBasketRequest createTruestBasketRequest() {
        var createBasketRequest = new TrustCreateBasketRequest();
        createBasketRequest.setOrders(Collections.emptyList());
        createBasketRequest.setAmount("1999");
        return createBasketRequest;
    }

    private void waitForBasketStatus(String token, PaymentStatusEnum status) {
        IntegrationUtils.waitForPredicateOrTimeout(
                () -> {
                    var basket = getMockTrustClient().getBasketStatus(token, trustUserInfo);
                    return basket.getPaymentStatus() == status;
                },
                Duration.ofSeconds(2),
                String.format("Basket must change to %s", status)
        );
    }
}
