package ru.yandex.travel.orders.services.promo.taxi2020;

import java.time.Clock;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDate;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.Callable;

import javax.persistence.EntityManager;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.jdbc.AutoConfigureTestDatabase;
import org.springframework.boot.test.context.SpringBootTest;
import org.springframework.boot.test.context.TestConfiguration;
import org.springframework.boot.test.mock.mockito.MockBean;
import org.springframework.boot.test.mock.mockito.SpyBean;
import org.springframework.context.annotation.Bean;
import org.springframework.test.annotation.DirtiesContext;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;
import org.springframework.transaction.support.TransactionTemplate;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.hotels.common.orders.OrderDetails;
import ru.yandex.travel.hotels.common.orders.TravellineHotelItinerary;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.TravellineOrderItem;
import ru.yandex.travel.orders.entities.promo.taxi2020.Taxi2020PromoCode;
import ru.yandex.travel.orders.entities.promo.taxi2020.Taxi2020PromoOrder;
import ru.yandex.travel.orders.repository.promo.taxi2020.Taxi2020PromoOrderRepository;
import ru.yandex.travel.orders.services.MailSenderService;
import ru.yandex.travel.orders.workflow.hotels.proto.EHotelOrderState;
import ru.yandex.travel.testing.time.SettableClock;
import ru.yandex.travel.utils.ClockService;
import ru.yandex.travel.workflow.entities.WorkflowEntity;

import static org.assertj.core.api.Assertions.assertThat;
import static org.mockito.ArgumentMatchers.any;
import static ru.yandex.travel.orders.entities.promo.taxi2020.Taxi2020PromoOrderStatus.ELIGIBLE;
import static ru.yandex.travel.orders.entities.promo.taxi2020.Taxi2020PromoOrderStatus.EMAIL_SCHEDULED;
import static ru.yandex.travel.orders.entities.promo.taxi2020.Taxi2020PromoOrderStatus.EMAIL_SENT;
import static ru.yandex.travel.orders.entities.promo.taxi2020.Taxi2020PromoOrderStatus.NOT_ELIGIBLE;
import static ru.yandex.travel.orders.entities.promo.taxi2020.Taxi2020PromoOrderStatus.PROMO_CODE_CAN_BE_ASSIGNED;
import static ru.yandex.travel.orders.integration.IntegrationUtils.waitForPredicateOrTimeout;
import static ru.yandex.travel.testing.misc.MockitoUtils.waitForMockCalls;
import static ru.yandex.travel.testing.spring.SpringUtils.unwrapAopProxy;

@RunWith(SpringRunner.class)
@SpringBootTest(
        webEnvironment = SpringBootTest.WebEnvironment.NONE,
        properties = {
                "promo.taxi2020-scheduler-task.enabled=true",
                "promo.taxi2020-scheduler-task.schedule-rate=100ms",
                // enabling WorkflowProcessService
                "quartz.enabled=true",
                "workflow-processing.pending-workflow-polling-interval=100ms",
                "single-node.auto-start=true",
        }
)
@ActiveProfiles("test")
@DirtiesContext
@AutoConfigureTestDatabase(replace = AutoConfigureTestDatabase.Replace.ANY)
@MockBean(MailSenderService.class)
@Slf4j
public class Taxi2020PromoServiceIntegrationTest {
    @Autowired
    private TransactionTemplate transactionTemplate;
    @Autowired
    private EntityManager em;
    @Autowired
    private SettableClock settableClock;

    @Autowired
    private Taxi2020PromoOrderRepository promoOrderRepository;
    @Autowired
    @SpyBean
    private Taxi2020PromoService promoOrderServiceAopProxy;
    private Taxi2020PromoService promoOrderService;

    @Before
    public void init() {
        promoOrderService = unwrapAopProxy(promoOrderServiceAopProxy);
    }

    @Test
    public void processPendingOrder_successFuture() {
        settableClock.setCurrentTime(Instant.parse("2020-10-11T16:43:07Z"));
        Duration timeout = Duration.ofSeconds(10);
        Duration retryDelay = Duration.ofMillis(50);

        // a new order gets registered for the promo
        runInTx(() -> createTestPromoCode("testCode1"));
        UUID orderId = callInTx(() -> createTestOrder(LocalDate.parse("2021-02-15"), "some@mail.com", 10000)).getId();
        log.info("A test order has been created: id {}", orderId);
        Taxi2020PromoOrder promoOrder = promoOrderRepository.findById(orderId).orElse(null);
        assertThat(promoOrder).isNotNull();
        assertThat(promoOrder.getStatus()).isEqualTo(ELIGIBLE);
        assertThat(promoOrder.getEmail()).isEqualTo("some@mail.com");
        assertThat(promoOrder.getEmailScheduledAt()).isEqualTo(Instant.parse("2021-02-14T21:00:00Z"));
        assertThat(promoOrder.getPromoCode()).isNull();
        assertThat(promoOrder.getSendEmailOperationId()).isNull();

        waitForMockCalls(promoOrderService, s -> s.findOrderIdsWaitingForProcessing(any()), 2,
                timeout, retryDelay, "The processor keeps working but nothing changes at this point");
        assertThat(promoOrderRepository.findById(orderId).orElseThrow().getStatus()).isEqualTo(ELIGIBLE);

        log.info("Waiting till the check-in date comes and the email is sent");
        settableClock.setCurrentTime(Instant.parse("2021-02-15T00:00:00Z"));
        waitForPredicateOrTimeout(() -> callInTx(
                () -> promoOrderRepository.getOne(orderId).getStatus() == EMAIL_SENT),
                timeout, retryDelay, "Promo email has been sent");
        promoOrder = promoOrderRepository.findAllById(List.of(orderId)).get(0);
        assertThat(promoOrder.getStatus()).isEqualTo(EMAIL_SENT);
        assertThat(promoOrder.getPromoCode()).isEqualTo("testCode1");
        assertThat(promoOrder.getSendEmailOperationId()).isNotNull();
        assertThat(promoOrder.getSendSmsOperationId()).isNotNull();
        assertThat(promoOrder.getSentAt()).isEqualTo("2021-02-15T00:00:00Z");
    }

    @Test
    public void processPendingOrder_successSameDay() {
        settableClock.setCurrentTime(Instant.parse("2020-10-11T16:43:07Z"));
        Duration timeout = Duration.ofSeconds(10);
        Duration retryDelay = Duration.ofMillis(50);

        log.info("Creating and registering the same check-in day order");
        runInTx(() -> createTestPromoCode("testCode2"));
        UUID orderId = callInTx(() -> createTestOrder(LocalDate.parse("2020-10-11"), "some2@mail.com", 9000)).getId();
        log.info("A test order has been created: id {}", orderId);
        Taxi2020PromoOrder promoOrder = promoOrderRepository.findById(orderId).orElse(null);
        assertThat(promoOrder).isNotNull();
        assertThat(promoOrder.getStatus()).isIn(ELIGIBLE, PROMO_CODE_CAN_BE_ASSIGNED, EMAIL_SCHEDULED);
        assertThat(promoOrder.getEmail()).isEqualTo("some2@mail.com");
        assertThat(promoOrder.getEmailScheduledAt()).isEqualTo(Instant.parse("2020-10-10T21:00:00Z"));

        log.info("Waiting for the next run of the Taxi 2020 promo processor to finish the ready promo order");
        waitForPredicateOrTimeout(() -> callInTx(
                () -> promoOrderRepository.getOne(orderId).getStatus() == EMAIL_SENT),
                timeout, retryDelay, "Promo email is ready to be sent");
        promoOrder = promoOrderRepository.findAllById(List.of(orderId)).get(0);
        assertThat(promoOrder.getStatus()).isEqualTo(EMAIL_SENT);
        assertThat(promoOrder.getPromoCode()).isEqualTo("testCode2");
        assertThat(promoOrder.getSendEmailOperationId()).isNotNull();
        assertThat(promoOrder.getSendSmsOperationId()).isNotNull();
        assertThat(promoOrder.getSentAt()).isEqualTo("2020-10-11T16:43:07Z");
    }

    @Test
    public void processPendingOrder_refunded() {
        settableClock.setCurrentTime(Instant.parse("2020-10-11T16:43:07Z"));
        Duration timeout = Duration.ofSeconds(10);
        Duration retryDelay = Duration.ofMillis(50);

        log.info("Creating and registering the new future check-in day order");
        UUID orderId = callInTx(() -> createTestOrder(LocalDate.parse("2021-02-15"), "some3@mail.com", 10000)).getId();
        log.info("A test order has been created: id {}", orderId);
        Taxi2020PromoOrder promoOrder = promoOrderRepository.findById(orderId).orElse(null);
        assertThat(promoOrder).isNotNull();
        assertThat(promoOrder.getStatus()).isEqualTo(ELIGIBLE);
        assertThat(promoOrder.getEmail()).isEqualTo("some3@mail.com");
        assertThat(promoOrder.getEmailScheduledAt()).isEqualTo(Instant.parse("2021-02-14T21:00:00Z"));
        assertThat(promoOrder.getPromoCode()).isNull();
        assertThat(promoOrder.getSendEmailOperationId()).isNull();
        assertThat(promoOrder.getSendSmsOperationId()).isNull();

        runInTx(() -> {
            var order = em.find(HotelOrder.class, orderId);
            order.setState(EHotelOrderState.OS_REFUNDED);
            forcefullySetUpdatedAt(order, Instant.parse("2020-10-11T16:46:07Z"));
        });

        log.info("Waiting for the next run of the Taxi 2020 promo processor to detect changes and cancel the order");
        waitForPredicateOrTimeout(() -> callInTx(
                () -> promoOrderRepository.getOne(orderId).getStatus() == NOT_ELIGIBLE),
                timeout, retryDelay, "Waiting until order promo participation is cancelled");
        promoOrder = promoOrderRepository.findAllById(List.of(orderId)).get(0);
        assertThat(promoOrder.getStatus()).isEqualTo(NOT_ELIGIBLE);
        assertThat(promoOrder.getEmail()).isNull();
        assertThat(promoOrder.getEmailScheduledAt()).isNull();
        assertThat(promoOrder.getPromoCode()).isNull();
        assertThat(promoOrder.getSendEmailOperationId()).isNull();
        assertThat(promoOrder.getSendSmsOperationId()).isNull();
    }

    private void runInTx(Runnable r) {
        callInTx(() -> {
            r.run();
            return null;
        });
    }

    private <T> T callInTx(Callable<T> r) {
        return transactionTemplate.execute(tx -> {
            try {
                return r.call();
            } catch (Exception e) {
                throw new RuntimeException(e);
            }
        });
    }

    private void forcefullySetCreatedAt(WorkflowEntity<?> entity, Instant now) {
        String query = String.format("update %s set createdAt = :ts where id = :id",
                entity.getClass().getSimpleName());
        int updated = em.createQuery(query)
                .setParameter("ts", now)
                .setParameter("id", entity.getId())
                .executeUpdate();
        Preconditions.checkArgument(updated == 1, "Expected exactly 1 updated entity but got %s", updated);
    }

    private void forcefullySetUpdatedAt(WorkflowEntity<?> entity, Instant now) {
        String query = String.format("update %s set updatedAt = :ts where id = :id",
                entity.getClass().getSimpleName());
        int updated = em.createQuery(query)
                .setParameter("ts", now)
                .setParameter("id", entity.getId())
                .executeUpdate();
        Preconditions.checkArgument(updated == 1, "Expected exactly 1 updated entity but got %s", updated);
    }

    private void createTestPromoCode(String code) {
        em.persist(Taxi2020PromoCode.builder()
                .code(code)
                .addedAt(Instant.now(settableClock))
                .expiresAt(Instant.parse("4000-08-12T00:00:00Z"))
                .build());
    }

    private HotelOrder createTestOrder(LocalDate checkIn, String email, int price) {
        TravellineOrderItem orderItem = new TravellineOrderItem();
        orderItem.setId(UUID.randomUUID());
        TravellineHotelItinerary itinerary = new TravellineHotelItinerary();
        itinerary.setFiscalPrice(Money.of(price, "RUB"));
        itinerary.setOrderDetails(OrderDetails.builder()
                .checkinDate(checkIn)
                .checkoutDate(checkIn.plusDays(14))
                .build());
        orderItem.setItinerary(itinerary);
        orderItem.setUpdatedAt(Instant.now(settableClock));

        HotelOrder order = new HotelOrder();
        order.setId(UUID.randomUUID());
        order.setCurrency(ProtoCurrencyUnit.RUB);
        order.setEmail(email);
        order.setState(EHotelOrderState.OS_CONFIRMED);
        order.addOrderItem(orderItem);
        em.persist(order);

        forcefullySetCreatedAt(order, Instant.now(settableClock));
        forcefullySetUpdatedAt(order, Instant.now(settableClock));
        forcefullySetUpdatedAt(orderItem, Instant.now(settableClock));
        // dropping the cached entities
        em.clear();

        order = em.find(HotelOrder.class, order.getId());
        promoOrderService.registerConfirmedOrder(order);

        return order;
    }

    @TestConfiguration
    public static class TestBeans {
        @Bean
        public Clock clock() {
            return new SettableClock();
        }

        @Bean
        public ClockService clockService(Clock clock) {
            return ClockService.create(clock);
        }
    }
}
