package ru.yandex.travel.orders.services.report;

import java.math.BigDecimal;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import javax.persistence.EntityManager;

import org.javamoney.moneta.Money;
import org.jetbrains.annotations.NotNull;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.test.autoconfigure.jdbc.AutoConfigureTestDatabase;
import org.springframework.boot.test.autoconfigure.orm.jpa.DataJpaTest;
import org.springframework.data.repository.CrudRepository;
import org.springframework.test.context.ActiveProfiles;
import org.springframework.test.context.junit4.SpringRunner;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.hotels.common.orders.Guest;
import ru.yandex.travel.hotels.common.orders.OrderDetails;
import ru.yandex.travel.hotels.common.orders.TravellineHotelItinerary;
import ru.yandex.travel.orders.entities.HotelOrder;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.TravellineOrderItem;
import ru.yandex.travel.orders.entities.WellKnownOrderItemDiscriminator;
import ru.yandex.travel.orders.entities.finances.BankOrder;
import ru.yandex.travel.orders.entities.finances.BankOrderDetail;
import ru.yandex.travel.orders.entities.finances.BankOrderPayment;
import ru.yandex.travel.orders.entities.finances.BillingTransaction;
import ru.yandex.travel.orders.entities.finances.BillingTransactionPaymentType;
import ru.yandex.travel.orders.entities.finances.BillingTransactionType;
import ru.yandex.travel.orders.entities.finances.FinancialEvent;
import ru.yandex.travel.orders.repository.BillingTransactionRepository;
import ru.yandex.travel.orders.repository.FinancialEventRepository;
import ru.yandex.travel.orders.repository.OrderItemRepository;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.repository.finances.BankOrderDetailRepository;
import ru.yandex.travel.orders.repository.finances.BankOrderPaymentRepository;
import ru.yandex.travel.orders.repository.finances.BankOrderRepository;
import ru.yandex.travel.orders.services.report.model.PartnerPaymentOrdersReport;

import static org.junit.Assert.assertEquals;

@RunWith(SpringRunner.class)
@DataJpaTest
@AutoConfigureTestDatabase(replace = AutoConfigureTestDatabase.Replace.ANY)
@ActiveProfiles("test")
public class PartnerTransactionsFetcherTest {

    @Autowired
    private BillingTransactionRepository billingTransactionRepository;

    @Autowired
    private BankOrderRepository bankOrderRepository;
    @Autowired
    private BankOrderDetailRepository bankOrderDetailsRepository;

    @Autowired
    private BankOrderPaymentRepository bankOrderPaymentRepository;

    @Autowired
    private FinancialEventRepository financialEventRepository;

    @Autowired
    private OrderItemRepository orderItemRepository;

    @Autowired
    private OrderRepository orderRepository;

    @Autowired
    private EntityManager entityManager;

    private PartnerTransactionsFetcher fetcher;

    BankOrderDetail costTransaction;
    BankOrderDetail rewardTransaction;
    BankOrderDetail yandexCostTransaction;
    BankOrderDetail yandexRewardTransaction;

    @Before
    public void prepareTransactions() {
        List.of(billingTransactionRepository,
                        bankOrderRepository,
                        bankOrderDetailsRepository,
                        bankOrderPaymentRepository,
                        financialEventRepository)
                .forEach(CrudRepository::deleteAll);

        fetcher = new PartnerTransactionsFetcher(billingTransactionRepository, bankOrderDetailsRepository,
                entityManager);

        costTransaction = new BankOrderDetail();
        costTransaction.setPaymentType(BillingTransactionPaymentType.COST);
        costTransaction.setTransactionType(BillingTransactionType.PAYMENT);
        costTransaction.setSum(BigDecimal.valueOf(1200.00));

        rewardTransaction = new BankOrderDetail();
        rewardTransaction.setPaymentType(BillingTransactionPaymentType.REWARD);
        rewardTransaction.setTransactionType(BillingTransactionType.PAYMENT);
        rewardTransaction.setSum(BigDecimal.valueOf(200.00));

        yandexCostTransaction = new BankOrderDetail();
        yandexCostTransaction.setPaymentType(BillingTransactionPaymentType.YANDEX_ACCOUNT_COST_WITHDRAW);
        yandexCostTransaction.setTransactionType(BillingTransactionType.PAYMENT);
        yandexCostTransaction.setSum(BigDecimal.valueOf(120.00));

        yandexRewardTransaction = new BankOrderDetail();
        yandexRewardTransaction.setPaymentType(BillingTransactionPaymentType.YANDEX_ACCOUNT_REWARD_WITHDRAW);
        yandexRewardTransaction.setTransactionType(BillingTransactionType.PAYMENT);
        yandexRewardTransaction.setSum(BigDecimal.valueOf(100.00));
    }

    @Test
    public void fetchPartnerPaymentTransactionsBasic() {
        var bankOrder = new BankOrder();
        create(bankOrder, List.of(costTransaction, rewardTransaction));

        List<PartnerPaymentOrdersReport.TransactionRow> transactionRows =
                fetcher.fetchPartnerPaymentOrderTransactions(bankOrder);
        assertEquals(transactionRows.toString(), 1, transactionRows.size());
        var transactionRow = transactionRows.get(0);
        assertEquals(costTransaction.getSum(), BigDecimal.valueOf(transactionRow.getPartnerAmount()));
    }

    @Test
    public void fetchPartnerPaymentTransactionsYandexWithdrawal() {
        var bankOrder = new BankOrder();
        create(bankOrder, List.of(costTransaction, rewardTransaction, yandexCostTransaction));

        List<PartnerPaymentOrdersReport.TransactionRow> transactionRows =
                fetcher.fetchPartnerPaymentOrderTransactions(bankOrder);
        assertEquals(transactionRows.toString(), 1, transactionRows.size());
        var transactionRow = transactionRows.get(0);
        assertEquals(costTransaction.getSum().add(yandexCostTransaction.getSum()),
                BigDecimal.valueOf(transactionRow.getPartnerAmount()));
    }

    private void create(BankOrder bankOrder, List<BankOrderDetail> details) {
        var bankOrderPayment = new BankOrderPayment();
        bankOrderPayment.setPaymentBatchId("12345");
        bankOrderPayment.setOrders(new ArrayList<>(List.of(bankOrder)));
        bankOrderPayment.setDetails(new ArrayList<>(details));

        bankOrder.setBankOrderPayment(bankOrderPayment);

        OrderItem orderItem = createOrderItem();

        for (int i = 0; i < details.size(); i++) {
            BankOrderDetail detail = details.get(i);
            detail.setBankOrderPayment(bankOrderPayment);
            detail.setYtId(1000L + i);

            BillingTransaction bt = new BillingTransaction();
            bt.setYtId(detail.getYtId());
            FinancialEvent financialEvent = new FinancialEvent();
            bt.setSourceFinancialEvent(financialEvent);
            bt.setTransactionType(detail.getTransactionType());
            bt.setPaymentType(detail.getPaymentType());
            financialEvent.setOrderItem(orderItem);
            billingTransactionRepository.save(bt);
            financialEventRepository.save(financialEvent);
        }

        bankOrderPaymentRepository.save(bankOrderPayment);
        bankOrderRepository.save(bankOrder);

    }

    @NotNull
    private TravellineOrderItem createOrderItem() {
        var orderItem = new TravellineOrderItem();
        orderItem.setType(WellKnownOrderItemDiscriminator.ORDER_ITEM_TRAVELLINE);
        var itinerary = new TravellineHotelItinerary();
        itinerary.setOrderDetails(OrderDetails.builder().build());
        itinerary.setGuests(List.of(new Guest()));
        itinerary.setFiscalPrice(Money.of(1400.00, ProtoCurrencyUnit.RUB));
        orderItem.setItinerary(itinerary);
        HotelOrder order = new HotelOrder();
        order.setId(UUID.randomUUID());
        orderItem.setOrder(orderRepository.save(order));
        orderItem.setConfirmedAt(Instant.EPOCH);
        return orderItemRepository.saveAndFlush(orderItem);
    }
}
