package ru.yandex.travel.orders.cache;

import java.time.Instant;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import org.springframework.boot.actuate.health.Health;
import org.springframework.boot.actuate.health.HealthIndicator;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Component;

import ru.yandex.travel.hotels.administrator.export.proto.ContractInfo;
import ru.yandex.travel.orders.configurations.BalanceContractDictionaryProperties;
import ru.yandex.travel.yt_lucene_index.NativeProtobufPlainYtLuceneIndex;
import ru.yandex.travel.yt_lucene_index.YtLuceneIndexParams;

@Component
@EnableConfigurationProperties(BalanceContractDictionaryProperties.class)
public class BalanceContractDictionary implements HealthIndicator {
    private static final String BALANCE_CONTRACTS = "balance_contracts";

    private final AtomicReference<Map<Long, ContractInfo>> balanceContractsRef =
            new AtomicReference<>(null);
    private final AtomicReference<Map<Long, ContractInfo>> contractMapRef = new AtomicReference<>(null);
    private final AtomicLong tableCacheUpdateTimestamp = new AtomicLong(0);

    private final NativeProtobufPlainYtLuceneIndex<ContractInfo> luceneIndex;

    public BalanceContractDictionary(BalanceContractDictionaryProperties properties) {
        YtLuceneIndexParams params = new YtLuceneIndexParams();
        params.setProxy(properties.getProxy());
        params.setToken(properties.getToken());
        params.setTablePath(properties.getBalanceContractsYtTable());
        params.setIndexPath(properties.getBaseLocalPath() + "/" + BALANCE_CONTRACTS + "-index");
        if (properties.getUpdateInterval() != null) {
            params.setUpdateInterval(properties.getUpdateInterval());
        }
        luceneIndex = new NativeProtobufPlainYtLuceneIndex<>(params, BALANCE_CONTRACTS, ContractInfo.class);
        luceneIndex.setIndexUpdateHandler(() -> {
            Map<Long, ContractInfo> balanceContracts = new HashMap<>();
            Map<Long, ContractInfo> contractMap = new HashMap<>();
            luceneIndex.forEachProtoRecord(ContractInfo.parser(), proto -> {
                balanceContracts.put(proto.getClientId(), proto);
                contractMap.put(proto.getContractId(), proto);
            });
            balanceContractsRef.set(balanceContracts);
            contractMapRef.set(contractMap);
            tableCacheUpdateTimestamp.set(Instant.now().toEpochMilli());
        });
    }

    @PostConstruct
    public void init() {
        luceneIndex.start();
    }

    @SuppressWarnings("UnstableApiUsage")
    @PreDestroy
    public void destroy() {
        luceneIndex.stop();
    }

    public ContractInfo findContractInfoByClientId(long clientId) {
        return balanceContractsRef.get().get(clientId);
    }

    public ContractInfo findContractInfoByContractId(long contractId) {
        return contractMapRef.get().get(contractId);
    }

    public boolean isReady() {
        return balanceContractsRef.get() != null;
    }

    @Override
    public Health health() {
        if (isReady()) {
            return Health.up().build();
        } else {
            return Health.down().build();
        }
    }
}
