package ru.yandex.travel.orders.cache;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;

import com.google.common.base.Preconditions;
import org.springframework.boot.actuate.health.Health;
import org.springframework.boot.actuate.health.HealthIndicator;

import ru.yandex.travel.hotels.administrator.export.proto.HotelAgreement;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.orders.services.finances.HotelAgreementServiceProperties;
import ru.yandex.travel.yt_lucene_index.NativeProtobufPlainYtLuceneIndex;
import ru.yandex.travel.yt_lucene_index.YtLuceneIndexParams;

public class HotelAgreementDictionary implements HealthIndicator {
    private static final String HOTEL_AGREEMENTS = "hotel_agreements";

    private final AtomicReference<Map<EPartnerId, Map<String, List<HotelAgreement>>>> hotelAgreementsRef =
            new AtomicReference<>(null);
    private final AtomicLong tableCacheUpdateTimestamp = new AtomicLong(0);

    private final NativeProtobufPlainYtLuceneIndex<HotelAgreement> luceneIndex;

    public HotelAgreementDictionary(YtLuceneIndexParams params, String name) {
        luceneIndex = new NativeProtobufPlainYtLuceneIndex<>(params, name, HotelAgreement.class);
        luceneIndex.setIndexUpdateHandler(() -> {
            Map<EPartnerId, Map<String, List<HotelAgreement>>> hotelAgreements = new HashMap<>();
            luceneIndex.forEachProtoRecord(HotelAgreement.parser(), proto -> {
                hotelAgreements
                        .computeIfAbsent(proto.getPartnerId(), partner -> new HashMap<>())
                        .computeIfAbsent(proto.getHotelId(), hotelId -> new ArrayList<>())
                        .add(proto);
            });
            hotelAgreementsRef.set(hotelAgreements);
            tableCacheUpdateTimestamp.set(Instant.now().toEpochMilli());
        });
    }

    public static HotelAgreementDictionary fromBaseProperties(HotelAgreementServiceProperties.YtCacheProperties properties) {
        YtLuceneIndexParams params = new YtLuceneIndexParams();
        params.setProxy(properties.getProxy());
        params.setToken(properties.getToken());
        params.setTablePath(properties.getHotelAgreementsYtTable());
        params.setIndexPath(properties.getBaseLocalPath() + "/" + HOTEL_AGREEMENTS + "-index");
        if (properties.getUpdateInterval() != null) {
            params.setUpdateInterval(properties.getUpdateInterval());
        }
        return new HotelAgreementDictionary(params, HOTEL_AGREEMENTS);
    }

    @PostConstruct
    public void init() {
        luceneIndex.start();
    }

    @SuppressWarnings("UnstableApiUsage")
    @PreDestroy
    public void destroy() {
        luceneIndex.stop();
    }

    public HotelAgreement findAgreementByHotelIdAndTimestamp(EPartnerId partnerId, String hotelId, Instant instant) {
        final long timestamp = instant.toEpochMilli();
        List<HotelAgreement> matchedAgreements = filterEnabledAtGivenTimestampWithMaxPriority(
                hotelAgreementsRef.get()
                        .getOrDefault(partnerId, new HashMap<>())
                        .getOrDefault(hotelId, new ArrayList<>())
                        .stream(),
                timestamp
        );


        Preconditions.checkState(
                matchedAgreements.size() == 1,
                String.format(
                        "%d agreements with maximum priority found for hotel with code %s, partner %s, timestamp = %d",
                        matchedAgreements.size(),
                        hotelId,
                        partnerId.toString(),
                        timestamp));
        return matchedAgreements.get(0);
    }

    public long getTableCacheUpdateTimestamp() {
        return tableCacheUpdateTimestamp.get();
    }

    public Set<Long> getAllClientIds() {
        if (hotelAgreementsRef.get() == null) {
            return new HashSet<>();
        }
        return getRawStreamOfAgreements()
                .map(HotelAgreement::getFinancialClientId)
                .collect(Collectors.toSet());
    }

    public boolean hasClientId(long financialClientId) {
        if (hotelAgreementsRef.get() == null) {
            return false;
        }
        return getRawStreamOfAgreements()
                .anyMatch(agreement -> agreement.getFinancialClientId() == financialClientId);
    }

    public boolean isReady() {
        return hotelAgreementsRef.get() != null;
    }

    @Override
    public Health health() {
        if (isReady()) {
            return Health.up().build();
        } else {
            return Health.down().build();
        }
    }

    public Set<Long> getAllFinancialClientsForPartnersInGivenPeriod(Set<EPartnerId> partners, Instant periodStart, Instant periodEnd) {
        return getRawStreamOfAgreements()
                .filter(filterEnabledInGivenPeriodByPartnerPredicate(partners, periodStart.toEpochMilli(), periodEnd.toEpochMilli()))
                .map(HotelAgreement::getFinancialClientId)
                .collect(Collectors.toSet());
    }

    private Stream<HotelAgreement> getRawStreamOfAgreements() {
        return hotelAgreementsRef.get()
                .values().stream()
                .flatMap(map -> map.values().stream())
                .flatMap(Collection::stream);
    }

    private static List<HotelAgreement> filterEnabledAtGivenTimestampWithMaxPriority(final Stream<HotelAgreement> agreementStream, final long timestamp) {
        return agreementStream
                .filter(filterEnabledInGivenPeriodByPartnerPredicate(null, timestamp, timestamp))
                .collect(Collectors.groupingBy(HotelAgreement::getPriority, Collectors.toList()))
                .entrySet().stream()
                .max(Map.Entry.comparingByKey())
                .map(Map.Entry::getValue)
                .orElse(Collections.emptyList());
    }

    private static Predicate<HotelAgreement> filterEnabledInGivenPeriodByPartnerPredicate(final Set<EPartnerId> partners,
                                                                                          final long periodStartTimestamp,
                                                                                          final long periodEndTimestamp) {
        return hotelAgreement -> {
            boolean isOneOfSelectedPartners = true;
            if (partners != null) {
                isOneOfSelectedPartners = partners.contains(hotelAgreement.getPartnerId());
            }
            boolean fitsInPeriod = hotelAgreement.getAgreementStartDate() <= periodEndTimestamp &&
                    (hotelAgreement.getAgreementEndDate() == 0 || periodStartTimestamp < hotelAgreement.getAgreementEndDate());
            boolean isEnabled = hotelAgreement.getEnabled();
            return isOneOfSelectedPartners && isEnabled && fitsInPeriod;
        };
    }
}
