package ru.yandex.travel.orders.configurations;

import java.io.ByteArrayInputStream;
import java.io.FileInputStream;
import java.io.InputStream;
import java.util.Arrays;
import java.util.stream.Collectors;

import javax.net.ssl.SSLException;

import io.micrometer.core.instrument.Gauge;
import io.micrometer.core.instrument.Metrics;
import io.netty.handler.ssl.SslContext;
import io.netty.handler.ssl.SslContextBuilder;
import io.netty.handler.ssl.SslProvider;
import io.opentracing.Tracer;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.asynchttpclient.AsyncHttpClient;
import org.asynchttpclient.DefaultAsyncHttpClientConfig;
import org.asynchttpclient.Dsl;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.hotels.common.partners.bnovo.DefaultBNovoClient;
import ru.yandex.travel.hotels.common.partners.dolphin.DefaultDolphinClient;
import ru.yandex.travel.hotels.common.partners.expedia.DefaultExpediaClient;
import ru.yandex.travel.hotels.common.partners.travelline.DefaultTravellineClient;
import ru.yandex.travel.orders.services.payments.DefaultTrustClient;
import ru.yandex.travel.orders.services.pdfgenerator.PdfGeneratorServiceImpl;
import ru.yandex.travel.orders.workflows.orderitem.suburban.SuburbanProperties;
import ru.yandex.travel.suburban.partners.aeroexpress.AeroexpressClient;
import ru.yandex.travel.suburban.partners.aeroexpress.DefaultAeroexpressClient;
import ru.yandex.travel.suburban.partners.movista.MovistaClient;
import ru.yandex.travel.train.partners.im.DefaultImClient;

@Slf4j
@Configuration
@EnableConfigurationProperties(AhcCommonProperties.class)
@RequiredArgsConstructor
public class AhcAutoConfiguration {

    private final AhcCommonProperties ahcCommonProperties;

    public AsyncHttpClient withMetrics(AsyncHttpClient meteredClient) {
        var poolName = meteredClient.getConfig().getThreadPoolName();
        var stats = meteredClient.getClientStats();
        Gauge.builder("http.client.stats.totalConnectionCount", stats::getTotalConnectionCount)
                .tag("poolName", poolName)
                .register(Metrics.globalRegistry);
        Gauge.builder("http.client.stats.totalActiveConnectionCount", stats::getTotalActiveConnectionCount)
                .tag("poolName", poolName)
                .register(Metrics.globalRegistry);
        Gauge.builder("http.client.stats.getTotalIdleConnectionCount", stats::getTotalIdleConnectionCount)
                .tag("poolName", poolName)
                .register(Metrics.globalRegistry);
        return meteredClient;
    }

    @Bean
    public AsyncHttpClient hotelAhcClient() {
        return withMetrics(Dsl.asyncHttpClient(Dsl.config()
                .setThreadPoolName("hotelAhcClient")
                .setIoThreadsCount(ahcCommonProperties.getIoThreads())
                .build()));
    }

    @Bean
    public AsyncHttpClient busAhcClient() {
        return withMetrics(Dsl.asyncHttpClient(Dsl.config()
                .setThreadPoolName("busAhcPool")
                .setIoThreadsCount(ahcCommonProperties.getIoThreads())
                .build()));
    }

    @Bean
    public AsyncHttpClient billingApiAhcClient() {
        return withMetrics(Dsl.asyncHttpClient(Dsl.config()
                .setThreadPoolName("billingApiAhcPool")
                .setIoThreadsCount(ahcCommonProperties.getIoThreads())
                .build()));
    }

    @Bean
    public AsyncHttpClient aeroflotAhcClient() {
        return withMetrics(Dsl.asyncHttpClient(Dsl.config()
                .setThreadPoolName("aeroflotAhcPool")
                .setIoThreadsCount(ahcCommonProperties.getIoThreads())
                .build()));
    }

    @Bean
    public AsyncHttpClient aviaApiAhcClient() {
        return withMetrics(Dsl.asyncHttpClient(Dsl.config()
                .setThreadPoolName("aviaApiAhcPool")
                .setIoThreadsCount(ahcCommonProperties.getIoThreads())
                .build()));
    }

    @Bean
    public AsyncHttpClient busApiAhcClient() {
        return withMetrics(Dsl.asyncHttpClient(Dsl.config()
                .setThreadPoolName("busApiAhcPool")
                .setIoThreadsCount(ahcCommonProperties.getIoThreads())
                .build()));
    }

    @Bean
    public AsyncHttpClient trainApiAhcClient() {
        return withMetrics(Dsl.asyncHttpClient(Dsl.config()
                .setThreadPoolName("trainApiAhcPool")
                .setIoThreadsCount(ahcCommonProperties.getIoThreads())
                .build()));
    }

    @Bean
    public AsyncHttpClient trustAhcClient() {
        return withMetrics(Dsl.asyncHttpClient(Dsl.config()
                .setThreadPoolName("trustAhcPool")
                .setIoThreadsCount(ahcCommonProperties.getIoThreads())
                .build()));
    }

    @Bean
    public AsyncHttpClient exchangeRateAhcClient() {
        return withMetrics(Dsl.asyncHttpClient(Dsl.config()
                .setThreadPoolName("exchangeRateAhcPool")
                .setIoThreadsCount(ahcCommonProperties.getIoThreads())
                .build()));
    }

    @Bean
    public AsyncHttpClient pdfGeneratorAhcClient() {
        return withMetrics(Dsl.asyncHttpClient(Dsl.config()
                .setThreadPoolName("pdfGeneratorAhcPool")
                .setIoThreadsCount(ahcCommonProperties.getIoThreads())
                .build()));
    }


    @Bean
    public AsyncHttpClient mailSenderAhcClient() {
        return withMetrics(Dsl.asyncHttpClient(Dsl.config()
                .setThreadPoolName("mailSenderAhcPool")
                .setIoThreadsCount(ahcCommonProperties.getIoThreads())
                .build()));
    }

    @Bean
    public AsyncHttpClient yaSmsAhcClient() {
        return withMetrics(Dsl.asyncHttpClient(Dsl.config()
                .setThreadPoolName("yaSmsAhcPool")
                .setIoThreadsCount(ahcCommonProperties.getIoThreads())
                .build()));
    }

    @Bean
    public AsyncHttpClient blackboxAhcClient() {
        return withMetrics(Dsl.asyncHttpClient(Dsl.config()
                .setThreadPoolName("blackboxAhcPool")
                .setIoThreadsCount(ahcCommonProperties.getIoThreads())
                .build()));
    }

    @Bean
    public AsyncHttpClient urlShortenerAhcClient() {
        return withMetrics(Dsl.asyncHttpClient(Dsl.config()
                .setThreadPoolName("urlShortenerAhcClient")
                .setIoThreadsCount(ahcCommonProperties.getIoThreads())
                .build()));
    }

    @Bean
    public AsyncHttpClient movistaAhcClient() {
        return withMetrics(Dsl.asyncHttpClient(Dsl.config()
                .setThreadPoolName("movistaAhcPool")
                .setIoThreadsCount(ahcCommonProperties.getIoThreads())
                .build()));
    }

    @Bean
    public AsyncHttpClient imSuburbanAhcClient() {
        return withMetrics(Dsl.asyncHttpClient(Dsl.config()
                .setThreadPoolName("imSuburbanAhcPool")
                .setIoThreadsCount(ahcCommonProperties.getIoThreads())
                .build()));
    }

    @Bean
    @SneakyThrows
    public DefaultAsyncHttpClientConfig.Builder aeroexpressAhcConfig(SuburbanProperties props) {
        DefaultAeroexpressClient.SslConfig sslConfig = props.getProviders().getAeroexpress().getClient().getSsl();
        if (!sslConfig.isEnabled()) {
            return Dsl.config();
        }

        InputStream keyStream;
        InputStream crtStream;
        if (!sslConfig.isGetFromFiles()) {
            keyStream = getPropertyStream(sslConfig.getPrivateKey());
            crtStream = getPropertyStream(sslConfig.getClientCertificate());
        } else { // Для локальной отладки
            keyStream = new FileInputStream(sslConfig.getPrivateKeyFile());
            crtStream = new FileInputStream(sslConfig.getClientCertificateFile());
        }

        try (keyStream; crtStream) {
            try {
                SslContext sslContext = SslContextBuilder.forClient()
                        .sslProvider(SslProvider.JDK)
                        .keyManager(crtStream, keyStream)
                        .build();
                return Dsl.config().setSslContext(sslContext);
            } catch (SSLException ex) {
                log.error("Aeroexpress client certificate or private key error", ex);
                throw ex;
            }
        }
    }

    private InputStream getPropertyStream(String propertyValue) {
        return new ByteArrayInputStream(propertyValue.getBytes());
    }

    @Bean
    public AsyncHttpClient aeroexpressAhcClient(DefaultAsyncHttpClientConfig.Builder aeroexpressAhcConfig) {
        return withMetrics(Dsl.asyncHttpClient(aeroexpressAhcConfig
                .setThreadPoolName("aeroexpressAhcPool")
                .setIoThreadsCount(ahcCommonProperties.getIoThreads())
                .build()));
    }

    @Bean
    public AsyncHttpClientWrapper travellineAhcClientWrapper(@Qualifier("hotelAhcClient") AsyncHttpClient client,
                                                             Tracer tracer) {
        return new AsyncHttpClientWrapper(client,
                LoggerFactory.getLogger("ru.yandex.travel.orders.partners.travelline.HttpLogger"),
                "travelline",
                tracer,
                DefaultTravellineClient.getMethods().getNames());
    }

    @Bean
    public AsyncHttpClientWrapper bNovoAhcClientWrapper(@Qualifier("hotelAhcClient") AsyncHttpClient client,
                                                        Tracer tracer) {
        return new AsyncHttpClientWrapper(client,
                LoggerFactory.getLogger("ru.yandex.travel.orders.partners.bnovo.HttpLogger"),
                "bnovo",
                tracer,
                DefaultBNovoClient.getMethods().getNames());
    }

    @Bean
    public AsyncHttpClientWrapper dolphinAhcClientWrapper(@Qualifier("hotelAhcClient") AsyncHttpClient client,
                                                          Tracer tracer) {
        return new AsyncHttpClientWrapper(client,
                LoggerFactory.getLogger("ru.yandex.travel.orders.partners.dolphin.HttpLogger"),
                "dolphin",
                tracer,
                DefaultDolphinClient.getMethods().getNames());
    }

    @Bean
    public AsyncHttpClientWrapper expediaAhcClientWrapper(@Qualifier("hotelAhcClient") AsyncHttpClient client,
                                                          Tracer tracer) {
        return new AsyncHttpClientWrapper(client,
                LoggerFactory.getLogger("ru.yandex.travel.orders.partners.expedia.HttpLogger"),
                "expedia",
                tracer,
                DefaultExpediaClient.getMethods().getNames());
    }

    @Bean
    public AsyncHttpClientWrapper billingApiAhcClientWrapper(@Qualifier("billingApiAhcClient") AsyncHttpClient client,
                                                             Tracer tracer) {
        return new AsyncHttpClientWrapper(client,
                LoggerFactory.getLogger("ru.yandex.travel.orders.partners.billingApi.HttpLogger"),
                "billingApi",
                tracer,
                null);
    }

    @Bean
    public AsyncHttpClientWrapper aeroflotAhcClientWrapper(@Qualifier("aeroflotAhcClient") AsyncHttpClient client,
                                                           Tracer tracer) {
        return new AsyncHttpClientWrapper(client,
                LoggerFactory.getLogger("ru.yandex.travel.orders.partners.aeroflot.HttpLogger"),
                "aeroflot",
                tracer,
                null); //TODO (mbobrov, tlg-13): add methods here, after refactoring inner HttpClient
    }

    @Bean
    public AsyncHttpClientWrapper aviaApiAhcClientWrapper(@Qualifier("aviaApiAhcClient") AsyncHttpClient client,
                                                          Tracer tracer) {
        return new AsyncHttpClientWrapper(client,
                LoggerFactory.getLogger("ru.yandex.travel.orders.avia.api.HttpLogger"),
                "avia", tracer);
    }

    @Bean
    public AsyncHttpClientWrapper trustAhcClientWrapper(@Qualifier("trustAhcClient") AsyncHttpClient client,
                                                        Tracer tracer) {
        return new AsyncHttpClientWrapper(client,
                LoggerFactory.getLogger("ru.yandex.travel.orders.services.payments.trust.HttpLogger"),
                "trust",
                tracer,
                DefaultTrustClient.Method.getNames());
    }

    @Bean
    public AsyncHttpClientWrapper exchangeRateAhcClientWrapper(@Qualifier("exchangeRateAhcClient") AsyncHttpClient client,
                                                               Tracer tracer) {
        return new AsyncHttpClientWrapper(client,
                LoggerFactory.getLogger("ru.yandex.travel.orders.services.payments.stocks.HttpLogger"),
                "stocks", tracer);
    }

    @Bean
    public AsyncHttpClientWrapper pdfGeneratorAhcClientWrapper(@Qualifier("pdfGeneratorAhcClient") AsyncHttpClient client,
                                                               Tracer tracer) {
        return new AsyncHttpClientWrapper(client,
                LoggerFactory.getLogger("ru.yandex.travel.orders.management.voucher.HttpLogger"),
                "pdfGenerator",
                tracer,
                Arrays.stream(PdfGeneratorServiceImpl.Method.values()).map(Enum::toString).collect(Collectors.toSet()));
    }

    @Bean
    public AsyncHttpClientWrapper mailSenderAhcClientWrapper(@Qualifier("mailSenderAhcClient") AsyncHttpClient client,
                                                             Tracer tracer) {
        return new AsyncHttpClientWrapper(client,
                LoggerFactory.getLogger("ru.yandex.travel.orders.management.mail.HttpLogger"),
                "mailSender", tracer);
    }

    @Bean
    public AsyncHttpClientWrapper templatedMailerAhcClientWrapper(@Qualifier("mailSenderAhcClient") AsyncHttpClient client,
                                                                  Tracer tracer) {
        return new AsyncHttpClientWrapper(client,
                LoggerFactory.getLogger("ru.yandex.travel.orders.management.templated_mail_sender.HttpLogger"),
                "templatedMailSender", tracer);
    }

    @Bean
    public AsyncHttpClientWrapper yaSmsAhcClientWrapper(@Qualifier("yaSmsAhcClient") AsyncHttpClient client,
                                                        Tracer tracer) {
        return new AsyncHttpClientWrapper(client,
                LoggerFactory.getLogger("ru.yandex.travel.orders.management.yasms.HttpLogger"),
                "yaSms", tracer);
    }

    @Bean
    public AsyncHttpClientWrapper urlShortenerAhcClientWrapper(@Qualifier("urlShortenerAhcClient") AsyncHttpClient client,
                                                               Tracer tracer) {
        return new AsyncHttpClientWrapper(client,
                LoggerFactory.getLogger("ru.yandex.travel.orders.management.url_shortener.HttpLogger"),
                "urlShortener", tracer);
    }

    @Bean
    public AsyncHttpClientWrapper blackboxAhcClientWrapper(@Qualifier("blackboxAhcClient") AsyncHttpClient client,
                                                           Tracer tracer) {
        return new AsyncHttpClientWrapper(client,
                LoggerFactory.getLogger("ru.yandex.travel.orders.management.blackbox.HttpLogger"),
                "blackbox", tracer);
    }

    @Bean
    public AsyncHttpClientWrapper movistaAhcClientWrapper(
            @Qualifier("movistaAhcClient") AsyncHttpClient client, Tracer tracer) {
        return new AsyncHttpClientWrapper(client,
                LoggerFactory.getLogger("ru.yandex.travel.orders.partners.movista.HttpLogger"),
                "movista",
                tracer,
                MovistaClient.Endpoint.endpointsSet()
        );
    }

    @Bean
    public AsyncHttpClientWrapper imSuburbanAhcClientWrapper(
            @Qualifier("imSuburbanAhcClient") AsyncHttpClient client, Tracer tracer) {
        return new AsyncHttpClientWrapper(client,
                LoggerFactory.getLogger("ru.yandex.travel.orders.partners.imSuburban.HttpLogger"),
                "imSuburban",
                tracer,
                Arrays.stream(DefaultImClient.Method.values()).map(Enum::toString).collect(Collectors.toSet()));
    }

    @Bean
    public AsyncHttpClientWrapper aeroexpressAhcClientWrapper(
            @Qualifier("aeroexpressAhcClient") AsyncHttpClient aeroexpressAhcClient, Tracer tracer) {
        return new AsyncHttpClientWrapper(aeroexpressAhcClient,
                LoggerFactory.getLogger("ru.yandex.travel.orders.partners.aeroexpress.HttpLogger"),
                "aeroexpress",
                tracer,
                AeroexpressClient.Endpoint.endpointsSet()
        );
    }
}
