package ru.yandex.travel.orders.configurations;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import com.google.common.base.Preconditions;
import io.grpc.BindableService;
import io.grpc.ServerInterceptor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.aop.support.AopUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.stereotype.Component;

import ru.yandex.travel.grpc.GrpcService;
import ru.yandex.travel.grpc.GrpcServiceInterceptorConfigurer;
import ru.yandex.travel.grpc.interceptors.LoggingServerInterceptor;
import ru.yandex.travel.grpc.interceptors.TracerServerInterceptor;
import ru.yandex.travel.grpc.interceptors.TvmHeaderServerInterceptor;
import ru.yandex.travel.grpc.interceptors.UserCredentialsServerInterceptor;
import ru.yandex.travel.orders.grpc.AuthorizationGrpcService;
import ru.yandex.travel.orders.grpc.OrdersAdminService;
import ru.yandex.travel.orders.grpc.OrdersService;
import ru.yandex.travel.orders.grpc.PromoCodeUserService;
import ru.yandex.travel.orders.grpc.UserInfoGrpcService;

@Slf4j
@Component
@ConditionalOnProperty(prefix = "grpc", name = "enabled", havingValue = "true", matchIfMissing = true)
public class OrderGrpcServiceInterceptorConfigurer implements GrpcServiceInterceptorConfigurer {
    private static final Set<Class<? extends BindableService>> TVM_USER_INT_AUTH = Set.of(OrdersAdminService.class);
    private static final Set<Class<? extends BindableService>> TVM_USER_EXT_AUTH = Set.of(OrdersService.class,
            PromoCodeUserService.class,
            AuthorizationGrpcService.class,
            UserInfoGrpcService.class);

    private final LoggingServerInterceptor loggingInterceptor;
    private final TracerServerInterceptor tracerServerInterceptor;
    private final TvmHeaderServerInterceptor tvmHeaderServerInterceptor;
    private final UserCredentialsServerInterceptor userCredentialsInternalServerInterceptor;
    private final UserCredentialsServerInterceptor userCredentialsExternalServerInterceptor;

    @Autowired
    public OrderGrpcServiceInterceptorConfigurer(LoggingServerInterceptor loggingInterceptor,
                                                 @Autowired(required = false) TracerServerInterceptor tracerServerInterceptor,
                                                 @Autowired(required = false) TvmHeaderServerInterceptor tvmHeaderServerInterceptor,
                                                 @Autowired(required = false) @Qualifier(
                                                         "userCredentialsInternalServerInterceptor") UserCredentialsServerInterceptor userCredentialsInternalServerInterceptor,
                                                 @Autowired(required = false) @Qualifier(
                                                         "userCredentialsExternalServerInterceptor") UserCredentialsServerInterceptor userCredentialsExternalServerInterceptor) {
        this.loggingInterceptor = loggingInterceptor;
        this.tracerServerInterceptor = tracerServerInterceptor;
        this.tvmHeaderServerInterceptor = tvmHeaderServerInterceptor;
        this.userCredentialsInternalServerInterceptor = userCredentialsInternalServerInterceptor;
        this.userCredentialsExternalServerInterceptor = userCredentialsExternalServerInterceptor;
    }

    @Override
    public List<ServerInterceptor> getInterceptors(BindableService service) {
        String serviceName = service.getClass().getName();
        List<ServerInterceptor> interceptors = new ArrayList<>();

        Class<?> originalClass = AopUtils.getTargetClass(service);
        GrpcService annotation = originalClass.getAnnotation(GrpcService.class);
        Preconditions.checkNotNull(annotation);
        if (TVM_USER_EXT_AUTH.contains(originalClass)) {
            if (userCredentialsExternalServerInterceptor != null) {
                interceptors.add(userCredentialsExternalServerInterceptor);
                log.info("Will enforce external user authentication for service '{}'", serviceName);
            } else {
                log.warn("Service '{}' requires external user authentication, but appropriate interceptor is not " +
                        "configured", serviceName);
            }
        }
        if (TVM_USER_INT_AUTH.contains(originalClass)) {
            if (userCredentialsInternalServerInterceptor != null) {
                interceptors.add(userCredentialsInternalServerInterceptor);
                log.info("Will enforce internal user authentication for service '{}'", serviceName);
            } else {
                log.warn("Service '{}' requires internal user authentication, but appropriate interceptor is not " +
                        "configured", serviceName);
            }
        }
        if (annotation.authenticateService()) {
            if (tvmHeaderServerInterceptor != null) {
                interceptors.add(tvmHeaderServerInterceptor);
                log.info("Will enforce cross-service authentication for service '{}'", serviceName);
            } else {
                log.warn("Service '{}' requires cross-service authentication, but appropriate interceptor is not " +
                        "configured", serviceName);
            }
        }
        if (annotation.log()) {
            interceptors.add(loggingInterceptor);
        }
        if (annotation.trace()) {
            if (tracerServerInterceptor != null) {
                interceptors.add(tracerServerInterceptor);
                log.info("Will trace service calls for service '{}'", serviceName);
            } else {
                log.warn("Service '{}' requires tracing of calls, but appropriate interceptor is not configured",
                        serviceName);
            }
        }
        return interceptors;
    }
}
