package ru.yandex.travel.orders.configurations;

import java.time.Duration;

import lombok.extern.slf4j.Slf4j;
import org.quartz.JobDetail;
import org.quartz.SimpleTrigger;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.quartz.SchedulerFactoryBeanCustomizer;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.scheduling.quartz.CronTriggerFactoryBean;
import org.springframework.scheduling.quartz.JobDetailFactoryBean;
import org.springframework.scheduling.quartz.SchedulerFactoryBean;
import org.springframework.scheduling.quartz.SimpleTriggerFactoryBean;

import ru.yandex.travel.orders.services.avia.aeroflot.AeroflotMqJob;
import ru.yandex.travel.orders.services.avia.aeroflot.AeroflotMqProperties;
import ru.yandex.travel.orders.services.report.HotelPartnerReportSenderJob;
import ru.yandex.travel.orders.services.report.HotelsReportProperties;
import ru.yandex.travel.orders.services.support.jobs.ScheduleSuccessfulHotelOrderNotificationsJob;
import ru.yandex.travel.orders.services.support.jobs.ScheduleUnsuccessfulHotelOrderNotificationsJob;
import ru.yandex.travel.orders.services.support.jobs.SendSuccessfulHotelOrderNotificationsJob;
import ru.yandex.travel.orders.services.support.jobs.SendUnsuccessfulHotelOrderNotificationsJob;

@Configuration
@Slf4j
public class QuartzSchedulerConfiguration {
    @Bean
    public JobDetailFactoryBean aeroflotOrderSyncJob() {
        JobDetailFactoryBean result = new JobDetailFactoryBean();
        result.setJobClass(AeroflotMqJob.class);
        result.setDurability(true);
        return result;
    }

    @ConditionalOnProperty(value = "aeroflot-mq.enabled", havingValue = "true")
    @Bean
    public SimpleTriggerFactoryBean aeroflotOrderSyncJobTrigger(JobDetail aeroflotOrderSyncJob,
                                                                AeroflotMqProperties properties) {
        SimpleTriggerFactoryBean simpleTriggerFactoryBean = new SimpleTriggerFactoryBean();
        simpleTriggerFactoryBean.setJobDetail(aeroflotOrderSyncJob);
        simpleTriggerFactoryBean.setRepeatInterval(properties.getRepeatInterval().toMillis());
        simpleTriggerFactoryBean.setRepeatCount(SimpleTrigger.REPEAT_INDEFINITELY);
        return simpleTriggerFactoryBean;
    }

    @Bean
    public JobDetailFactoryBean hotelsPartnerReportPlannerJob() {
        JobDetailFactoryBean result = new JobDetailFactoryBean();
        result.setJobClass(HotelPartnerReportSenderJob.class);
        result.setDurability(true);
        return result;
    }

    @ConditionalOnProperty(value = "hotels-report.planner.enabled", havingValue = "true")
    @Bean
    public CronTriggerFactoryBean hotelsPartnerReportPlannerJobTrigger(JobDetail hotelsPartnerReportPlannerJob,
                                                                        HotelsReportProperties properties) {
        CronTriggerFactoryBean cronTriggerFactoryBean = new CronTriggerFactoryBean();
        cronTriggerFactoryBean.setJobDetail(hotelsPartnerReportPlannerJob);
        cronTriggerFactoryBean.setCronExpression(properties.getPlanner().getCronExpression());
        return cronTriggerFactoryBean;
    }


    public static class DisableAutoStartupCustomizer implements SchedulerFactoryBeanCustomizer {

        @Override
        public void customize(SchedulerFactoryBean schedulerFactoryBean) {
            log.info("Disabling automatic scheduler startup");
            schedulerFactoryBean.setAutoStartup(false);
        }
    }

    @Bean
    @ConditionalOnProperty(value = "quartz.enabled", havingValue = "false")
    public SchedulerFactoryBeanCustomizer schedulerFactoryBeanCustomizer() {
        return new DisableAutoStartupCustomizer();
    }

    public static class SupportNotificationsConfiguration {
        @Bean
        public JobDetailFactoryBean scheduleSuccessfulHotelOrderNotificationsJob() {
            JobDetailFactoryBean result = new JobDetailFactoryBean();
            result.setJobClass(ScheduleSuccessfulHotelOrderNotificationsJob.class);
            result.setDurability(true);
            return result;
        }

        @ConditionalOnProperty(value = "hotels-support.successful-orders.notifications-enabled")
        @Bean
        public SimpleTriggerFactoryBean scheduleSuccessfulHotelOrderNotificationsJobTrigger(JobDetail scheduleSuccessfulHotelOrderNotificationsJob) {
            SimpleTriggerFactoryBean simpleTriggerFactoryBean = new SimpleTriggerFactoryBean();
            simpleTriggerFactoryBean.setJobDetail(scheduleSuccessfulHotelOrderNotificationsJob);
            simpleTriggerFactoryBean.setRepeatInterval(Duration.ofSeconds(60).toMillis());
            simpleTriggerFactoryBean.setRepeatCount(SimpleTrigger.REPEAT_INDEFINITELY);
            return simpleTriggerFactoryBean;
        }

        @Bean
        public JobDetailFactoryBean sendSuccessfulHotelOrderNotificationsJob() {
            JobDetailFactoryBean result = new JobDetailFactoryBean();
            result.setJobClass(SendSuccessfulHotelOrderNotificationsJob.class);
            result.setDurability(true);
            return result;
        }

        @ConditionalOnProperty(value = "hotels-support.successful-orders.notifications-enabled")
        @Bean
        public SimpleTriggerFactoryBean sendSuccessfulHotelOrderNotificationsJobTrigger(JobDetail sendSuccessfulHotelOrderNotificationsJob) {
            SimpleTriggerFactoryBean simpleTriggerFactoryBean = new SimpleTriggerFactoryBean();
            simpleTriggerFactoryBean.setJobDetail(sendSuccessfulHotelOrderNotificationsJob);
            simpleTriggerFactoryBean.setRepeatInterval(Duration.ofSeconds(60).toMillis());
            simpleTriggerFactoryBean.setRepeatCount(SimpleTrigger.REPEAT_INDEFINITELY);
            return simpleTriggerFactoryBean;
        }

        @Bean
        public JobDetailFactoryBean scheduleUnsuccessfulHotelOrderNotificationsJob() {
            JobDetailFactoryBean result = new JobDetailFactoryBean();
            result.setJobClass(ScheduleUnsuccessfulHotelOrderNotificationsJob.class);
            result.setDurability(true);
            return result;
        }

        @ConditionalOnProperty(value = "hotels-support.unsuccessful-orders.notifications-enabled")
        @Bean
        public SimpleTriggerFactoryBean scheduleUnsuccessfulHotelOrderNotificationsJobTrigger(JobDetail scheduleUnsuccessfulHotelOrderNotificationsJob) {
            SimpleTriggerFactoryBean simpleTriggerFactoryBean = new SimpleTriggerFactoryBean();
            simpleTriggerFactoryBean.setJobDetail(scheduleUnsuccessfulHotelOrderNotificationsJob);
            simpleTriggerFactoryBean.setRepeatInterval(Duration.ofSeconds(60).toMillis());
            simpleTriggerFactoryBean.setRepeatCount(SimpleTrigger.REPEAT_INDEFINITELY);
            return simpleTriggerFactoryBean;
        }

        @Bean
        public JobDetailFactoryBean sendUnsuccessfulHotelOrderNotificationsJob() {
            JobDetailFactoryBean result = new JobDetailFactoryBean();
            result.setJobClass(SendUnsuccessfulHotelOrderNotificationsJob.class);
            result.setDurability(true);
            return result;
        }

        @ConditionalOnProperty(value = "hotels-support.unsuccessful-orders.notifications-enabled")
        @Bean
        public SimpleTriggerFactoryBean sendUnsuccessfulHotelOrderNotificationsJobTrigger(JobDetail sendUnsuccessfulHotelOrderNotificationsJob) {
            SimpleTriggerFactoryBean simpleTriggerFactoryBean = new SimpleTriggerFactoryBean();
            simpleTriggerFactoryBean.setJobDetail(sendUnsuccessfulHotelOrderNotificationsJob);
            simpleTriggerFactoryBean.setRepeatInterval(Duration.ofSeconds(60).toMillis());
            simpleTriggerFactoryBean.setRepeatCount(SimpleTrigger.REPEAT_INDEFINITELY);
            return simpleTriggerFactoryBean;
        }
    }
}
