package ru.yandex.travel.orders.configurations;

import com.amazonaws.auth.AWSCredentials;
import com.amazonaws.auth.AWSStaticCredentialsProvider;
import com.amazonaws.auth.BasicAWSCredentials;
import com.amazonaws.auth.BasicSessionCredentials;
import com.amazonaws.client.builder.AwsClientBuilder.EndpointConfiguration;
import com.amazonaws.services.s3.AmazonS3;
import com.amazonaws.services.s3.AmazonS3ClientBuilder;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.orders.services.cloud.s3.S3ServiceProperties;
import ru.yandex.travel.tvm.TvmWrapper;

@Configuration
public class S3Configuration {
    @Bean
    public AmazonS3 s3client(S3ServiceProperties properties, @Autowired(required = false) TvmWrapper tvm) {
        EndpointConfiguration endpoint = new EndpointConfiguration(properties.getEndpoint(), null);
        AWSCredentials credentials;
        if (properties.getUseTvm() == Boolean.TRUE) {
            Preconditions.checkArgument(tvm != null, "tvm is enabled but not provided");
            credentials = new TvmSessionCredentials(tvm, properties.getTvmAlias());
        } else {
            Preconditions.checkArgument(!Strings.isNullOrEmpty(properties.getAccessKey()), "no access key");
            Preconditions.checkArgument(!Strings.isNullOrEmpty(properties.getAccessSecret()), "no access secret");
            credentials = new BasicAWSCredentials(properties.getAccessKey(), properties.getAccessSecret());
        }
        return AmazonS3ClientBuilder.standard()
                .withEndpointConfiguration(endpoint)
                .withCredentials(new AWSStaticCredentialsProvider(credentials))
                .build();
    }

    private static class TvmSessionCredentials extends BasicSessionCredentials {
        private final TvmWrapper tvm;
        private final String dstTvmAlias;

        public TvmSessionCredentials(TvmWrapper tvm, String dstTvmAlias) {
            super("TVM_V2_" + tvm.getSelfTvmId(), "unused", null);
            this.tvm = tvm;
            this.dstTvmAlias = dstTvmAlias;

            tvm.validateAlias(dstTvmAlias);
        }

        @Override
        public String getSessionToken() {
            return "TVM2 " + tvm.getServiceTicket(dstTvmAlias);
        }
    }
}
