package ru.yandex.travel.orders.configurations;

import java.util.concurrent.TimeUnit;
import java.util.function.Supplier;

import com.netflix.concurrency.limits.Limit;
import com.netflix.concurrency.limits.MetricRegistry;
import com.netflix.concurrency.limits.limit.AIMDLimit;
import com.netflix.concurrency.limits.limit.FixedLimit;
import com.netflix.concurrency.limits.limiter.SimpleLimiter;
import io.micrometer.core.instrument.Gauge;
import io.micrometer.core.instrument.Metrics;
import lombok.RequiredArgsConstructor;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.orders.workflows.orderitem.train.TrainWorkflowProperties;

@Configuration
@RequiredArgsConstructor
public class TrainLimitersConfiguration {
    private final TrainWorkflowProperties trainWorkflowProperties;

    public static MetricRegistry limiterCallMeterRegistry(String name) {
        return new MetricRegistry() {
            @Override
            public SampleListener distribution(String id, String... tagNameValuePairs) {
                return value -> {
                };
            }

            @Override
            public void gauge(String id, Supplier<Number> supplier, String... tagNameValuePairs) {
            }

            // only need this to work with ignored, rejected, dropped, and success counters
            @Override
            public Counter counter(String id, String... tagNameValuePairs) {
                return () -> io.micrometer.core.instrument.Counter.builder("limiter." + id)
                        .tags(tagNameValuePairs).register(Metrics.globalRegistry).increment();
            }
        };
    }

    public static void registerLimiterGauges(String name, SimpleLimiter<Void> limiter) {
        Gauge.builder("limiter.inflight", limiter::getInflight).tag("name", name).register(Metrics.globalRegistry);
        Gauge.builder("limiter.current_limit", limiter::getLimit).tag("name", name).register(Metrics.globalRegistry);
    }

    // don't make it a bean for now, as we don't have a clear vision on other limiter usages
    @Bean
    public SimpleLimiter<Void> reserveCallLimiter() {
        return createLimiter(trainWorkflowProperties.getReservationConcurrencyLimit(),
                "train_reservation");
    }

    @Bean
    public SimpleLimiter<Void> updateOnTheFlyLimiter() {
        return createLimiter(trainWorkflowProperties.getUpdateOnTheFlyConcurrencyLimit(),
                "update_on_the_fly");
    }

    private SimpleLimiter<Void> createLimiter(TrainWorkflowProperties.LimiterProperties props,
                                              String limiterName) {
        Limit limit;
        if (props.getMode() ==
                TrainWorkflowProperties.ConcurrencyLimitMode.ADAPTIVE) {
            limit = AIMDLimit.newBuilder()
                    .initialLimit(props.getAdaptive().getMaxConcurrentCalls())
                    .minLimit(props.getAdaptive().getMinConcurrentCalls())
                    .maxLimit(props.getAdaptive().getMaxConcurrentCalls())
                    .timeout(props.getAdaptive().getMaxTimeout().toMillis(), TimeUnit.MILLISECONDS)
                    .build();
        } else {
            // no limits
            limit = FixedLimit.of(props.getFixed().getLimit());
        }

        SimpleLimiter<Void> result = SimpleLimiter.newBuilder()
                .limit(limit)
                .named(limiterName)
                .metricRegistry(limiterCallMeterRegistry(limiterName))
                .build();

        registerLimiterGauges(limiterName, result);
        return result;
    }
}
