package ru.yandex.travel.orders.configurations;

import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnMissingBean;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;

import ru.yandex.travel.PropertyUtils;
import ru.yandex.travel.commons.logging.AsyncHttpClientWrapper;
import ru.yandex.travel.credentials.UserCredentialsAuthValidator;
import ru.yandex.travel.credentials.UserCredentialsAuthValidatorStubImpl;
import ru.yandex.travel.credentials.UserCredentialsAuthValidatorTvmImpl;
import ru.yandex.travel.credentials.UserCredentialsBuilder;
import ru.yandex.travel.credentials.UserCredentialsPassportExtractor;
import ru.yandex.travel.credentials.UserCredentialsPassportExtractorStubImpl;
import ru.yandex.travel.credentials.UserCredentialsPassportExtractorTvmImpl;
import ru.yandex.travel.credentials.UserCredentialsProperties;
import ru.yandex.travel.credentials.UserCredentialsValidator;
import ru.yandex.travel.grpc.GrpcTvmProperties;
import ru.yandex.travel.grpc.interceptors.TvmHeaderServerInterceptor;
import ru.yandex.travel.grpc.interceptors.UserCredentialsServerInterceptor;
import ru.yandex.travel.orders.services.BlackboxProperties;
import ru.yandex.travel.orders.services.BlackboxService;
import ru.yandex.travel.orders.services.DefaultUserInfoService;
import ru.yandex.travel.orders.services.UserInfoService;
import ru.yandex.travel.tvm.TvmHelper;
import ru.yandex.travel.tvm.TvmWrapper;

@Configuration
@EnableConfigurationProperties({GrpcTvmProperties.class, TvmServiceProperties.class, TvmUserInternalProperties.class,
        TvmUserExternalProperties.class, UserCredentialsProperties.class, BlackboxProperties.class})
@RequiredArgsConstructor
public class TvmConfiguration {
    private final GrpcTvmProperties grpcTvmProperties;
    private final TvmServiceProperties tvmServiceProperties;
    private final TvmUserInternalProperties tvmUserInternalProperties;
    private final TvmUserExternalProperties tvmUserExternalProperties;
    private final UserCredentialsProperties userCredentialsProperties;

    @Bean
    @Primary
    @ConditionalOnProperty(value = "tvm-service.enabled")
    public TvmWrapper tvmService() {
        return TvmHelper.getTvmWrapper(tvmServiceProperties);
    }

    @Bean
    @ConditionalOnProperty(value = "tvm-internal.enabled")
    public TvmWrapper tvmUserInternal() {
        return TvmHelper.getTvmWrapper(tvmUserInternalProperties);
    }

    @Bean
    @ConditionalOnProperty(value = "tvm-external.enabled")
    public TvmWrapper tvmUserExternal() {
        return TvmHelper.getTvmWrapper(tvmUserExternalProperties);
    }

    @Bean
    @ConditionalOnProperty("grpc-tvm.enabled")
    public TvmHeaderServerInterceptor tvmHeaderServerInterceptor(@Autowired @Qualifier("tvmService") TvmWrapper tvm) {
        return new TvmHeaderServerInterceptor(tvm,
                PropertyUtils.stringToListOfStrings(grpcTvmProperties.getAllowedConsumers()));
    }

    @Bean
    @ConditionalOnProperty("credentials.auth-enabled")
    public UserCredentialsPassportExtractor userCredentialsInternalPassportExtractor(@Qualifier(
            "tvmUserInternal") TvmWrapper tvm) {
        return new UserCredentialsPassportExtractorTvmImpl(tvm);
    }

    @Bean
    @Primary
    @ConditionalOnProperty("credentials.auth-enabled")
    public UserCredentialsPassportExtractor userCredentialsExternalPassportExtractor(@Qualifier(
            "tvmUserExternal") TvmWrapper tvm) {
        return new UserCredentialsPassportExtractorTvmImpl(tvm);
    }

    @Bean(name = {"userCredentialsInternalPassportExtractor", "userCredentialsExternalPassportExtractor"})
    @ConditionalOnProperty(value = "credentials.auth-enabled", havingValue = "false")
    public UserCredentialsPassportExtractor userCredentialsStubPassportExtractor() {
        return new UserCredentialsPassportExtractorStubImpl();
    }

    @Bean
    @ConditionalOnProperty("credentials.auth-enabled")
    public UserCredentialsAuthValidator userCredentialsInternalAuthValidator(@Qualifier(
            "tvmUserInternal") TvmWrapper tvm) {
        return new UserCredentialsAuthValidatorTvmImpl(tvm);
    }

    @Bean
    @Primary
    @ConditionalOnProperty("credentials.auth-enabled")
    public UserCredentialsAuthValidator userCredentialsExternalAuthValidator(@Qualifier(
            "tvmUserExternal") TvmWrapper tvm) {
        return new UserCredentialsAuthValidatorTvmImpl(tvm);
    }

    @Bean(name = {"userCredentialsInternalAuthValidator", "userCredentialsExternalAuthValidator"})
    @ConditionalOnProperty(value = "credentials.auth-enabled", havingValue = "false")
    public UserCredentialsAuthValidator userCredentialsStubAuthValidator() {
        return new UserCredentialsAuthValidatorStubImpl();
    }

    @Bean
    @ConditionalOnProperty(value = "credentials.enabled")
    public UserCredentialsServerInterceptor userCredentialsInternalServerInterceptor(
            UserCredentialsBuilder userCredentialsBuilder,
            @Qualifier("userCredentialsInternalPassportExtractor") UserCredentialsPassportExtractor userCredentialsPassportExtractor,
            @Qualifier("userCredentialsInternalAuthValidator") UserCredentialsAuthValidator userCredentialsAuthValidator
    ) {
        return new UserCredentialsServerInterceptor(userCredentialsBuilder, userCredentialsPassportExtractor,
                new UserCredentialsValidator(userCredentialsAuthValidator));
    }

    @Bean
    @ConditionalOnProperty(value = "credentials.enabled")
    public UserCredentialsServerInterceptor userCredentialsExternalServerInterceptor(
            UserCredentialsBuilder userCredentialsBuilder,
            @Qualifier("userCredentialsExternalPassportExtractor") UserCredentialsPassportExtractor userCredentialsPassportExtractor,
            @Qualifier("userCredentialsExternalAuthValidator") UserCredentialsAuthValidator userCredentialsAuthValidator
    ) {
        return new UserCredentialsServerInterceptor(userCredentialsBuilder, userCredentialsPassportExtractor,
                new UserCredentialsValidator(userCredentialsAuthValidator));
    }

    @Bean
    @ConditionalOnProperty(value = "tvm-service.enabled")
    public UserInfoService userInfoService(BlackboxProperties blackboxProperties,
                                           @Qualifier("tvmService") TvmWrapper tvm,
                                           @Qualifier("blackboxAhcClientWrapper") AsyncHttpClientWrapper httpClient) {
        return new BlackboxService(blackboxProperties, tvm, httpClient);
    }

    @Bean
    @ConditionalOnMissingBean(UserInfoService.class)
    public UserInfoService defaultUserInfoService() {
        return new DefaultUserInfoService();
    }
}
