package ru.yandex.travel.orders.configurations.jdbc;

import java.lang.reflect.Method;
import java.util.concurrent.TimeUnit;

import javax.sql.DataSource;

import net.ttddyy.dsproxy.listener.logging.DefaultJsonQueryLogEntryCreator;
import net.ttddyy.dsproxy.listener.logging.SLF4JSlowQueryListener;
import net.ttddyy.dsproxy.support.ProxyDataSource;
import net.ttddyy.dsproxy.support.ProxyDataSourceBuilder;
import org.aopalliance.intercept.MethodInterceptor;
import org.aopalliance.intercept.MethodInvocation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.aop.framework.ProxyFactory;
import org.springframework.beans.factory.config.BeanPostProcessor;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.util.ReflectionUtils;

@Configuration
@EnableConfigurationProperties(DatasourceProxyProperties.class)
public class DatasourceProxyConfiguration {

    private static final Logger SLOW_QUERY_LOGGER = LoggerFactory.getLogger("ru.yandex.travel.jdbc.SlowQuery");

    @Bean
    public DatasourceProxyBeanPostProcessor datasourceProxyBeanPostProcessor(DatasourceProxyProperties props) {
        return new DatasourceProxyBeanPostProcessor(props);
    }

    public static class DatasourceProxyBeanPostProcessor implements BeanPostProcessor {

        private final DatasourceProxyProperties props;

        public DatasourceProxyBeanPostProcessor(DatasourceProxyProperties props) {
            this.props = props;
        }

        @Override
        public Object postProcessAfterInitialization(Object bean, String beanName) {
            if (bean instanceof DataSource && !(bean instanceof ProxyDataSource)) {
                // Instead of directly returning a less specific datasource bean
                // (e.g.: HikariDataSource -> DataSource), return a proxy object.
                // See following links for why:
                //   https://stackoverflow.com/questions/44237787/how-to-use-user-defined-database-proxy-in-datajpatest
                //   https://gitter.im/spring-projects/spring-boot?at=5983602d2723db8d5e70a904
                //   http://blog.arnoldgalovics.com/2017/06/26/configuring-a-datasource-proxy-in-spring-boot/
                final ProxyFactory factory = new ProxyFactory(bean);
                factory.setProxyTargetClass(true);
                factory.addAdvice(new ProxyDataSourceInterceptor((DataSource) bean, props));
                return factory.getProxy();
            }
            return bean;
        }

        @Override
        public Object postProcessBeforeInitialization(Object bean, String beanName) {
            return bean;
        }

    }

    private static class ProxyDataSourceInterceptor implements MethodInterceptor {
        private final DataSource dataSource;

        ProxyDataSourceInterceptor(DataSource dataSource, DatasourceProxyProperties props) {
            SLF4JSlowQueryListener slowQueryLogListener =
                    new SLF4JSlowQueryListener(props.getSlowQueryThreshold().toMillis(),
                            TimeUnit.MILLISECONDS);
            slowQueryLogListener.setQueryLogEntryCreator(new DefaultJsonQueryLogEntryCreator());
            slowQueryLogListener.setWriteDataSourceName(false);
            slowQueryLogListener.setWriteConnectionId(false);
            slowQueryLogListener.setLogger(SLOW_QUERY_LOGGER);

            this.dataSource = ProxyDataSourceBuilder.create(dataSource)
                    .name("orderAppDs")
                    .listener(new QueryStatsExecutionListener())
                    .listener(slowQueryLogListener)
                    .build();
        }

        @Override
        public Object invoke(final MethodInvocation invocation) throws Throwable {
            final Method proxyMethod = ReflectionUtils.findMethod(this.dataSource.getClass(),
                    invocation.getMethod().getName());
            if (proxyMethod != null) {
                return proxyMethod.invoke(this.dataSource, invocation.getArguments());
            }
            return invocation.proceed();
        }
    }

}
