package ru.yandex.travel.orders.configurations.jdbc;

import java.util.EnumMap;
import java.util.Map;

import javax.sql.DataSource;

import com.zaxxer.hikari.HikariDataSource;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.jdbc.DataSourceProperties;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.jdbc.datasource.lookup.AbstractRoutingDataSource;
import org.springframework.util.StringUtils;

@Configuration
@ConditionalOnProperty("multiple-datasource.enabled")
public class MultipleDataSourceConfiguration {

    @Bean
    @ConfigurationProperties("multiple-datasource")
    public MultipleDataSourceConfigurationProperties multipleDataSourceConfigurationProperties() {
        return new MultipleDataSourceConfigurationProperties();
    }

    @SuppressWarnings("unchecked")
    protected static <T> T createDataSource(DataSourceProperties properties,
                                            Class<? extends DataSource> type) {
        return (T) properties.initializeDataSourceBuilder().type(type).build();
    }

    @Bean
    @ConfigurationProperties(prefix = "multiple-datasource.primary-datasource.hikari")
    public DataSource primaryDatasource(MultipleDataSourceConfigurationProperties properties) {
        return createDataSource(properties.getPrimaryDatasource());
    }

    @Bean
    @ConfigurationProperties(prefix = "multiple-datasource.grpc-datasource.hikari")
    public DataSource grpcDatasource(MultipleDataSourceConfigurationProperties properties) {
        return createDataSource(properties.getGrpcDatasource());
    }

    @Bean
    @ConfigurationProperties(prefix = "multiple-datasource.read-only.hikari")
    public DataSource readOnlyDatasource(MultipleDataSourceConfigurationProperties properties) {
        return createDataSource(properties.getReadOnlyDatasource());
    }

    private DataSource createDataSource(DataSourceProperties props) {
        HikariDataSource dataSource = createDataSource(props, HikariDataSource.class);
        if (StringUtils.hasText(props.getName())) {
            dataSource.setPoolName(props.getName());
        }
        return dataSource;
    }

    @Bean
    @Primary
    public DataSource dataSource(
            @Qualifier("primaryDatasource") DataSource primaryDataSource,
            @Qualifier("grpcDatasource") DataSource grpcDataSource,
            @Qualifier("readOnlyDatasource") DataSource readOnlyDataSource
    ) {
        CustomRoutingDatasource result = new CustomRoutingDatasource();
        result.setTargetDataSources(
                new EnumMap(Map.of(
                        TxScopeType.DEFAULT, primaryDataSource,
                        TxScopeType.GRPC, grpcDataSource,
                        TxScopeType.READ_ONLY, readOnlyDataSource
                ))
        );
        result.setDefaultTargetDataSource(primaryDataSource);
        return result;
    }

    // WARNING! Don't make this class final so a proxy can be created from it (ProxyDatasource for slow sql log)
    public static class CustomRoutingDatasource extends AbstractRoutingDataSource {

        public CustomRoutingDatasource() {
        }

        @Override
        protected Object determineCurrentLookupKey() {
            return TxScope.currentScopeType();
        }
    }
}
