package ru.yandex.travel.orders.configurations.jdbc;

import java.time.Duration;
import java.util.List;
import java.util.Map;
import java.util.concurrent.TimeUnit;

import com.google.common.collect.ImmutableMap;
import io.micrometer.core.instrument.Counter;
import io.micrometer.core.instrument.Metrics;
import io.micrometer.core.instrument.Timer;
import net.ttddyy.dsproxy.ExecutionInfo;
import net.ttddyy.dsproxy.QueryInfo;
import net.ttddyy.dsproxy.QueryType;
import net.ttddyy.dsproxy.listener.QueryExecutionListener;
import net.ttddyy.dsproxy.listener.QueryUtils;

import ru.yandex.travel.commons.metrics.MetricsUtils;

public class QueryStatsExecutionListener implements QueryExecutionListener {
    private static final Duration[] EXECUTION_DURATION_SLA = {
            Duration.ofMillis(1), Duration.ofMillis(10), Duration.ofMillis(100),
            Duration.ofMillis(1000), Duration.ofMillis(10000), Duration.ofMillis(100000),
            Duration.ofMillis(500000)
    };

    private final Map<QueryType, Counter> successCounters;
    private final Map<QueryType, Counter> failureCounters;
    private final Timer executionSuccessTimer;
    private final Timer executionFailureTimer;

    public QueryStatsExecutionListener() {
        ImmutableMap.Builder<QueryType, Counter> successCountersBuilder = new ImmutableMap.Builder<>();
        ImmutableMap.Builder<QueryType, Counter> failureCountersBuilder = new ImmutableMap.Builder<>();
        for (QueryType queryType : QueryType.values()) {
            successCountersBuilder.put(queryType, Counter.builder("datasource.query.statement")
                    .tag("statement_type", queryType.toString())
                    .tag("result_type", "success")
                    .register(Metrics.globalRegistry)
            );

            failureCountersBuilder.put(queryType, Counter.builder("datasource.query.statement")
                    .tag("statement_type", queryType.toString())
                    .tag("result_type", "failure")
                    .register(Metrics.globalRegistry));
        }

        successCounters = successCountersBuilder.build();
        failureCounters = failureCountersBuilder.build();
        executionSuccessTimer = Timer.builder("datasource.query.execution")
                .tag("result_type", "success")
                .serviceLevelObjectives(EXECUTION_DURATION_SLA)
                .publishPercentiles(MetricsUtils.higherPercentiles())
                .register(Metrics.globalRegistry);
        executionFailureTimer = Timer.builder("datasource.query.execution")
                .tag("result_type", "failure")
                .serviceLevelObjectives(EXECUTION_DURATION_SLA)
                .publishPercentiles(MetricsUtils.higherPercentiles())
                .register(Metrics.globalRegistry);

    }

    @Override
    public void beforeQuery(ExecutionInfo execInfo, List<QueryInfo> queryInfoList) {
    }

    @Override
    public void afterQuery(ExecutionInfo execInfo, List<QueryInfo> queryInfoList) {

        final long elapsedTime = execInfo.getElapsedTime();
        if (execInfo.isSuccess()) {
            executionSuccessTimer.record(elapsedTime, TimeUnit.MILLISECONDS);
        } else {
            executionFailureTimer.record(elapsedTime, TimeUnit.MILLISECONDS);
        }

        for (QueryInfo queryInfo : queryInfoList) {
            final String query = queryInfo.getQuery();
            final QueryType type = QueryUtils.getQueryType(query);
            if (execInfo.isSuccess()) {
                successCounters.get(type).increment();
            } else {
                failureCounters.get(type).increment();
            }
        }
    }
}
