package ru.yandex.travel.orders.configurations.jdbc;

import java.util.Objects;
import java.util.function.Supplier;

import com.google.common.base.Preconditions;

public class TxScope {
    private final static ThreadLocal<TxScopeType> scopeType = new ThreadLocal<>();

    public static TxScopeType currentScopeType() {
        TxScopeType current = scopeType.get();
        return Objects.requireNonNullElse(current, TxScopeType.DEFAULT);
    }

    public static void runWithTxScope(TxScopeType desiredType, Runnable runnable) {
        ensureCurrentScopeNotChanged(desiredType);
        try {
            scopeType.set(desiredType);
            runnable.run();
        } finally {
            scopeType.remove();
        }
    }

    public static <T> T supplyWithTxScope(TxScopeType desiredType, Supplier<T> supplier) {
        ensureCurrentScopeNotChanged(desiredType);
        try {
            scopeType.set(desiredType);
            return supplier.get();
        } finally {
            scopeType.remove();
        }
    }

    private static void ensureCurrentScopeNotChanged(TxScopeType desiredType) {
        TxScopeType current = scopeType.get();
        Preconditions.checkState(current == null || current == desiredType,
                "SEVERE ERROR! The code is trying to change tx scope " +
                        "level from %s to %s", current, desiredType);
    }
}
