package ru.yandex.travel.orders.entities;

import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.UUID;

import javax.persistence.Entity;
import javax.persistence.Id;
import javax.persistence.OneToMany;
import javax.persistence.Table;

import com.google.common.base.Preconditions;
import lombok.Data;
import org.hibernate.annotations.BatchSize;
import org.hibernate.annotations.CreationTimestamp;
import org.hibernate.annotations.Type;
import org.hibernate.annotations.UpdateTimestamp;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;

@Entity
@Table(name = "accounts")
@Data
@BatchSize(size = 50)
public class Account {

    @Id
    private UUID id;

    @CreationTimestamp
    private LocalDateTime createdAt;

    @UpdateTimestamp
    private LocalDateTime updatedAt;

    @OneToMany(mappedBy = "account")
    private List<AccountRecord> accountRecords;

    @Type(type = "proto-currency-unit")
    private ProtoCurrencyUnit currency;

    public static Account createWellKnownAccount(WellKnownAccount wkAcc) {
        Account result = createAccountInner(wkAcc.getCurrency());
        result.setId(wkAcc.getUuid());
        return result;
    }

    public static Account createAccount(ProtoCurrencyUnit currencyUnit) {
        Account result = createAccountInner(currencyUnit);
        UUID id;
        while (true) {
            id = UUID.randomUUID();
            UUID finalId = id;
            if (Arrays.stream(WellKnownAccount.values()).noneMatch(it -> it.getUuid().equals(finalId))) {
                break;
            }
        }
        result.setId(id);
        return result;
    }

    private static Account createAccountInner(ProtoCurrencyUnit currencyUnit) {
        Preconditions.checkArgument(currencyUnit != null, "Account currency can't be null");
        Account result = new Account();
        result.setCurrency(currencyUnit);
        return result;
    }

    public void addRecord(AccountRecord record) {
        if (this.accountRecords == null) {
            this.accountRecords = new ArrayList<>();
        }
        record.setAccount(this);
        this.accountRecords.add(record);
    }
}
