package ru.yandex.travel.orders.entities;

import java.math.BigDecimal;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import javax.persistence.CascadeType;
import javax.persistence.Entity;
import javax.persistence.FetchType;
import javax.persistence.Id;
import javax.persistence.OneToMany;
import javax.persistence.Table;

import com.google.common.base.Preconditions;
import lombok.Data;
import lombok.NoArgsConstructor;
import org.hibernate.annotations.CreationTimestamp;
import org.hibernate.annotations.Type;

@Entity
@Table(name = "account_transactions")
@Data
@NoArgsConstructor
public class AccountTransaction {

    @Id
    private UUID id;

    @CreationTimestamp
    private LocalDateTime createdAt;

    private String description;

    @OneToMany(cascade = CascadeType.ALL, fetch = FetchType.LAZY, mappedBy = "accountTransaction")
    private List<AccountRecord> accountRecords;

    @Type(type = "jsonb-object")
    private FxRate fxRate;

    private AccountTransaction(List<AccountRecord> recordList, String description, FxRate fxRate) {
        this.id = UUID.randomUUID();
        for (AccountRecord record : recordList) {
            record.setAccountTransaction(this);
        }
        this.accountRecords = recordList;
        this.description = description;
        this.fxRate = fxRate;
    }

    public static Builder builder(FxContext fxContext) {
        return new Builder(fxContext);
    }

    public static final class Builder {

        private final List<AccountRecord> records = new ArrayList<>();
        private final Map<Account, BigDecimal> fromAccounts = new HashMap<>();
        private final Map<Account, BigDecimal> toAccounts = new HashMap<>();
        private final FxContext fxContext;
        private String description;

        Builder(FxContext fxContext) {
            this.fxContext = fxContext;
        }

        private void addRecord(Account account, BigDecimal amount) {
            this.records.add(new AccountRecord(account, amount));
        }

        /**
         * Add transfer of money between accounts with respect to their currencies.
         *
         * @param fromAccount account to charge for money
         * @param toAccount   account to transfer money to
         * @param value       value of money in `fromAccount` currency
         * @return builder
         */
        public Builder addTransfer(Account fromAccount, Account toAccount, BigDecimal value) {
            BigDecimal exchangeRate = fxContext.getExchangeRateFor(fromAccount.getCurrency(), toAccount.getCurrency());
            BigDecimal exchangedValue = value.multiply(exchangeRate);

            this.fromAccounts.compute(fromAccount, (k, v) -> (v == null) ? value.negate() : v.subtract(value));
            this.toAccounts.compute(toAccount, (k, v) -> (v == null) ? exchangedValue : v.add(exchangedValue));
            return this;
        }

        /**
         * Set the description of transaction
         *
         * @param description
         * @return builder
         */
        public Builder setDescription(String description) {
            this.description = description;
            return this;
        }

        public AccountTransaction build() {
            this.fromAccounts.forEach(this::addRecord);
            this.toAccounts.forEach(this::addRecord);
            Preconditions.checkArgument(this.records.size() != 0, "There should be non zero amount of records");
            return new AccountTransaction(this.records, this.description, this.fxContext.getFxRate());
        }
    }

}
