package ru.yandex.travel.orders.entities;

import java.io.Serializable;
import java.util.UUID;

import javax.persistence.Embeddable;
import javax.persistence.EmbeddedId;
import javax.persistence.Entity;
import javax.persistence.EnumType;
import javax.persistence.Enumerated;
import javax.persistence.Table;
import javax.validation.constraints.NotNull;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.AccessLevel;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.Setter;
import org.jetbrains.annotations.TestOnly;

import ru.yandex.travel.credentials.UserCredentials;


@Entity
@Table(name = "authorized_users")
public class AuthorizedUser {
    @EmbeddedId
    private Key id;

    @Setter(value = AccessLevel.PRIVATE)
    @Getter
    private String login;

    @Setter(value = AccessLevel.PRIVATE)
    @Getter
    private String yandexUid;

    @Enumerated(EnumType.STRING)
    @Setter(value = AccessLevel.PRIVATE)
    @Getter
    private OrderUserRole role;

    @Setter
    private Boolean userIsStaff;

    @Setter
    private Boolean userIsPlus;

    private AuthorizedUser() {
        id = new Key();
    }

    @TestOnly
    @VisibleForTesting
    public static AuthorizedUser createGuest(UUID orderId, @NotNull String sessionKey, String yandexUid,
                                             OrderUserRole role) {
        Preconditions.checkNotNull(sessionKey);
        AuthorizedUser user = new AuthorizedUser();
        user.id.setOrderId(orderId);
        user.id.setSessionKey(sessionKey);
        user.id.setPassportId("");
        user.setYandexUid(yandexUid);
        user.setRole(role);
        return user;
    }

    @TestOnly
    @VisibleForTesting
    public static AuthorizedUser createLogged(UUID orderId, String yandexUid, @NotNull String passportId,
                                              @NotNull String login, OrderUserRole role) {
        Preconditions.checkNotNull(passportId);
        Preconditions.checkNotNull(login);
        AuthorizedUser user = new AuthorizedUser();
        user.id.setOrderId(orderId);
        user.id.setPassportId(passportId);
        user.id.setSessionKey("");
        user.setYandexUid(yandexUid);
        user.setRole(role);
        user.setLogin(login);
        return user;
    }

    public static AuthorizedUser create(UUID orderId, UserCredentials credentials, OrderUserRole role) {
        AuthorizedUser user = new AuthorizedUser();
        user.setYandexUid(credentials.getYandexUid());
        user.id.setOrderId(orderId);
        user.setRole(role);
        user.setUserIsStaff(credentials.isUserIsStaff());
        user.setUserIsPlus(credentials.isUserIsPlus());
        if (credentials.isLoggedIn()) {
            user.id.setPassportId(credentials.getPassportId());
            user.id.setSessionKey("");
            user.setLogin(credentials.getLogin());
        } else {
            user.id.setSessionKey(credentials.getSessionKey());
            user.id.setPassportId("");
        }
        return user;
    }

    public String getSessionKey() {
        return id.getSessionKey();
    }

    public String getPassportId() {
        return id.getPassportId();
    }

    public UUID getOrderId() {
        return id.getOrderId();
    }

    public boolean getUserIsStaff() {
        return userIsStaff != null && userIsStaff;
    }

    public boolean getUserIsPlus() {
        return userIsPlus != null && userIsPlus;
    }

    public enum OrderUserRole {
        OWNER,
        VIEWER
    }

    @Embeddable
    @Getter(value = AccessLevel.PRIVATE)
    @Setter(value = AccessLevel.PRIVATE)
    @EqualsAndHashCode
    public static class Key implements Serializable {

        private UUID orderId;
        private String sessionKey;
        private String passportId = "";

        public String getPassportId() {
            return Strings.emptyToNull(passportId);
        }

        public void setPassportId(String passportId) {
            this.passportId = Strings.nullToEmpty(passportId);
        }
    }
}


