package ru.yandex.travel.orders.entities;

import java.time.LocalDateTime;

import javax.persistence.Column;
import javax.persistence.DiscriminatorValue;
import javax.persistence.Entity;

import lombok.Getter;
import lombok.Setter;
import org.hibernate.annotations.Type;

import ru.yandex.travel.hotels.common.orders.BNovoHotelItinerary;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.hotels.proto.EPartnerId;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.partners.BillingPartnerAgreement;
import ru.yandex.travel.orders.entities.partners.DirectHotelBillingPartnerAgreement;
import ru.yandex.travel.orders.entities.partners.DirectHotelBillingPartnerAgreementProvider;
import ru.yandex.travel.orders.workflow.hotels.bnovo.proto.EBNovoItemState;
import ru.yandex.travel.workflow.entities.WorkflowEntity;

@Entity
@DiscriminatorValue(WellKnownOrderItemDiscriminator.ORDER_ITEM_BNOVO)
@Getter
@Setter
public class BNovoOrderItem extends HotelOrderItem implements WorkflowEntity<EBNovoItemState>,
        DirectHotelBillingPartnerAgreementProvider {
    @Type(type = "proto-enum")
    private EBNovoItemState state;

    @Type(type = "jsonb-object")
    @Column(name = "payload")
    private BNovoHotelItinerary itinerary;

    @Type(type = "jsonb-object")
    @Column(name = "agreement")
    private DirectHotelBillingPartnerAgreement agreement;


    @Override
    public HotelOrderItemState getPublicState() {
        //noinspection DuplicatedCode
        switch (getState()) {
            case IS_RESERVED:
                return HotelOrderItemState.RESERVED;
            case IS_NEW:
                return HotelOrderItemState.NEW;
            case IS_CONFIRMED:
                return HotelOrderItemState.CONFIRMED;
            case IS_CANCELLED:
                return HotelOrderItemState.CANCELLED;
            case IS_REFUNDED:
                return HotelOrderItemState.REFUNDED;
            default:
                return HotelOrderItemState.OTHER;
        }
    }

    @Override
    public HotelItinerary getHotelItinerary() {
        return itinerary;
    }

    @Override
    public EPartnerId getPartnerId() {
        return EPartnerId.PI_BNOVO;
    }

    @Override
    public Enum getItemState() {
        return state;
    }

    @Override
    public EServiceType getPublicType() {
        return EServiceType.PT_BNOVO_HOTEL;
    }

    @Override
    public LocalDateTime getServicedAt() {
        return itinerary.getOrderDetails().getCheckoutDate().atStartOfDay();
    }

    @Override
    public BillingPartnerAgreement getBillingPartnerAgreement() {
        return agreement;
    }

    @Override
    public String getEntityType() {
        return WellKnownWorkflowEntityType.BNOVO_ORDER_ITEM.getDiscriminatorValue();
    }

    @Override
    public EBNovoItemState getState() {
        return state;
    }

    @Override
    public void setState(EBNovoItemState state) {
        this.state = state;
    }

    @Override
    public Object getPayload() {
        return itinerary;
    }
}
