package ru.yandex.travel.orders.entities;

import java.time.LocalDateTime;
import java.util.List;

import javax.persistence.Column;
import javax.persistence.DiscriminatorValue;
import javax.persistence.Entity;
import javax.persistence.OneToMany;

import lombok.Getter;
import lombok.Setter;
import org.hibernate.annotations.Type;
import org.javamoney.moneta.Money;

import ru.yandex.travel.bus.model.BusReservation;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.partners.BusBillingPartnerAgreement;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.workflow.entities.WorkflowEntity;


@DiscriminatorValue(WellKnownOrderItemDiscriminator.ORDER_ITEM_BUS)
@Getter
@Setter
@Entity
public class BusOrderItem extends OrderItem implements WorkflowEntity<EOrderItemState> {
    @Type(type = "proto-enum")
    private EOrderItemState state;

    @Type(type = "jsonb-object")
    @Column(name = "payload")
    private BusReservation reservation;

    @Type(type = "jsonb-object")
    @Column(name = "agreement")
    private BusBillingPartnerAgreement billingPartnerAgreement;

    @OneToMany(mappedBy = "orderItem")
    private List<BusTicketRefund> busTicketRefunds;

    @Override
    public BusReservation getPayload() {
        return reservation;
    }

    @Override
    public String getEntityType() {
        return WellKnownWorkflowEntityType.BUS_ORDER_ITEM.getDiscriminatorValue();
    }

    @Override
    public EServiceType getPublicType() {
        return EServiceType.PT_BUS;
    }

    @Override
    public LocalDateTime getServicedAt() {
        return getPayload().getRide().getLocalDepartureTime();
    }

    @Override
    public EOrderItemState getItemState() {
        return state;
    }

    public void setReservation(BusReservation reserv) {
        this.reservation = reserv;
    }

    @Override
    public Money preliminaryTotalCost() {
        return Money.zero(ProtoCurrencyUnit.RUB);
    }

    @Override
    public Money totalCostAfterReservation() {
        Money result = Money.zero(ProtoCurrencyUnit.RUB);
        if (this.reservation.getOrder() != null) {
            for (var ticket : this.reservation.getOrder().getTickets()) {
                if (ticket != null && ticket.getPrice() != null) {
                    result = result.add(ticket.getTotalPrice());
                }
            }
        }
        return result;
    }
}
