package ru.yandex.travel.orders.entities;

import java.time.Instant;
import java.util.UUID;

import javax.persistence.Entity;
import javax.persistence.Id;
import javax.persistence.ManyToOne;
import javax.persistence.OneToOne;
import javax.persistence.Table;
import javax.persistence.Version;

import lombok.Data;
import org.hibernate.annotations.CreationTimestamp;
import org.hibernate.annotations.Type;
import org.hibernate.annotations.UpdateTimestamp;

import ru.yandex.travel.bus.model.BusesTicketRefund;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.workflow.orderitem.bus.proto.TBusRefundTicket;
import ru.yandex.travel.orders.workflow.orderitem.bus.ticketrefund.proto.EBusTicketRefundState;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.entities.WorkflowEntity;

@Entity
@Table(name = "bus_ticket_refunds")
@Data
public class BusTicketRefund implements WorkflowEntity<EBusTicketRefundState> {
    @Id
    private UUID id;

    @OneToOne
    private Workflow workflow;

    @Type(type = "proto-enum")
    private EBusTicketRefundState state;

    @CreationTimestamp
    private Instant createdAt;

    @UpdateTimestamp
    private Instant updatedAt;

    private Instant lastTransitionAt;

    private Instant nextCheckAt;

    private boolean backgroundJobActive;

    @ManyToOne
    private BusOrderItem orderItem;

    private UUID orderItemWorkflowId;

    @Type(type = "jsonb-object")
    private BusesTicketRefund payload;

    @OneToOne
    private OrderRefund orderRefund;

    @Version
    private Integer version;

    public static BusTicketRefund createRefund(BusOrderItem orderItem,
                                               TBusRefundTicket refundTicket,
                                               OrderRefund orderRefund) {
        var refund = new BusTicketRefund();
        refund.setId(UUID.randomUUID());
        refund.setState(EBusTicketRefundState.RS_NEW);
        refund.setOrderItem(orderItem);
        refund.setOrderItemWorkflowId(orderItem.getWorkflow().getId());
        refund.setOrderRefund(orderRefund);
        var payload = new BusesTicketRefund();
        refund.setPayload(payload);
        payload.setTicketId(refundTicket.getTicketId());
        payload.setCalculatedRefundAmount(ProtoUtils.fromTPrice(refundTicket.getRefundAmount()));
        payload.setSupplierId(orderItem.getPayload().getRide().getSupplierId());
        payload.setOrderId(orderItem.getPayload().getOrder().getId());
        return refund;
    }

    @Override
    public String getEntityType() {
        return WellKnownWorkflowEntityType.BUS_TICKET_REFUND.getDiscriminatorValue();
    }

    public Instant getLastTransitionAt() {
        if (lastTransitionAt == null) {
            return createdAt;
        }
        return lastTransitionAt;
    }
}
