package ru.yandex.travel.orders.entities;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.money.CurrencyUnit;
import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.ManyToOne;
import javax.persistence.OneToMany;
import javax.persistence.SequenceGenerator;
import javax.persistence.Table;

import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.Builder;
import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.Getter;
import lombok.NoArgsConstructor;
import lombok.ToString;
import org.hibernate.annotations.BatchSize;
import org.hibernate.annotations.Columns;
import org.hibernate.annotations.Type;
import org.javamoney.moneta.Money;

import ru.yandex.travel.orders.entities.promo.FiscalItemDiscount;

@Entity
@Table(name = "fiscal_items")
@Data
@NoArgsConstructor
@AllArgsConstructor
@Builder(toBuilder = true)
@EqualsAndHashCode(exclude = "orderItem")
@ToString(exclude = "orderItem")
@BatchSize(size = 100)
public class FiscalItem implements InvoiceItemProvider {
    @Id
    @GeneratedValue(
            strategy = GenerationType.SEQUENCE,
            generator = "fiscal_items_id_seq"
    )
    @SequenceGenerator(
            name = "fiscal_items_id_seq",
            sequenceName = "fiscal_items_id_seq",
            allocationSize = 1
    )
    private Long id;

    @ManyToOne
    private OrderItem orderItem;

    @Type(type = "custom-enum")
    private FiscalItemType type;

    private String title;

    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "money_amount"), @Column(name = "money_currency")
    })
    private Money moneyAmount;

    @Type(type = "money-proto-enum")
    @Columns(columns = {
            @Column(name = "yandex_plus_withdraw_amount"), @Column(name = "yandex_plus_withdraw_currency")
    })
    private Money yandexPlusToWithdraw;

    @Type(type = "custom-enum")
    private VatType vatType;

    @OneToMany(mappedBy = "fiscalItem", cascade = CascadeType.ALL)
    @BatchSize(size = 100)
    @Getter(AccessLevel.NONE)
    private List<FiscalItemDiscount> fiscalItemDiscounts;

    private Integer internalId; // id to map to data inside OrderItem
    private String inn;

    public List<FiscalItemDiscount> getFiscalItemDiscounts() {
        return fiscalItemDiscounts == null ? Collections.emptyList() :
                Collections.unmodifiableList(fiscalItemDiscounts);
    }

    public Money getDiscountAmount() {
        return getFiscalItemDiscounts().stream()
                .map(FiscalItemDiscount::getDiscount)
                .reduce(Money::add)
                .orElseGet(() -> Money.zero(getCurrency()));
    }

    public Money getDiscountAmountExceptPromo() {
        return getFiscalItemDiscounts().stream()
                .filter(fiscalItemDiscount -> !fiscalItemDiscount.isPromoCodeDiscount())
                .map(FiscalItemDiscount::getDiscount)
                .reduce(Money::add)
                .orElseGet(() -> Money.zero(getCurrency()));
    }

    public CurrencyUnit getCurrency() {
        return moneyAmount.getCurrency();
    }

    @Override
    public Long getFiscalItemId() {
        return id;
    }

    /**
     * The total cost of the item.
     * Consists of actual user money and used plus points (if any).
     * Doesn't contain discounted money.
     */
    @Override
    public Money getMoneyAmount() {
        return moneyAmount.subtract(getDiscountAmount());
    }

    /**
     * Returns the fiscal price without discount by special promo campaigns
     */
    public Money getMoneyAmountExceptPromo() {
        return moneyAmount.subtract(getDiscountAmountExceptPromo());
    }

    /**
     * The real money user has to pay.
     * Doesn't include plus points he spends (if any).
     */
    public Money getMoneyAmountWithoutPlus() {
        return getMoneyAmount().subtract(getYandexPlusToWithdraw());
    }

    @Override
    public Money getYandexPlusToWithdraw() {
        return yandexPlusToWithdraw != null ? yandexPlusToWithdraw : Money.zero(getCurrency());
    }

    public MoneyMarkup getMoneyAmountMarkup() {
        return MoneyMarkup.builder()
                .card(getMoneyAmountWithoutPlus())
                .yandexAccount(getYandexPlusToWithdraw())
                .build();
    }

    /**
     * The source total item cost with discounted money included.
     */
    public Money getOriginalCost() {
        return moneyAmount;
    }

    public void applyDiscount(Money amount) {
        if (fiscalItemDiscounts == null) {
            fiscalItemDiscounts = new ArrayList<>(1);
        }
        final var fid = new FiscalItemDiscount();
        fid.setFiscalItem(this);
        fid.setDiscount(amount);
        fiscalItemDiscounts.add(fid);
    }
}
