package ru.yandex.travel.orders.entities;

import java.time.Duration;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.UUID;

import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.ManyToOne;
import javax.persistence.SequenceGenerator;
import javax.persistence.Table;

import lombok.Data;
import lombok.EqualsAndHashCode;
import lombok.NoArgsConstructor;
import lombok.ToString;
import org.hibernate.annotations.BatchSize;
import org.hibernate.annotations.Type;

import ru.yandex.travel.orders.services.payments.PaymentProfile;

@Entity
@Table(name = "fiscal_receipts")
@Data
@NoArgsConstructor
@EqualsAndHashCode(exclude = {"invoice"})
@ToString(exclude = {"invoice"})
@BatchSize(size = 100)
public class FiscalReceipt {
    @Id
    @GeneratedValue(
            strategy = GenerationType.SEQUENCE,
            generator = "fiscal_receipts_id_seq"
    )
    @SequenceGenerator(
            name = "fiscal_receipts_id_seq",
            sequenceName = "fiscal_receipts_id_seq",
            allocationSize = 1
    )
    private Long id;

    @ManyToOne
    private Invoice invoice;

    @Type(type = "custom-enum")
    private FiscalReceiptType receiptType;

    private String receiptUrl;

    @Type(type = "custom-enum")
    private FiscalReceiptState state;

    private String parentId;

    private Instant nextRefreshAt;

    private Integer nextRefreshAttempt;

    @Type(type = "custom-enum")
    private PaymentProfile paymentProfile;

    private UUID orderRefundId;

    public static FiscalReceipt createReceipt(FiscalReceiptType type, String parentId, PaymentProfile paymentProfile,
                                              UUID orderRefundId) {
        FiscalReceipt receipt = new FiscalReceipt();
        receipt.setState(FiscalReceiptState.CREATED);
        receipt.setNextRefreshAt(Instant.now()); //TODO (mbobrov): delay initial refresh?
        receipt.setNextRefreshAttempt(1);
        receipt.setReceiptType(type);
        receipt.setParentId(parentId);
        receipt.setPaymentProfile(paymentProfile);
        receipt.setOrderRefundId(orderRefundId);
        return receipt;
    }

    public void rescheduleNextRefresh(Duration duration) {
        nextRefreshAt = Instant.now().plus(duration.toMillis(), ChronoUnit.MILLIS);
        nextRefreshAttempt = getNextRefreshAttempt() + 1; // to fallback for null
    }

    public void resetNextRefresh() {
        nextRefreshAt = null;
        nextRefreshAttempt = 0;
    }

    public int getNextRefreshAttempt() {
        if (nextRefreshAttempt == null) {
            nextRefreshAttempt = 0;
        }
        return nextRefreshAttempt;
    }
}
