package ru.yandex.travel.orders.entities;

import java.util.ArrayList;
import java.util.List;
import java.util.UUID;

import javax.persistence.DiscriminatorValue;
import javax.persistence.Entity;
import javax.persistence.OneToMany;

import lombok.Getter;
import lombok.Setter;
import org.hibernate.annotations.Type;
import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.orders.proto.EOrderRefundState;
import ru.yandex.travel.orders.proto.EOrderRefundType;
import ru.yandex.travel.orders.workflow.orderitem.bus.ticketrefund.proto.EBusTicketRefundState;
import ru.yandex.travel.orders.workflow.orderitem.train.ticketrefund.proto.ETrainTicketRefundState;

@DiscriminatorValue("generic_order_user_refund")
@Getter
@Setter
@Entity
public class GenericOrderUserRefund extends OrderRefund implements TrainOrderRefund {
    @OneToMany(mappedBy = "orderRefund")
    private List<TrainTicketRefund> trainTicketRefunds;

    @OneToMany(mappedBy = "orderRefund")
    private List<BusTicketRefund> busTicketRefunds;

    public static GenericOrderUserRefund createForOrder(Order order) {
        var refund = new GenericOrderUserRefund();
        refund.setId(UUID.randomUUID());
        refund.setBusTicketRefunds(new ArrayList<>());
        refund.setTrainTicketRefunds(new ArrayList<>());
        refund.setState(EOrderRefundState.RS_NEW);
        refund.setPayload(new OrderRefundPayload());
        order.addOrderRefund(refund);
        return refund;
    }

    @Type(type = "jsonb-object")
    private OrderRefundPayload payload;

    @Override
    public EOrderRefundType getRefundType() {
        return EOrderRefundType.RT_GENERIC_USER_REFUND;
    }

    @Override
    public Money calculateRefundedAmount() {
        ProtoCurrencyUnit currency = getOrder().getCurrency();
        Money refundedSum = getTrainTicketRefunds().stream()
                .filter(tr -> tr.getState() == ETrainTicketRefundState.RS_REFUNDED)
                .map(tr -> tr.getPayload().calculateActualRefundSum())
                .reduce(Money::add).orElse(Money.of(0, currency));
        refundedSum = refundedSum.add(getBusTicketRefunds().stream()
                .filter(br -> br.getState() == EBusTicketRefundState.RS_REFUNDED)
                .map(br -> br.getPayload().getRefundAmount())
                .reduce(Money::add).orElse(Money.of(0, currency)));
        return refundedSum;
    }
}
